/*--- formatted by Jindent 2.1, (www.c-lab.de/~jindent) ---*/

/*
 * ImageJ filter to export images as DICOM secondary-captured files.
 *
 * Copyright (C) 2000 Thomas Hacklaender, e-mail: reply@thomas-hacklaender.de
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License 2
 * as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * http://www.gnu.org/copyleft/copyleft.html
 *
 * The ImageJ project:
 * http://rsb.info.nih.gov/ij/default.html
 * Author: Wayne Rasband, wayne@codon.nih.gov
 * Research Services Branch, National Institute of Mental Health, Bethesda, Maryland, USA.
 * Download: ftp://codon.nih.gov/pub/image-j/
 */

import java.io.*;
import java.util.*;
import java.awt.*;
import java.awt.event.*;
import javax.swing.*;

import ij.*;
import ij.plugin.*;
import ij.plugin.filter.*;
import ij.process.*;

import rad.dicom.dcm.*;
import rad.dicom.ima.*;
import rad.ijplugin.dcm.*;
import rad.ijplugin.util.*;


/**
 * Diese PlugIn exportier ImageJ Bilder als DICOM 3
 * Secondary Capture files.<br>
 * Beim Aufruf des Plugins kann optional ein Argument (als String)
 * uebergeben werden. Diese kann aus mehreren Einzelelementen
 * zusammengesetzt sein, die durch ein Tabulatorzeichen '\t'
 * voneinander getrennt sind.<br>
 * Beginnt ein solches Unterargument mit "-p", wird damit ein
 * Property File bezeichnet. Alle Zeichen bis zum naechsten Tabulator
 * werden als Filename des Property Files gewertet. Beginnt dieser
 * Filename mit '.' handelt es sich um einen relativen Pfad zum
 * user.dir. Beginnt er mit einem anderen Zeichen handelt es sich
 * um einen vollstaendigen Filenamen.<br>
 * Beginnt das Unterargument nicht mit '-' handelt es sich um den Namen
 * unter dem das Bild gespeichert werden soll. Beginnt dieser
 * Filename mit '.' handelt es sich um einen relativen Pfad zum
 * output.dir des Property Files bzw. der Voreinstellung in ExportData.
 * Beginnt er mit einem anderen Zeichen handelt es sich um einen
 * vollstaendigen Filenamen.<br>
 * </pre>
 * <DL><DT><B>Modifications: </B><DD>
 * Thomas Hacklaender 2000.08.22:
 * Funktionalitaet von PanelDialog durch JOptionPane.showConfirmDialog
 * ersetzt.<br>
 * Thomas Hacklaender 2000.08.22:
 * Klassen PDPanel und PanelDialog aus dem Projekt entfernt.<br>
 * </DD></DL>
 * @author   Thomas Hacklaender
 * @version  2002.5.12
 */
public class DICOM_export implements PlugInFilter {

	/**
	 * Die Versionsnummer dieses Panels
	 */
	private final String		EXPORT_VERSION = "5.0";


	/*
	 * Mit dieser Klasse werden Daten zwischen dem Plugin und der Dialgobox
	 * sowie deren Panels ausgetauscht.
	 */
	private ExportData			exda;

	/* Das ImagePlus, das beim setup Aufruf uebergeben wurde */
	private ImagePlus				imaPlus;

	/* Die Nummer des aktuelle Bildes */
	private int							imageNumber;

	/* Der ImageProcessor des aktuellen Bildes */
	private ImageProcessor	imaProc;


	/**
	 * Konstruktor. Nur zum Debuggen.
	 */
	public DICOM_export() {}


	/**
	 * Diese Methode wird einmal aufgerufen, wenn der Filter geladen wird.
	 * @param arg  Der path des DICOM files, in den geschrieben werden soll.
	 * Falls in "ij.properties" ein Argument fuer dieses
	 * Filter definiert ist, hat arg dessen Wert.
	 * Ist arg = "", wird eine save-Dialogbox angezeigt.
	 * @param imaPlus  Das aktuell aktive Bild oder Stack (ImagePlus Objekt)
	 * @return     Flags, die die Moeglichkeiten des Filters beschreiben.
	 */
	public int setup(String arg, ImagePlus imaProc) {
		String			fnp;
		ExportPanel exPan;
		InputStream pfs;
		Properties	prop = null;
		String[]		sl = null;


		// Das Look and Feel der zugrund liegenden Maschine waehlen
		try {
			UIManager.setLookAndFeel(UIManager.getSystemLookAndFeelClassName());
		} catch (Exception e) {}

		// beenden, wenn kein Bild vorhanden ist.
		if (imaProc == null) {
			return DONE;

			// ImagePlus fuer run Methode speichern
		}
		imaPlus = imaProc;

		// Wir verarbeiten noch kein Bild
		imageNumber = 0;

		// Datenblockzur Kommunikation zwischen Plugin und Dialogbox generieren
		exda = new ExportData();

		// Neue SecondaryCaptureIOD
		exda.theSC = new SecondaryCaptureIOD();

		// Patientenname ist gleich dem Titel des Bildes
		exda.theSC.patientName = imaPlus.getTitle();

		// Versucht einen Property File zu finden
		pfs = Util.getPropertyFileStream(arg);
		if (pfs != null) {
			exda.getProperties(pfs);

			// Versucht einen Filenamen fuer die Verarbeitung zu finden
		}
		fnp = Util.getFileNameToProcess(arg, exda.inputDir.getPath());

		/*
		 * prop = imaPlus.getProperties();
		 * if (prop != null) sl = (String[]) prop.get(DcmUID.DCM_STRING_HEADER_PROPERTY);
		 * if (sl != null) sci.setPatientModule(sl[0]);
		 */

		if (fnp == null) {
			exPan = new ExportPanel();
			exPan.initWithDataBlock(exda);

      // Geaendert: tha 2000.8.22
      exPan.copyrightL.setText("(C) 2000 T. Hacklaender under the terms of the GPL license. Dialog Ver.: " + EXPORT_VERSION + ", Dcm Ver.: " + DcmUID.DCM_VERSION);
      int diaRes = JOptionPane.showConfirmDialog(null, exPan, "Save as DICOM Secondary Capture", JOptionPane.OK_CANCEL_OPTION, JOptionPane.PLAIN_MESSAGE, null);
      if (diaRes == JOptionPane.OK_OPTION) {
				// "OK"
				exPan.updateDataBlock();
			} else {
				// "Cancel"
				return DONE;
			}
      /*
			panDia = new PanelDialog(exPan, "Save as DICOM Secondary Capture", "(C) 2000 T. Hacklaender under the terms of the GPL license. Dialog " + EXPORT_VERSION + ", Dcm " + DcmUID.DCM_VERSION + "    ", PanelDialog.OK_CANCEL_BUTTON);
			if (panDia.isOkPressed()) {
				// "OK"
				exPan.updateDataBlock();
			} else {
				// "Cancel"
				return DONE;
			}
      */
		} else {
			File	f = new File(fnp);

			exda.fileNameToSave = f.getName();
			exda.inputDir = new File(f.getParent());
		}

		// Diese Bildtypen werden unterstuetzt
		return DOES_8G + DOES_8C + DOES_16 + DOES_STACKS + NO_CHANGES;

	}


	/**
	 * Die Filterroutine. Falls der Filter die Verarbeitung von Stacks
	 * unterstuetzt, wird diese Methode fuer jedes Bild des Stacks
	 * aufgerufen. Image/J locked das Bild  bevor der Filter aufgerufen
	 * wird und unlocked es danach wieder.
	 * @param imaProc  Das ImageProcessor Object des aktuell aktiven Bildes
	 */
	public void run(ImageProcessor ip) {
		String					fn;
		String					pn;
		String					is;
		int							dImageNumber;
		DcmDataObject		ddo;
		File						f;

		imaProc = ip;

		// Die Nummer des aktuellen Bildes. Erstes Bild = 1.
		imageNumber++;

		// Der endgueltige Filename fuer das Bild
		fn = exda.fileNameToSave;
		if (imaPlus.getStackSize() > 1) {

			// Bei einem Stack wird an den Filename die 3stellige Bildnummer
			// mit einem '_' angehaengt. Die Nummer des ersten Bildes ist
			// dabei die in der Dialogbox vorgegebene Imagenummer.
			try {
				dImageNumber = Integer.parseInt(exda.theSC.imageNumber);
			} catch (Exception ex) {
				dImageNumber = 1;
			}
			is = "000" + Integer.toString(dImageNumber + imageNumber - 1);
			fn = fn + "_" + is.substring(is.length() - 3, is.length());
		}
		fn += exda.fileNameSuffix;

		// Die Bildabhaengigen Parameter werden gesetzt
		switch (imaPlus.getType()) {
		case ImagePlus.GRAY8:
			doGRAY8();
			break;
		case ImagePlus.COLOR_256:
			doCOLOR8();
			break;
		case ImagePlus.GRAY16:
			doGRAY16();
			break;
		}

		// erzeugt ein neues DcmDataObject mit der gewuenschten Transfersyntax
		ddo = exda.theSC.getDDO(exda.encoding, exda.structure, exda.storage);

		// Speichert die Datei
		try {

			// Generiert den vollstaendigen Pfad
			pn = exda.outputDir.getPath() + File.separator + fn;
			f = new File (pn);
			DcmOutputStream.saveDDO(ddo, f);
		} catch (Exception ex) {
			ex.printStackTrace();
		}

	}


	/**
	 * Extrahiert die Pixelinformation eines 8 Bit ImageJ Graustufenbildes,
	 * sowie dessen Center/Window Werte und speichert sie in der aktuellen
	 * SecondaryCaptureIOD ab
	 */
	private void doGRAY8() {
		byte[]	pixel8;

		pixel8 = new byte[imaProc.getHeight() * imaProc.getWidth()];
		for (int i = 0; i < pixel8.length; i++) {
			pixel8[i] = ((byte[]) imaProc.getPixels())[i];
		}
		exda.theSC.set8BitGrayImage(pixel8, imaProc.getHeight(), imaProc.getWidth());
		int c = (int) ((imaProc.getMax() + imaProc.getMin()) / 2);
		int w = (int) (imaProc.getMax() - imaProc.getMin());

		exda.theSC.setVOILUTModule(c, w);
	}


	/**
	 * Extrahiert die Pixelinformation eines 8 Bit ImageJ Farb-Palettenbildes,
	 * sowie dessen Center/Window Werte und speichert sie in der aktuellen
	 * SecondaryCaptureIOD ab
	 */
	private void doCOLOR8() {
		byte[]	pixel8;

		pixel8 = new byte[imaProc.getHeight() * imaProc.getWidth()];
		for (int i = 0; i < pixel8.length; i++) {
			pixel8[i] = ((byte[]) imaProc.getPixels())[i];
		}
		exda.theSC.set8BitColorImage(pixel8, imaProc.getHeight(), imaProc.getWidth(), imaProc.getColorModel());
		imaProc.setColorModel(imaProc.getColorModel());
		int c = (int) ((imaProc.getMax() + imaProc.getMin()) / 2);
		int w = (int) (imaProc.getMax() - imaProc.getMin());

		exda.theSC.setVOILUTModule(c, w);
	}


	/**
	 * Extrahiert die Pixelinformation eines 16 Bit ImageJ Graustufenbildes,
	 * sowie dessen Center/Window Werte und speichert sie in der aktuellen
	 * SecondaryCaptureIOD ab
	 */
	private void doGRAY16() {
		short[] pixel;

		pixel = (short[]) imaProc.getPixels();
		exda.theSC.set16UBitGrayImage(pixel, imaProc.getHeight(), imaProc.getWidth(), 15);
		int c = (int) ((imaProc.getMax() + imaProc.getMin()) / 2);
		int w = (int) (imaProc.getMax() - imaProc.getMin());

		exda.theSC.setVOILUTModule(c, w);
	}

}


/*--- formatting done in "My Own Convention" style on 04-30-2000 ---*/

