/*--- formatted by Jindent 2.1, (www.c-lab.de/~jindent) ---*/

package mrcp.ddd;
import javax.swing.JTree;
import javax.swing.tree.*;
import javax.swing.JProgressBar;
import javax.swing.ProgressMonitor;
import java.util.Enumeration;
import java.util.Vector;
import ij.*;
import java.awt.event.*;
import ij.gui.ProgressBar;
import mrcp.gui.*;
import mrcp.graphics.*;
import mrcp.tools.*;
import mrcp.dd.*;
import mrcp.ddd.*;


/**
 * Die Klasse kapselt mittels einer hierrachischen Baumstruktur die gesamte
 * dreidimensionale Funktionalitt. Durch Markierung der Baumknoten erfolgt
 * die Selektion der Serien, die fr die Berechnungen als Quellen in Frage
 * kommen. Als zentrale Klasse wird hier die Funktionsauswertung, Integration und Interpolation
 * gesteuert.
 * 
 * 
 * @author Thomas Demuth
 * @version 2000.08.12
 */
public class Slice_Tree extends JTree {


	/**
	 * Wurzelelement des Schichtbaumes
	 */
	public DefaultMutableTreeNode root;


	/**
	 * Das Modell des Baumes
	 */
	public DefaultTreeModel				model;


	/**
	 * Anzahl der Serien im Baum
	 */
	public int										NumberOfSeries;		// Anzahl der Serien


	/**
	 * Array der aktuell markierten Serienindizes
	 */
	public int[]									SelectedSlices = new int[Global_Options.MaxNumberOfSeries];		// Die markierten Serien


	/**
	 * Anzahl der markierten Serien
	 */
	public int										NumberOfSelectesSeries;


	/**
	 * Array ber die Anzahl der Schichten pro Serie
	 */
	public int[]									NumberOfSlices;		// Anzahl der Schichten pro Serie


	/**
	 * Array der Schichtnummern des ersten Bildes pro Serie
	 */
	public int[]									SeriesOffsets;		// Offset der Schichtnummern


	/**
	 * Hilfsarray, um den Zugriff auf die DICOM-Attribute zu beschleunigen
	 */
	private Dicom_Slice[]					Slices;						// Slice Array


	/**
	 * Hilfsarray, um den Zugriff auf die Serien-Attribute zu beschleunigen
	 */
	private Series_Node[]					Series;						// Serien Teilbume


	/**
	 * x-Komponente des aktuell betrachteten Raumpunktes
	 */

	public float									newx = -1.0f;			// Matrix Koordinaten


	/**
	 * y-Komponente des aktuell betrachteten Raumpunktes
	 */
	public float									newy = -1.0f;			// Matrix Koordinaten


	/**
	 * z-Komponente des aktuell betrachteten Raumpunktes
	 */
	public float									newz = -1.0f;			// Matrix Koordinaten


	/**
	 * Interpolationsfaktor zwischen den Schichten
	 */
	public float									factor = 0.0f;	// Interpolationsfaktor zwischen den Schichtne


	/**
	 * Index der aktuell betrachteten Schicht
	 */
	public int										akt_index = 0;		// akuelle betrachte Schicht


	/**
	 * Position des aktuellen Raumpunktes relativ zu der zugehrigen Schicht
	 */
	private float									Slice_Position = 0.0f;	// Schicht Position des Funktionswertes


	/**
	 * Index einer Interpolationshilfsschicht
	 */
	private int										index2 = 0;				// Interpolationshilfsschicht !


	/**
	 * Index einer Interpolationshilfsschicht
	 */
	private int										index3 = 0;				// Interpolationshilfsschicht !


	/**
	 * Index einer Interpolationshilfsschicht
	 */
	public int										index4 = 0;				// Interpolationshilfsschicht !


	/**
	 * Schleifenvariable zur Positionssteuerung in Richtung des Reihenvektors
	 */
	public Point3D								RowCount = new Point3D(0, 0, 0);


	/**
	 * Schleifenvariable zur Positionssteuerung in Richtung des Spaltenvektors
	 */
	public Point3D								ColCount = new Point3D(0, 0, 0);


	/**
	 * Minimale z-Koordinate des 3D-Datensatzes
	 */
	public float									MinZ = Float.POSITIVE_INFINITY;


	/**
	 * Minimale x-Koordinate des 3D-Datensatzes
	 */
	public float									MinX = Float.POSITIVE_INFINITY;


	/**
	 * Minimale y-Koordinate des 3D-Datensatzes
	 */
	public float									MinY = Float.POSITIVE_INFINITY;


	/**
	 * Maximale x-Koordinate des 3D-Datensatzes
	 */
	public float									MaxX = Float.NEGATIVE_INFINITY;


	/**
	 * Maximale y-Koordinate des 3D-Datensatzes
	 */
	public float									MaxY = Float.NEGATIVE_INFINITY;


	/**
	 * Maximale z-Koordinate des 3D-Datensatzes
	 */
	public float									MaxZ = Float.NEGATIVE_INFINITY;

	// Intersection Box


	/**
	 * Minimale z-Koordinate des Schnittquaders
	 */
	public float									MinZ2 = Float.NEGATIVE_INFINITY;


	/**
	 * Minimale x-Koordinate des Schnittquaders
	 */
	public float									MinX2 = Float.NEGATIVE_INFINITY;


	/**
	 * Minimale y-Koordinate des Schnittquaders
	 */
	public float									MinY2 = Float.NEGATIVE_INFINITY;


	/**
	 * Maximale x-Koordinate des Schnittquaders
	 */
	public float									MaxX2 = Float.POSITIVE_INFINITY;


	/**
	 * Maximale y-Koordinate des Schnittquaders
	 */
	public float									MaxY2 = Float.POSITIVE_INFINITY;


	/**
	 * Maximale z-Koordinate des Schnittquaders
	 */
	public float									MaxZ2 = Float.POSITIVE_INFINITY;


	/**
	 * Fehleranzeige
	 */
	final private int							Error = -1;


	/**
	 * Hilfsarray zur Speicherung von Funktionswerten fr die Methoden
	 * "Gewichtete Summen " und "Lineare Gleichungen"
	 */
	private float[]								Functionvalues = new float[Global_Options.MaxNumberOfSeries];


	/**
	 * Hilfsarray zur Speicherung von Schichtindizes
	 */
	private int[]									SeriesPosition = new int[Global_Options.MaxNumberOfSeries];


	/**
	 * Hilfsarray zur Speicherung der Position des aktuellen Raumpunktes relativen zu der zugehrigen Schicht
	 */
	private float[]								Slice_Positions = new float[Global_Options.MaxNumberOfSeries];


	/**
	 * Array zur Speicherung der markierten Serien Indizes
	 */
	public int[]									Series_sel = new int[Global_Options.MaxNumberOfSeries];


	/**
	 * Erzeugt einen leeren Schichtbaum mit dem angegebenden Baummodell.
	 * 
	 * 
	 * @param model Baummodell
	 * 
	 */
	public Slice_Tree(DefaultTreeModel model) {	
            super (model) ;
                this.model = model ;
      	}





	/**
	 * Die Methode initialisiert den Baum zur Darstellung in JPanel_Source
	 * Dazu werden Modell und Wurzel festgelegt.
	 * 
	 * @param top Wurzelelement
	 * @param model Baummodell
	 * 
	 */
	public void initTree(DefaultMutableTreeNode top, DefaultTreeModel model) {

		// Tree
		this.model = model;
		this.root = top;
		this.setEditable(true);
		this.getSelectionModel().setSelectionMode(TreeSelectionModel.DISCONTIGUOUS_TREE_SELECTION);
		this.setShowsRootHandles(true);
		this.setCellRenderer(new MyRenderer());
		this.putClientProperty("JTree.lineStyle", "Angled");
	} 


	/**
	 * Die Methode wird aufgerufen nachdem sich der Inhalt des Baumes gendert hat.
	 * Dazu werden alle Knoteninformationen neu berechnet (Schichtknoten, Serienknoten, Wurzelknoten)
	 * Insbesondere mu hier darauf geachtet werden, da die zugefhrten Quellen
	 * vollstndig und fehlerfrei sind. Die Ausdehnungen der Serien und die des 3D-Datensatzes
	 * sind zu aktualisieren. Um den Zugriff auf einzelne Schichten und Serie zu beschleunigen,
	 * werden Hilfsarrays verwaltet.
	 */
	public void update() {
		NumberOfSeries = root.getChildCount();
		NumberOfSlices = new int[NumberOfSeries];
		SeriesOffsets = new int[NumberOfSeries];
		Series = new Series_Node[NumberOfSeries];

		int SliceCount = 0;
		int SliceOffset = 0;

		for (int i = 0; i < NumberOfSeries; i++) {
			DefaultMutableTreeNode	Serie = (DefaultMutableTreeNode) root.getChildAt(i);
			DefaultMutableTreeNode	FirstChild = (DefaultMutableTreeNode) Serie.getFirstChild();
			Slice_Leaf							leaf = (Slice_Leaf) FirstChild.getUserObject();
			Dicom_Slice							FirstSlice = leaf.cube.ds;
			Series_Node							sn = (Series_Node) Serie.getUserObject();

			Series[i] = sn;
			NumberOfSlices[i] = Serie.getChildCount();
			SeriesOffsets[i] = SliceOffset;



			// Vector Berechnung
			if (FirstSlice.isTransversal()) {
				sn.FirstImagePosZY = new Point2D(FirstSlice.FirstPixelCorner.getZ(), FirstSlice.FirstPixelCorner.getY());
				sn.FirstImagePosZX = new Point2D(FirstSlice.FirstPixelCorner.getZ(), FirstSlice.FirstPixelCorner.getX());
				sn.SliceVectorZY = new Point2D(FirstSlice.SliceVector.getZ(), FirstSlice.SliceVector.getY());
				sn.SliceVectorZX = new Point2D(FirstSlice.SliceVector.getZ(), FirstSlice.SliceVector.getX());
			} 

			if (FirstSlice.isCoronar()) {
				sn.FirstImagePosYX = new Point2D(FirstSlice.FirstPixelCorner.getY(), FirstSlice.FirstPixelCorner.getX());
				sn.FirstImagePosYZ = new Point2D(FirstSlice.FirstPixelCorner.getY(), FirstSlice.FirstPixelCorner.getZ());
				sn.SliceVectorYX = new Point2D(FirstSlice.SliceVector.getY(), FirstSlice.SliceVector.getX());
				sn.SliceVectorYZ = new Point2D(FirstSlice.SliceVector.getY(), FirstSlice.SliceVector.getZ());
			} 

			if (FirstSlice.isSagittal()) {
				sn.FirstImagePosXY = new Point2D(FirstSlice.FirstPixelCorner.getX(), FirstSlice.FirstPixelCorner.getY());
				sn.FirstImagePosXZ = new Point2D(FirstSlice.FirstPixelCorner.getX(), FirstSlice.FirstPixelCorner.getZ());
				sn.SliceVectorXY = new Point2D(FirstSlice.SliceVector.getX(), FirstSlice.SliceVector.getY());
				sn.SliceVectorXZ = new Point2D(FirstSlice.SliceVector.getX(), FirstSlice.SliceVector.getZ());
			} 

			sn.MinZ = Float.POSITIVE_INFINITY;
			sn.MinX = Float.POSITIVE_INFINITY;
			sn.MinY = Float.POSITIVE_INFINITY;
			sn.MaxX = Float.NEGATIVE_INFINITY;
			sn.MaxY = Float.NEGATIVE_INFINITY;
			sn.MaxZ = Float.NEGATIVE_INFINITY;

			// Berechnung der Min Max Werte
			Enumeration leafs = Serie.children();

			while (leafs.hasMoreElements()) {
				DefaultMutableTreeNode	child = (DefaultMutableTreeNode) leafs.nextElement();
				Slice_Leaf							sl = (Slice_Leaf) child.getUserObject();
				float										xmin = sl.cube.ds.getMinX();
				float										ymin = sl.cube.ds.getMinY();
				float										zmin = sl.cube.ds.getMinZ();
				float										xmax = sl.cube.ds.getMaxX();
				float										ymax = sl.cube.ds.getMaxY();
				float										zmax = sl.cube.ds.getMaxZ();

				if (xmin < sn.MinX) {
					sn.MinX = xmin;
				} 
				if (xmax > sn.MaxX) {
					sn.MaxX = xmax;
				} 
				if (ymin < sn.MinY) {
					sn.MinY = ymin;
				} 
				if (ymax > sn.MaxY) {
					sn.MaxY = ymax;
				} 
				if (zmin < sn.MinZ) {
					sn.MinZ = zmin;
				} 
				if (zmax > sn.MaxZ) {
					sn.MaxZ = zmax;
				} 
			}		// Enumeration
 
			SliceOffset += Serie.getChildCount();
			SliceCount += Serie.getChildCount();

			// Min Max Werte fr Bounding Box berechnen !
			if (sn.MinX < MinX) {
				MinX = sn.MinX;
			} 
			if (sn.MinY < MinY) {
				MinY = sn.MinY;
			} 
			if (sn.MinZ < MinZ) {
				MinZ = sn.MinZ;
			} 
			if (sn.MaxX > MaxX) {
				MaxX = sn.MaxX;
			} 
			if (sn.MaxY > MaxY) {
				MaxY = sn.MaxY;
			} 
			if (sn.MaxZ > MaxZ) {
				MaxZ = sn.MaxZ;

				// Min Max Werte fr Schnitt Box berechnen !
			} 
			if (sn.MinX > MinX2) {
				MinX2 = sn.MinX;
			} 
			if (sn.MinY > MinY2) {
				MinY2 = sn.MinY;
			} 
			if (sn.MinZ > MinZ2) {
				MinZ2 = sn.MinZ;
			} 
			if (sn.MaxX < MaxX2) {
				MaxX2 = sn.MaxX;
			} 
			if (sn.MaxY < MaxY2) {
				MaxY2 = sn.MaxY;
			} 
			if (sn.MaxZ < MaxZ2) {
				MaxZ2 = sn.MaxZ;
			} 
		} 

		// BB berechen
		Point3D p1 = new Point3D(MinX, MinY, MinZ);
		Point3D p2 = new Point3D(MaxX, MinY, MinZ);
		Point3D p3 = new Point3D(MinX, MaxY, MinZ);
		Point3D p4 = new Point3D(MaxX, MaxY, MinZ);
		Point3D p5 = new Point3D(MinX, MinY, MaxZ);
		Point3D p6 = new Point3D(MaxX, MinY, MaxZ);
		Point3D p7 = new Point3D(MinX, MaxY, MaxZ);
		Point3D p8 = new Point3D(MaxX, MaxY, MaxZ);

		Update_control.BB = new Bounding_Cube(p1, p2, p3, p4, p5, p6, p7, p8, Global_Options.BP);

		p1 = new Point3D(MinX2, MinY2, MinZ2);
		p2 = new Point3D(MaxX2, MinY2, MinZ2);
		p3 = new Point3D(MinX2, MaxY2, MinZ2);
		p4 = new Point3D(MaxX2, MaxY2, MinZ2);
		p5 = new Point3D(MinX2, MinY2, MaxZ2);
		p6 = new Point3D(MaxX2, MinY2, MaxZ2);
		p7 = new Point3D(MinX2, MaxY2, MaxZ2);
		p8 = new Point3D(MaxX2, MaxY2, MaxZ2);
		Update_control.IB = new Bounding_Cube(p1, p2, p3, p4, p5, p6, p7, p8, Global_Options.BP);

		// fill Slices Array ;
		Slices = new Dicom_Slice[SliceCount];
		int counter = 0;

		for (int i = 0; i < NumberOfSeries; i++) {
			for (int x = 0; x < NumberOfSlices[i]; x++) {
				DefaultMutableTreeNode	Serie = (DefaultMutableTreeNode) root.getChildAt(i);
				Slice_Leaf							sl = (Slice_Leaf) ((DefaultMutableTreeNode) Serie.getChildAt(x)).getUserObject();

				Slices[counter] = sl.cube.ds;
				counter++;
			} 
		} 

		Update_control.Trans_Z = (MaxZ + MinZ) / 2;
		Update_control.Sag_X = (MaxX + MinX) / 2;
		Update_control.Cor_Y = (MaxY + MinY) / 2;


		// Kontrolle der Vollstndigkeit und Fehlerfreiheit !

		for (int i = 0; i < NumberOfSeries; i++) {
			DefaultMutableTreeNode	Serie = (DefaultMutableTreeNode) root.getChildAt(i);
			Series_Node							sn = ((Series_Node) Serie.getUserObject());
			float										sum = sn.Sum;
			Slice_Leaf							fl = (Slice_Leaf) ((DefaultMutableTreeNode) Serie.getFirstChild()).getUserObject();
			Dicom_Slice							fds = fl.cube.ds;
			Point3D									SLVector = fds.SliceVector;
			Point3D									Start = fds.ImagePosition;
			float										count = 0;

			for (int s = 0; s < NumberOfSlices[i]; s++) {

				Slice_Leaf	sl = (Slice_Leaf) ((DefaultMutableTreeNode) Serie.getChildAt(s)).getUserObject();
				Dicom_Slice ds = sl.cube.ds;

				Point3D			np = Start.VC_Add(SLVector.Sk_Mult(s * sum));

				if (!np.equal(ds.ImagePosition)) {
					IJ.error("Serie " + sn.name + "\n unvollstndig bzw. redundant, ab Bild " + ds.Filename);
					IJ.error("Es wird dringend empfohlen, die Bilder in der Serie " + sn.name + "\n ab dem Bild " + ds.Filename + " zu lschen");
				} 

			}		// s
 
		}			// i
 
	}				// update
 

	/**
	 * Die Methode liefert die erste Schicht bezglich der definierten Ordnung zurck.
	 * 
	 * 
	 * @param Serie Index der Serie
	 * 
	 * @return Schichtobjekt vom typ "Dicom_Slice"
	 * 
	 */
	public Dicom_Slice getFirstSlice(int Serie) {
		return Slices[SeriesOffsets[Serie]];
	} 


	/**
	 * Die Methode liefert den Schichtabstand einer Serie zurck,
	 * 
	 * 
	 * @param Serie Index der Serie
	 * 
	 * @return Schichtabstand
	 * 
	 */
	public float getSliceDist(int Serie) {
		return Series[Serie].Slice_Distant;
	} 


	/**
	 * Die Methode liefert den Serienknoten im Schichtbaum zurck
	 * 
	 * 
	 * @param Serie Index der Serie
	 * 
	 * @return Serienknoten vom typ "Series_Node"
	 * 
	 * 
	 */
	public Series_Node getSerie(int Serie) {
		return Series[Serie];
	} 


	/**
	 * Die Methode berechnet die Schichtnummer (Rckgabewert) und die Matrixkoordinaten
	 * ("newx", "newy") fr
	 * einen Raumpunkt (x,y,z) innerhalb einer Serie. Dazu wird geprft, ob
	 * sich das Serienvolumen und der Raumpunkt berschneiden. Falls nicht, dann
	 * wird ein definierter Fehlerwert zurckgegeben.
	 * Die Methode wird von "getFunktionValue" aufgerufen.
	 * 
	 * @param x x-Komponente des Raumpunktes in MRT.Koordinaten
	 * @param y y-Komponente des Raumpunktes in MRT.Koordinaten
	 * @param z z-Komponente des Raumpunktes in MRT.Koordinaten
	 * @param Serie Index der zu untersuchenden Serie
	 * 
	 * @return Schichtnummer in Serie
	 * 
	 * 
	 */
	public int calc_SliceNumber(float x, float y, float z, int Serie) {
		Dicom_Slice ds = getFirstSlice(Serie);
		Series_Node sn = getSerie(Serie);
		float				sum = sn.Sum;

		// *******************************************************
		// *  Alle mglichen Fehler werden hier abgefangen !!!!  *
		// *******************************************************


		if (ds.isStrictTransversal())					// streng transversal
		 {
			float dist = z - sn.FirstImagePosZY.getX();

			// Tets ob innerhalb der betrachteten Schichten
			if (dist > (sn.MaxZ - sn.MinZ) || dist < 0.0) {
				return Error;

				// MatrixKoordinaten berechnen !
			} 
			newx = ds.Width_scale * (x - ds.MinX);
			newy = ds.Height_scale * (y - ds.MinY);
			newz = dist;

			// Matrixkoordinaten gltig ?
			if (newx < 0 || newy < 0 || newx > ds.Col_dec || newy > ds.Row_dec) {
				return Error;

				// Schichtnummer als Rckgabe, falls im gltigen Bereich
			} 
			int ind = MathTools.floor(dist / sum);

			if (ind < 0 || ind > (NumberOfSlices[Serie] - 1)) {
				return Error;
			} else {
				return ind;
			} 
		}																			// Transversal
 

		else if (ds.isTra_Cor())							// trans nach Cor gekippt
		 {
			Point2D P = new Point2D(z, y);

			P = P.VC_Min(sn.FirstImagePosZY);		// P = P - IP

			// Vektor Vergleich
			Point2D PV = new Point2D(P.getX(), P.getY());
			Point2D CV = new Point2D(ds.ColVector.getZ(), ds.ColVector.getY());

			PV.norm();
			float t1 = PV.VC_SK_Mult(sn.SliceVectorZY);
			float t2 = CV.VC_SK_Mult(PV);

			if (t1 < 0 || t2 < 0) {
				return Error;
			} 
			float dist = P.VC_SK_Mult(sn.SliceVectorZY);

			// siehe Seite 134 Analytische Geometrie
			if (dist < 0 || dist > (sum * NumberOfSlices[Serie] - sn.Slice_Distant)) {
				return Error;

				// MatrixKoordinaten berechnen !
			} 
			newy = (float) Math.sqrt(P.square() - (dist * dist));
			newx = ds.Width_scale * (x - ds.MinX);
			newy = ds.Row_scale * newy;
			newz = dist;

			// Matrixkoordinaten gltig ?
			if (newx < 0 || newy < 0 || newx > ds.Col_dec || newy > ds.Row_dec) {
				return Error;

				// Schichtnummer als Rckgabe, falls im gltigen Bereich
			} 
			int ind = MathTools.floor(dist / sum);

			if (ind < 0 || ind > (NumberOfSlices[Serie] - 1)) {
				return Error;
			} else {
				return ind;
			} 
		} else if (ds.isTra_Sag()) {
			Point2D P = new Point2D(z, x);

			P = P.VC_Min(sn.FirstImagePosZX);		// P = P - IP

			Point2D PV = new Point2D(P.getX(), P.getY());
			Point2D CV = new Point2D(ds.RowVector.getZ(), ds.RowVector.getX());

			PV.norm();
			float t1 = PV.VC_SK_Mult(sn.SliceVectorZX);
			float t2 = CV.VC_SK_Mult(PV);

			if (t1 < 0 || t2 < 0) {
				return Error;

			} 
			float dist = P.VC_SK_Mult(sn.SliceVectorZX);

			if (dist < 0.0 || dist > (sum * NumberOfSlices[Serie] - sn.Slice_Distant)) {
				return Error;

				// MatrixKoordinaten berechnen !
			} 
			newx = (float) Math.sqrt(P.square() - (dist * dist));
			newy = ds.Height_scale * (y - ds.MinY);
			newx = ds.Col_scale * newx;
			newz = dist;

			// Matrixkoordinaten gltig ?
			if (newx < 0 || newy < 0 || newx > ds.Col_dec || newy > ds.Row_dec) {
				return Error;

				// Schichtnummer als Rckgabe, falls im gltigen Bereich
			} 
			int ind = MathTools.floor(dist / sum);

			if (ind < 0 || ind > (NumberOfSlices[Serie] - 1)) {
				return Error;
			} else {
				return ind;
			} 
		} else if (ds.isStrictCoronar())			// streng Coronar
		 {
			float dist = y - sn.FirstImagePosYZ.getX();

			// Tets ob innerhalb der betrachteten Schichten
			if (dist > (sn.MaxY - sn.MinY) || dist < 0.0) {
				return Error;

				// MatrixKoordinaten berechnen !
			} 
			newx = ds.Width_scale * (x - ds.MinX);
			newy = ds.Row_scale * (ds.MaxZ - z);
			newz = dist;

			// Matrixkoordinaten gltig ?
			if (newx < 0 || newy < 0 || newx > ds.Col_dec || newy > ds.Row_dec) {
				return Error;

				// Schichtnummer als Rckgabe, falls im gltigen Bereich
			} 
			int ind = MathTools.floor(dist / sum);

			if (ind < 0 || ind > (NumberOfSlices[Serie] - 1)) {
				return Error;
			} else {
				return ind;
			} 
		}																			// Coronar
 

		else if (ds.isCor_Sag())							// Cor nach Sag gekippt
		 {
			Point2D P = new Point2D(y, x);

			P = P.VC_Min(sn.FirstImagePosYX);		// P = P - IP

			// Vektor Vergleich
			Point2D PV = new Point2D(P.getX(), P.getY());
			Point2D CV = new Point2D(ds.RowVector.getY(), ds.RowVector.getX());

			PV.norm();
			float t1 = PV.VC_SK_Mult(sn.SliceVectorYX);
			float t2 = CV.VC_SK_Mult(PV);

			if (t1 < 0 || t2 < 0) {
				return Error;

			} 
			float dist = P.VC_SK_Mult(sn.SliceVectorYX);

			if (dist < 0.0 || dist > (sum * NumberOfSlices[Serie] - sn.Slice_Distant)) {
				return Error;

				// MatrixKoordinaten berechnen !
				// hier sind wir !!!
			} 
			newx = (float) Math.sqrt(P.square() - (dist * dist));
			newy = ds.Height_scale * (ds.MaxZ - z);
			newx = ds.Col_scale * newx;
			newz = dist;

			// Matrixkoordinaten gltig ?
			if (newx < 0 || newy < 0 || newx > ds.Col_dec || newy > ds.Row_dec) {
				return Error;

				// Schichtnummer als Rckgabe, falls im gltigen Bereich
			} 
			int ind = MathTools.floor(dist / sum);

			if (ind < 0 || ind > (NumberOfSlices[Serie] - 1)) {
				return Error;
			} else {
				return ind;
			} 

		} else if (ds.isCor_Tra()) {
			Point2D P = new Point2D(y, z);

			P = P.VC_Min(sn.FirstImagePosYZ);		// P = P - IP

			// Vektor Vergleich
			Point2D PV = new Point2D(P.getX(), P.getY());
			Point2D CV = new Point2D(ds.ColVector.getY(), ds.ColVector.getZ());

			PV.norm();
			float t1 = PV.VC_SK_Mult(sn.SliceVectorYZ);
			float t2 = CV.VC_SK_Mult(PV);

			if (t1 < 0 || t2 < 0) {
				return Error;

			} 
			float dist = P.VC_SK_Mult(sn.SliceVectorYZ);

			// siehe Seite 134 Analytische Geometrie
			if (dist < 0.0 || dist > (sum * NumberOfSlices[Serie] - sn.Slice_Distant)) {
				return Error;

				// MatrixKoordinaten berechnen !
			} 
			newy = (float) Math.sqrt(P.square() - (dist * dist));
			newx = ds.Width_scale * (x - ds.MinX);
			newy = ds.Row_scale * newy;
			newz = dist;

			// Matrixkoordinaten gltig ?
			if (newx < 0 || newy < 0 || newx > ds.Col_dec || newy > ds.Row_dec) {
				return Error;

				// Schichtnummer als Rckgabe, falls im gltigen Bereich
			} 
			int ind = MathTools.floor(dist / sum);

			if (ind < 0 || ind > (NumberOfSlices[Serie] - 1)) {
				return Error;
			} else {
				return ind;
			} 
		} 

		if (ds.isStrictSagittal())						// streng sagittal
		 {

			// sagtest
			float dist = sn.FirstImagePosXY.getX() - x;

			// Tets ob innerhalb der betrachteten Schichten
			if (dist > (sn.MaxX - sn.MinX) || dist < 0.0) {
				return Error;

				// Test ob Zielpunkt zwischen zwei Schichten liegt !
				// MatrixKoordinaten berechnen !
			} 
			newx = ds.Width_scale * (y - ds.MinY);
			newy = ds.Height_scale * (ds.MaxZ - z);		// 4 Stunden den Fehler gesucht !!!!!!
			newz = dist;

			// Matrixkoordinaten gltig ?
			if (newx < 0 || newy < 0 || newx > ds.Col_dec || newy > ds.Row_dec) {
				return Error;

				// Schichtnummer als Rckgabe, falls im gltigen Bereich
			} 
			int ind = MathTools.floor(dist / sum);

			if (ind < 0 || ind > (NumberOfSlices[Serie] - 1)) {
				return Error;
			} else {
				return ind;
			} 
		}																			// Transversal
 

		else if (ds.isSag_Tra())							// Sag nach Tra gekippt
		 {
			Point2D P = new Point2D(x, z);

			P = P.VC_Min(sn.FirstImagePosXZ);		// P = P - IP

			// Vektor Vergleich
			Point2D PV = new Point2D(P.getX(), P.getY());
			Point2D CV = new Point2D(ds.ColVector.getX(), ds.ColVector.getZ());

			PV.norm();
			float t1 = PV.VC_SK_Mult(sn.SliceVectorXZ);
			float t2 = CV.VC_SK_Mult(PV);

			if (t1 < 0 || t2 < 0) {
				return Error;

			} 
			float dist = P.VC_SK_Mult(sn.SliceVectorXZ);

			// siehe Seite 134 Analytische Geometrie
			if (dist < 0.0 || dist > (sum * NumberOfSlices[Serie] - sn.Slice_Distant)) {
				return Error;

				// MatrixKoordinaten berechnen !
			} 
			newy = (float) Math.sqrt(P.square() - (dist * dist));
			newx = ds.Width_scale * (y - ds.MinY);
			newy = ds.Row_scale * newy;
			newz = dist;

			// Matrixkoordinaten gltig ?
			if (newx < 0 || newy < 0 || newx > ds.Col_dec || newy > ds.Row_dec) {
				return Error;

				// Schichtnummer als Rckgabe, falls im gltigen Bereich
			} 
			int ind = MathTools.floor(dist / sum);

			if (ind < 0 || ind > (NumberOfSlices[Serie] - 1)) {
				return Error;
			} else {
				return ind;
			} 
		} else if (ds.isSag_Cor()) {
			Point2D P = new Point2D(x, y);

			P = P.VC_Min(sn.FirstImagePosXY);		// P = P - IP

			// Vektor Vergleich
			Point2D PV = new Point2D(P.getX(), P.getY());
			Point2D CV = new Point2D(ds.RowVector.getX(), ds.RowVector.getY());

			PV.norm();
			float t1 = PV.VC_SK_Mult(sn.SliceVectorXY);
			float t2 = CV.VC_SK_Mult(PV);

			if (t1 < 0 || t2 < 0) {
				return Error;

			} 
			float dist = P.VC_SK_Mult(sn.SliceVectorXY);

			if (dist < 0.0 || dist > (sum * NumberOfSlices[Serie] - sn.Slice_Distant)) {
				return Error;

				// MatrixKoordinaten berechnen !
			} 
			newx = (float) Math.sqrt(P.square() - (dist * dist));
			newy = ds.Height_scale * (ds.MaxZ - z);
			newx = ds.Col_scale * newx;
			newz = dist;

			// Matrixkoordinaten gltig ?
			if (newx < 0 || newy < 0 || newx > ds.Col_dec || newy > ds.Row_dec) {
				return Error;

				// Schichtnummer als Rckgabe, falls im gltigen Bereich
			} 
			int ind = MathTools.floor(dist / sum);

			if (ind < 0 || ind > (NumberOfSlices[Serie] - 1)) {
				return Error;
			} else {
				return ind;
			} 
		} else {
			IJ.write("Fehler bei Zuordnung der Orientierung");
			return Error;
		} 
	}																				// getIndex
 

	/**
	 * Die Methode berechnet den Funktionswert (Grauwert) eines Raumpunktes
	 * (x,y,z) bezglich einer Serie. Eine Ausnahme wird ausgelst, falls der
	 * Raumpunkt nicht durch das Serienvolumen abgedeckt wird. Je nach globaler Einstellung
	 * der Interpolationsmethode, werden weitere Grauwertnachbarn des Raumpunktes (x,y,z)
	 * bestimmt und ein interpolierter Wert ausgegeben.
	 * 
	 * @param x x-Komponente des Raumpunktes
	 * @param y y-Komponente des Raumpunktes
	 * @param z z-Komponente des Raumpunktes
	 * @param Serie Index der Serie
	 * 
	 * @return resultierender interpolierter Grauwert
	 * 
	 * @throws IllegalArgumentException
	 * 
	 * 
	 */
	private float getFunktionValue(float x, float y, float z, int Serie) throws IllegalArgumentException {
		float				val;
		float[]			values;
		int[][]			Matrix;
		Dicom_Slice ds;



		int					index = calc_SliceNumber(x, y, z, Serie);

		if (index == Error) {
			throw new IllegalArgumentException("Fehler bei Funktionsauswertung");
		} 
		akt_index = SeriesOffsets[Serie] + index;
		ds = Slices[akt_index];
		Series_Node sn = getSerie(Serie);

		float				sum = sn.Sum;

		Slice_Position = newz - (MathTools.floor((newz / sum)) * sum);

		// zwischen zwei schichten
		if (Slice_Position > ds.SliceThicknes) {
			Slice_Position = -1;

		} 
		if (Global_Options.Interpolationmodel == 2)		// Tri Lienar !
		 {
			float dt = index * sum + ds.SliceThicknes / 2;

			if (newz - dt > 0)													// RECHTS VON DT
			 {

				// zwischen ind und ind + 1
				// berechne Interpolationsfaktor
				factor = (newz - dt) / sum;
				index2 = index + 1;
				if (index2 > (NumberOfSlices[Serie] - 1)) {

					// keine Interpolation
					index2 = index;
					factor = 0;
				} 
			} else {

				// zwischen ind und ind - 1
				// berechne Interpolationsfaktor
				factor = (dt - newz) / sum;
				index2 = index - 1;
				if (index2 < 0) {

					// keine Interpolation
					index2 = index;
					factor = 0;
				} 
			} 
		}																							// Index2 Berechnung
 
		else if (Global_Options.Interpolationmodel == 4 || Global_Options.Interpolationmodel >= 6)	// Tri
		 {

			index2 = index - 1;
			index3 = index + 1;
			index4 = index + 2;

			if (index2 < 0) {
				index2 = index;
			} 
			if (index3 > (NumberOfSlices[Serie] - 1)) {
				index3 = index;
			} 

			if (index4 > (NumberOfSlices[Serie] - 1)) {
				index4 = index3;
			} 

			factor = (newz - ((index - 1) * sum) - ds.SliceThicknes / 2) / (sum + sum) * 2;

		}																							// Index2, Index3, Index4  Berechnung
 

		// Test ob Zielpunkt zwischen zwei Schichten liegt !
		// float sum = sn.Slice_Distant + ds.SliceThicknes ;
		// float tmp = (newz / sum) ;
		// if ( newz - Math.floor(tmp) * sum > ds.SliceThicknes ) return Error ;




		// nach dem Mittelwersatz der Integralrechnung


		switch (Global_Options.Interpolationmodel) {

		// Nchster Nachbar
		case 0:
			int tmp = ds.getPixel2(Math.round(newx), Math.round(newy));

			return tmp / ds.Voxelvolume;

		// Bi-Lineare Interpolation

		case 1:
			int		xval1 = (int) (newx);
			int		yval1 = (int) (newy);
			int		xval2 = xval1 + 1;
			int		yval2 = yval1 + 1;

			int		val1 = ds.getPixel2(xval1, yval1);
			int		val2 = ds.getPixel2(xval2, yval1);
			int		val3 = ds.getPixel2(xval1, yval2);
			int		val4 = ds.getPixel2(xval2, yval2);


			// Interpolationsfaktoren
			float intfacx = newx - xval1;
			float intfacy = newy - yval1;


			float val12 = MathTools.LI2(val1, val2, intfacx);
			float val34 = MathTools.LI2(val3, val4, intfacx);

			val = MathTools.LI2(val12, val34, intfacy);
			return val / ds.Voxelvolume;

		// Tri-Lineare Interpolation
		case 2:
			if (NumberOfSlices[Serie] < 2) {
				index2 = index;


			} 
			xval1 = (int) (newx);
			yval1 = (int) (newy);
			xval2 = xval1 + 1;
			yval2 = yval1 + 1;

			val1 = ds.getPixel2(xval1, yval1);
			val2 = ds.getPixel2(xval2, yval1);
			val3 = ds.getPixel2(xval1, yval2);
			val4 = ds.getPixel2(xval2, yval2);


			// Interpolationsfaktoren
			intfacx = newx - xval1;
			intfacy = newy - yval1;


			val12 = MathTools.LI2(val1, val2, intfacx);
			val34 = MathTools.LI2(val3, val4, intfacx);

			float val1234 = MathTools.LI2(val12, val34, intfacy);

			ds = Slices[SeriesOffsets[Serie] + index2];

			val1 = ds.getPixel2(xval1, yval1);
			val2 = ds.getPixel2(xval2, yval1);
			val3 = ds.getPixel2(xval1, yval2);
			val4 = ds.getPixel2(xval2, yval2);

			val12 = MathTools.LI2(val1, val2, intfacx);
			val34 = MathTools.LI2(val3, val4, intfacx);

			float val5678 = MathTools.LI2(val12, val34, intfacy);


			val = MathTools.LI2(val1234, val5678, factor);

			return val / ds.Voxelvolume;



		// Bi-Lagrange Interpolation
		case 3:

			int n = MathTools.floor((Global_Options.NumberOfPoints - 1) / 2);

			values = new float[Global_Options.NumberOfPoints];
			Matrix = ds.getMatrix(newx, newy, Global_Options.NumberOfPoints, n);
			float facy = newy - MathTools.floor(newy) + n;
			float facx = newx - MathTools.floor(newx) + n;

			for (int i = 0; i < Global_Options.NumberOfPoints; i++) {
				values[i] = MathTools.Lagrange(Matrix[i], facy);

			} 

			val1234 = MathTools.Lagrange(values, facx);
			return val1234 / ds.Voxelvolume;



		// Tri-Lagrange Interpolation
		case 4:
			n = MathTools.floor((Global_Options.NumberOfPoints - 1) / 2);
			values = new float[Global_Options.NumberOfPoints];
			facy = newy - MathTools.floor(newy) + n;
			facx = newx - MathTools.floor(newx) + n;

			// Schicht 1
			ds = Slices[SeriesOffsets[Serie] + index2];
			Matrix = ds.getMatrix(newx, newy, Global_Options.NumberOfPoints, n);
			for (int i = 0; i < Global_Options.NumberOfPoints; i++) {
				values[i] = MathTools.Lagrange(Matrix[i], facy);
			} 
			val1234 = MathTools.Lagrange(values, facx);

			// Schicht 2
			ds = Slices[SeriesOffsets[Serie] + index];
			Matrix = ds.getMatrix(newx, newy, Global_Options.NumberOfPoints, n);
			for (int i = 0; i < Global_Options.NumberOfPoints; i++) {
				values[i] = MathTools.Lagrange(Matrix[i], facy);
			} 
			val5678 = MathTools.Lagrange(values, facx);

			// Schicht 3
			ds = Slices[SeriesOffsets[Serie] + index3];
			Matrix = ds.getMatrix(newx, newy, Global_Options.NumberOfPoints, n);
			for (int i = 0; i < Global_Options.NumberOfPoints; i++) {
				values[i] = MathTools.Lagrange(Matrix[i], facy);
			} 
			float		val9012 = MathTools.Lagrange(values, facx);


			// berechne Wert
			float[] values_global = {
				val1234, val5678, val9012
			};

			val = MathTools.Lagrange(values_global, factor);
			return val / ds.Voxelvolume;

		// Bi-Kubische Spline Interpolation
		case 5:

			values = new float[4];
			Matrix = ds.getMatrix(newx, newy, 4, 1);
			facy = newy - MathTools.floor(newy);
			facx = newx - MathTools.floor(newx);
			float h1 = ds.PixelSpacingColumn;

			for (int i = 0; i < 4; i++) {
				values[i] = MathTools.KubicSpline(Matrix[i], h1, facy);
			} 

			val = MathTools.KubicSpline(values, ds.PixelSpacingRow, facx);
			return val / ds.Voxelvolume;

		// Tri-Kubische Spline Interpolation
		case 6:
			values = new float[4];

			// Schicht 1
			ds = Slices[SeriesOffsets[Serie] + index2];
			Matrix = ds.getMatrix(newx, newy, 4, 1);
			facy = newy - MathTools.floor(newy);
			facx = newx - MathTools.floor(newx);
			h1 = ds.PixelSpacingColumn;
			for (int i = 0; i < 4; i++) {
				values[i] = MathTools.KubicSpline(Matrix[i], h1, facy);
			} 
			val1234 = MathTools.KubicSpline(values, ds.PixelSpacingRow, facx);

			// Schicht 2
			ds = Slices[SeriesOffsets[Serie] + index];
			Matrix = ds.getMatrix(newx, newy, 4, 1);
			for (int i = 0; i < 4; i++) {
				values[i] = MathTools.KubicSpline(Matrix[i], h1, facy);
			} 
			val5678 = MathTools.KubicSpline(values, ds.PixelSpacingRow, facx);

			// Schicht 3
			ds = Slices[SeriesOffsets[Serie] + index3];
			Matrix = ds.getMatrix(newx, newy, 4, 1);
			for (int i = 0; i < 4; i++) {
				values[i] = MathTools.KubicSpline(Matrix[i], h1, facy);
			} 
			val9012 = MathTools.KubicSpline(values, ds.PixelSpacingRow, facx);

			// Schicht 4
			ds = Slices[SeriesOffsets[Serie] + index4];
			Matrix = ds.getMatrix(newx, newy, 4, 1);
			for (int i = 0; i < 4; i++) {
				values[i] = MathTools.KubicSpline(Matrix[i], h1, facy);
			} 
			float		val3456 = MathTools.KubicSpline(values, ds.PixelSpacingRow, facx);

			// berechne Wert
			float[] values_global2 = {
				val1234, val5678, val9012, val3456
			};

			val = MathTools.KubicSpline(values_global2, ds.PixelSpacingRow, factor);

			return val / ds.Voxelvolume;


		// Tri-Kubische Spline Interpolation allgemein
		case 7:

			n = MathTools.floor((Global_Options.NumberOfPoints_Kubic - 1) / 2);
			values = new float[Global_Options.NumberOfPoints_Kubic];

			// Schicht 1
			ds = Slices[SeriesOffsets[Serie] + index2];
			Matrix = ds.getMatrix(newx, newy, Global_Options.NumberOfPoints_Kubic, n);
			facy = newy - MathTools.floor(newy) + n;
			facx = newx - MathTools.floor(newx) + n;
			h1 = ds.PixelSpacingColumn;

			for (int i = 0; i < Global_Options.NumberOfPoints_Kubic; i++) {
				values[i] = MathTools.KubicSpline2(Global_Options.NumberOfPoints_Kubic, h1, Matrix[i], facy);
			} 
			val1234 = MathTools.KubicSpline2(Global_Options.NumberOfPoints_Kubic, ds.PixelSpacingRow, values, facx);

			// Schicht 2
			ds = Slices[SeriesOffsets[Serie] + index];
			Matrix = ds.getMatrix(newx, newy, Global_Options.NumberOfPoints_Kubic, n);
			for (int i = 0; i < Global_Options.NumberOfPoints_Kubic; i++) {
				values[i] = MathTools.KubicSpline2(Global_Options.NumberOfPoints_Kubic, h1, Matrix[i], facy);
			} 
			val5678 = MathTools.KubicSpline2(Global_Options.NumberOfPoints_Kubic, ds.PixelSpacingRow, values, facx);

			// Schicht 3
			ds = Slices[SeriesOffsets[Serie] + index3];
			Matrix = ds.getMatrix(newx, newy, Global_Options.NumberOfPoints_Kubic, n);
			for (int i = 0; i < Global_Options.NumberOfPoints_Kubic; i++) {
				values[i] = MathTools.KubicSpline2(Global_Options.NumberOfPoints_Kubic, h1, Matrix[i], facy);
			} 
			val9012 = MathTools.KubicSpline2(Global_Options.NumberOfPoints_Kubic, ds.PixelSpacingRow, values, facx);

			// Schicht 4
			ds = Slices[SeriesOffsets[Serie] + index4];
			Matrix = ds.getMatrix(newx, newy, Global_Options.NumberOfPoints_Kubic, n);
			for (int i = 0; i < Global_Options.NumberOfPoints_Kubic; i++) {
				values[i] = MathTools.KubicSpline2(Global_Options.NumberOfPoints_Kubic, h1, Matrix[i], facy);
			} 
			val3456 = MathTools.KubicSpline2(Global_Options.NumberOfPoints_Kubic, ds.PixelSpacingRow, values, facx);

			// berechne Wert
			float[] values_global3 = {
				val1234, val5678, val9012, val3456
			};

			val = MathTools.KubicSpline2(Global_Options.NumberOfPoints_Kubic, ds.PixelSpacingRow, values_global3, factor + 1);

			return val / ds.Voxelvolume;

		default:
			return 0;
		}
	}		// getFunktion
 

	/**
	 * Die Methode berechnet den Funktionswert (Grauwert) eines Raumpunktes
	 * p bezglich einer Serie. Eine Ausnahme wird ausgelst, falls der
	 * Raumpunkt nicht durch das Serienvolumen abgedeckt wird. Je nach globaler Einstellung
	 * der Interpolationsmethode, werden weitere Grauwertnachbarn des Raumpunktes p
	 * bestimmt und ein interpolierter Wert ausgegeben.
	 * 
	 * @param p Ortsvektor des Grauwertpunktes
	 * 
	 * @param Serie Index der Serie
	 * 
	 * @return resultierender interpolierter Grauwert
	 * 
	 * 
	 */
	private float getFunktionValue(Point3D p, int Serie) {
		return getFunktionValue(p.getX(), p.getY(), p.getZ(), Serie);
	} 


	/**
	 * Die Methode berechnet den Funktionswert (Grauwert) eines Raumpunktes
	 * p  mit Hilfe der markierten Serien. Dazu werden zunchst
	 * die Grauwerte der einzelnen markierten Serien ermittelt und je nach globaler Einstellung
	 * kombiniert.
	 * Falls in den globalen Optionen eine komplexere Berechnungmethode ausgewhlt wurde (Gewichtete Summen oder Lineare Gleichungen), wird zunchst berprft,
	 * ob die erforderliche Anzahl an markierten Serien zur Verfgung steht. Falls ja, kommen
	 * spezielle Algorithmen zur Anwendung die meherer Serien gleichzeitig betrachten aus auswerten.
	 * Dadurch kann ein genauerer Grauwert berechnet werden.
	 * 
	 * 
	 * @param p Ortsvektor des Raumpunktes
	 * 
	 * @return Grauwert an der Stelle p
	 * 
	 * 
	 */
	public float getFunktionValue(Point3D p) {
		return getFunktionValue(p.getX(), p.getY(), p.getZ());
	} 


	/**
	 * Die Methode berechnet den Funktionswert (Grauwert) eines Raumpunktes
	 * (x,y,z)  mit Hilfe der markierten Serien. Dazu werden zunchst
	 * die Grauwerte der einzelnen markierten Serien ermittelt und je nach globaler Einstellung
	 * kombiniert.
	 * Falls in den globalen Optionen eine komplexere Berechnungmethode ausgewhlt wurde (Gewichtete Summen oder Lineare Gleichungen), wird zunchst berprft,
	 * ob die erforderliche Anzahl an markierten Serien zur Verfgung steht. Falls ja, kommen
	 * spezielle Algorithmen zur Anwendung die meherer Serien gleichzeitig betrachten aus auswerten.
	 * Dadurch kann ein genauerer Grauwert berechnet werden.
	 * 
	 * @param x x-Komponente des Raumpunktes
	 * @param y y-Komponente des Raumpunktes
	 * @param z z-Komponente des Raumpunktes
	 * 
	 * @return Grauwert an der Stelle (x,y,z)
	 * 
	 * 
	 */
	public float getFunktionValue(float x, float y, float z) {

		// Kombination von Funktionswerte der Serien je nach Methode
		float val = 0;
		int		count = 0;


		for (int i = 0; i < NumberOfSelectesSeries; i++) {
			try {
				Functionvalues[count] = getFunktionValue(x, y, z, SelectedSlices[i]);
				val += Functionvalues[count];
				SeriesPosition[count] = akt_index;
				Slice_Positions[count] = Slice_Position;
				Series_sel[count] = SelectedSlices[i];
				count++;
			} catch (Exception e) {		// entsprechender Funktionswert nicht in Schicht verfgbar !
			} 
		}		// for
 
		if (count == 0) {
			return 0;

		} 
		switch (Global_Options.ApproxMethod) {
		case 0:																	// Additiv verknpft
			return val;

		case 1:																	// Mittelwert
			return val / count;

		case 2:																	// gewichtet Summen

			// mindestens 2 Serien werden bentigt
			if (count < 2) {
				return val / count;


			} 
			float[] Matrix_values = new float[Global_Options.NumberOfPairs];
			int			Matrix_counter = 0;
			float		angle = 0.0f;

			for (int x1 = 0; x1 < count; x1++)		// Paar x
			 {
				for (int x2 = 0; x2 < count; x2++)	// Paar y
				 {

					// paarweiser Vergleich x1 , x2
					if (x1 == x2) {
						continue;

					} 
					Dicom_Slice ds1 = Slices[SeriesPosition[x1]];
					Dicom_Slice ds2 = Slices[SeriesPosition[x2]];



					float				fac = Global_Options.getMatrixFactors(ds1, ds2);
					Point3D			Origin = new Point3D(x, y, z);
					Point3D			Vector_1, Vector_2, p0;
					float				thickness_1, thickness_2, Res_1, Res_2, trans_1, trans_2;

					Vector_1 = Vector_2 = p0 = MathTools.Nullvector;
					thickness_1 = thickness_2 = Res_1 = Res_2 = 0;
					float		fl;
					float[] Integral_2 = new float[Global_Options.Matrix_Resolution];
					float[] Integral_1 = new float[Global_Options.Matrix_Resolution];
					float		sum_x = 0;
					float		sum_z = 0;

					fl = trans_1 = trans_2 = 0;


					if (ds1.isTransversal() && ds2.isSagittal()) {

						if (Slice_Positions[x1] < 0 || Slice_Positions[x2] < 0) {
							return val / count;

							// Berechne Schnittebne E1

							// E1 = (x - o) x p
						} 
						Point3D o = Origin;
						Point3D p = ds1.SliceVector.VC_Mult(ds2.SliceVector);

						// Berechne g1  fr p0
						Point3D g1o = new Point3D(0, 0, 0);
						Point3D g1m = new Point3D(0, 0, 0);

						MathTools.calcIntersection_PP(ds1.ImagePosition, ds1.SliceVector, ds2.ImagePosition, ds2.ColVector, ds2.RowVector, g1o, g1m);

						// Berechne g2  fr p1
						Point3D g2o = new Point3D(0, 0, 0);
						Point3D g2m = new Point3D(0, 0, 0);

						MathTools.calcIntersection_PP(ds1.ImagePosition2, ds1.SliceVector, ds2.ImagePosition, ds2.ColVector, ds2.RowVector, g2o, g2m);

						// Berechne g3  fr p2
						Point3D g3o = new Point3D(0, 0, 0);
						Point3D g3m = new Point3D(0, 0, 0);

						MathTools.calcIntersection_PP(ds1.ImagePosition, ds1.SliceVector, ds2.ImagePosition2, ds2.ColVector, ds2.RowVector, g3o, g3m);

						// Berechne Punkte
						p0 = MathTools.calcIntersection_GP(g1o, g1m, Origin, p);
						Point3D ps1 = MathTools.calcIntersection_GP(g2o, g2m, Origin, p);
						Point3D ps2 = MathTools.calcIntersection_GP(g3o, g3m, Origin, p);

						Vector_1 = ps1.VC_Min(p0);
						Vector_2 = ps2.VC_Min(p0);
						thickness_1 = Vector_1.Length();
						thickness_2 = Vector_2.Length();
						Res_2 = thickness_2 / Global_Options.Matrix_Resolution;
						Res_1 = thickness_1 / Global_Options.Matrix_Resolution;
						Vector_1.norm();
						Vector_2.norm();
						trans_1 = Slice_Positions[x1] / ds1.SliceThicknes * thickness_1;
						trans_2 = Slice_Positions[x2] / ds2.SliceThicknes * thickness_2;

						// Berechne Flcheninhalt der Zellen (fl)
						Point3D cs = (Vector_1.Sk_Mult(Res_1)).VC_Mult((Vector_2.Sk_Mult(Res_2)));

						fl = (float) cs.Length();

						for (int mr = 0; mr < Global_Options.Matrix_Resolution; mr++) {

							Point3D p1 = p0.VC_Add(Vector_2.Sk_Mult(mr * Res_2));
							Point3D p2 = p0.VC_Add(Vector_1.Sk_Mult(mr * Res_1));

							try {
								Integral_1[mr] = getIntegral(p1, Res_2, 1, thickness_1, Vector_2, MathTools.Nullvector, Vector_1, Series_sel[x1]);
								Integral_2[mr] = getIntegral(p2, thickness_2, 1, Res_1, Vector_2, MathTools.Nullvector, Vector_1, Series_sel[x2]);
								sum_z += Integral_1[mr];
								sum_x += Integral_2[mr];
							} catch (Exception excep) {
								return val / count;
							} 

						}																// mr
 					 }																// tra / sag
 
					else if (ds1.isTransversal() && ds2.isCoronar()) {

						if (Slice_Positions[x1] < 0 || Slice_Positions[x2] < 0) {
							return val / count;

							// Berechne Schnittebne E1

							// E1 = (x - o) x p
						} 
						Point3D o = Origin;
						Point3D p = ds1.SliceVector.VC_Mult(ds2.SliceVector);

						// Berechne g1  fr p0
						Point3D g1o = new Point3D(0, 0, 0);
						Point3D g1m = new Point3D(0, 0, 0);

						MathTools.calcIntersection_PP(ds1.ImagePosition, ds1.SliceVector, ds2.ImagePosition, ds2.ColVector, ds2.RowVector, g1o, g1m);

						// Berechne g2  fr p1
						Point3D g2o = new Point3D(0, 0, 0);
						Point3D g2m = new Point3D(0, 0, 0);

						MathTools.calcIntersection_PP(ds1.ImagePosition, ds1.SliceVector, ds2.ImagePosition2, ds2.ColVector, ds2.RowVector, g2o, g2m);

						// Berechne g3  fr p2
						Point3D g3o = new Point3D(0, 0, 0);
						Point3D g3m = new Point3D(0, 0, 0);

						MathTools.calcIntersection_PP(ds1.ImagePosition2, ds1.SliceVector, ds2.ImagePosition, ds2.ColVector, ds2.RowVector, g3o, g3m);

						// Berechne Punkte
						p0 = MathTools.calcIntersection_GP(g1o, g1m, Origin, p);
						Point3D ps1 = MathTools.calcIntersection_GP(g2o, g2m, Origin, p);
						Point3D ps2 = MathTools.calcIntersection_GP(g3o, g3m, Origin, p);

						Vector_1 = ps1.VC_Min(p0);
						Vector_2 = ps2.VC_Min(p0);
						thickness_1 = Vector_1.Length();
						thickness_2 = Vector_2.Length();
						Res_2 = thickness_2 / Global_Options.Matrix_Resolution;
						Res_1 = thickness_1 / Global_Options.Matrix_Resolution;
						Vector_1.norm();
						Vector_2.norm();
						trans_1 = Slice_Positions[x2] / ds2.SliceThicknes * thickness_1;
						trans_2 = Slice_Positions[x1] / ds1.SliceThicknes * thickness_2;

						// Berechne Flcheninhalt der Zellen (fl)
						Point3D cs = (Vector_1.Sk_Mult(Res_1)).VC_Mult((Vector_2.Sk_Mult(Res_2)));

						fl = (float) cs.Length();

						for (int mr = 0; mr < Global_Options.Matrix_Resolution; mr++) {

							Point3D p1 = p0.VC_Add(Vector_2.Sk_Mult(mr * Res_2));
							Point3D p2 = p0.VC_Add(Vector_1.Sk_Mult(mr * Res_1));

							try {
								Integral_1[mr] = getIntegral(p1, 1, thickness_1, Res_2, MathTools.Nullvector, Vector_1, Vector_2, Series_sel[x1]);
								Integral_2[mr] = getIntegral(p2, 1, Res_1, thickness_2, MathTools.Nullvector, Vector_1, Vector_2, Series_sel[x2]);
								sum_x += Integral_1[mr];
								sum_z += Integral_2[mr];
							} catch (Exception excep) {
								return val / count;
							} 

						}																// mr
 					 }																// tra / cor
 
					else if (ds1.isCoronar() && ds2.isSagittal()) {

						if (Slice_Positions[x1] < 0 || Slice_Positions[x2] < 0) {
							return val / count;

							// Berechne Schnittebne E1

							// E1 = (x - o) x p
						} 
						Point3D o = Origin;
						Point3D p = ds1.SliceVector.VC_Mult(ds2.SliceVector);

						// Berechne g1  fr p0
						Point3D g1o = new Point3D(0, 0, 0);
						Point3D g1m = new Point3D(0, 0, 0);

						MathTools.calcIntersection_PP(ds1.ImagePosition, ds1.SliceVector, ds2.ImagePosition, ds2.ColVector, ds2.RowVector, g1o, g1m);

						// Berechne g2  fr p1
						Point3D g2o = new Point3D(0, 0, 0);
						Point3D g2m = new Point3D(0, 0, 0);

						MathTools.calcIntersection_PP(ds1.ImagePosition2, ds1.SliceVector, ds2.ImagePosition, ds2.ColVector, ds2.RowVector, g2o, g2m);

						// Berechne g3  fr p2
						Point3D g3o = new Point3D(0, 0, 0);
						Point3D g3m = new Point3D(0, 0, 0);

						MathTools.calcIntersection_PP(ds1.ImagePosition, ds1.SliceVector, ds2.ImagePosition2, ds2.ColVector, ds2.RowVector, g3o, g3m);

						// Berechne Punkte
						p0 = MathTools.calcIntersection_GP(g1o, g1m, Origin, p);
						Point3D ps1 = MathTools.calcIntersection_GP(g2o, g2m, Origin, p);
						Point3D ps2 = MathTools.calcIntersection_GP(g3o, g3m, Origin, p);

						Vector_1 = ps1.VC_Min(p0);
						Vector_2 = ps2.VC_Min(p0);
						thickness_1 = Vector_1.Length();
						thickness_2 = Vector_2.Length();
						Res_2 = thickness_2 / Global_Options.Matrix_Resolution;
						Res_1 = thickness_1 / Global_Options.Matrix_Resolution;
						Vector_1.norm();
						Vector_2.norm();
						trans_1 = Slice_Positions[x1] / ds1.SliceThicknes * thickness_1;
						trans_2 = Slice_Positions[x2] / ds2.SliceThicknes * thickness_2;

						// Berechne Flcheninhalt der Zellen (fl)
						Point3D cs = (Vector_1.Sk_Mult(Res_1)).VC_Mult((Vector_2.Sk_Mult(Res_2)));

						fl = (float) cs.Length();

						for (int mr = 0; mr < Global_Options.Matrix_Resolution; mr++) {

							Point3D p1 = p0.VC_Add(Vector_2.Sk_Mult(mr * Res_2));
							Point3D p2 = p0.VC_Add(Vector_1.Sk_Mult(mr * Res_1));

							try {
								Integral_1[mr] = getIntegral(p1, Res_2, thickness_1, 1, Vector_2, Vector_1, MathTools.Nullvector, Series_sel[x1]);
								Integral_2[mr] = getIntegral(p2, thickness_2, Res_1, 1, Vector_2, Vector_1, MathTools.Nullvector, Series_sel[x2]);
								sum_z += Integral_1[mr];
								sum_x += Integral_2[mr];
							} catch (Exception excep) {
								return val / count;
							} 

						}																// mr
 					 }																// cor / sag
 
					else															// keine Ansicht Zuordnung mglich
					 {
						continue;
					} 

					// Alles hat geklappt
					int		split_2 = (int) (trans_2 * Global_Options.Matrix_Resolution / thickness_2);
					int		split_1 = (int) (trans_1 * Global_Options.Matrix_Resolution / thickness_1);

					float sum_all = sum_x + sum_z;
					float vl1 = (Integral_1[split_2] / (sum_x + 1)) * (Integral_2[split_1]);
					float vl2 = (Integral_2[split_1] / (sum_z + 1)) * (Integral_1[split_2]);


					// Faktor fr die Reihen / Spalten
					Matrix_values[Matrix_counter] = fac * vl1 / (fl) + (1.0f - fac) * vl2 / (fl);

					// Faktor fr das Paar
					Matrix_values[Matrix_counter] *= Global_Options.getFactors(ds1, ds2);
					Matrix_counter++;

					// return (vl1 + vl2) / (2* Res_2 * Res_1)  ;


				}																		// x1
 			 }																		// x2
 
			val = 0;
			for (int i = 0; i < Matrix_counter; i++) {
				val += Matrix_values[i];
			} 

			// IJ.write ("Winkel : " + new Float (angle).toString());
			return val / Matrix_counter;

		// --------------------------------------------------------------------------

		case 3:																	// Lineare Gleichungssysteme


			// mindestens 3 Serien werden bentigt
			if (count != 3) {
				return val / count;

			} 
			Matrix_values = new float[Global_Options.NumberOfPairs];
			Matrix_counter = 0;
			float factor_eq = 0;

			angle = 0.0f;
			int x3 = 0;

			for (int x1 = 0; x1 < count; x1++)		// Paar x
			 {
				for (int x2 = 0; x2 < count; x2++)	// Paar y
				 {

					// paarweiser Vergleich x1 , x2
					if (x1 == x2) {
						continue;

						// suche x3 als 3. Parameterschicht
					} 
					x3 = 0;
					while (x3 == x2 || x3 == x1) {
						x3++;
					} 


					Dicom_Slice ds1 = Slices[SeriesPosition[x1]];
					Dicom_Slice ds2 = Slices[SeriesPosition[x2]];
					Dicom_Slice ds3 = Slices[SeriesPosition[x3]];


					// float fac = Global_Options.getMatrixFactors(ds1,ds2) ;
					Point3D			Origin = new Point3D(x, y, z);
					Point3D			Vector_1, Vector_2, p0;
					float				thickness_1, thickness_2, Res_1, Res_2, trans_1, trans_2;

					Vector_1 = Vector_2 = p0 = MathTools.Nullvector;
					thickness_1 = thickness_2 = Res_1 = Res_2 = 0;
					float		fl;
					float[] Integral_2 = new float[Global_Options.Matrix_Resolution];
					float[] Integral_1 = new float[Global_Options.Matrix_Resolution];
					float		sum_x = 0;
					float		sum_z = 0;

					fl = trans_1 = trans_2 = 0;

					// Aufstellen des Vektors b
					int				cn = Global_Options.Matrix_Resolution * 2 + Global_Options.Matrix_Resolution * Global_Options.Matrix_Resolution;
					double[]	b = new double[cn];

					if (ds1.isTransversal() && ds2.isSagittal() && Global_Options.TS_C) {

						if (Slice_Positions[x1] < 0 || Slice_Positions[x2] < 0) {
							return val / count;

						} 
						factor_eq = Global_Options.Factor_eq_TS;

						// Berechne Schnittebne E1

						// E1 = (x - o) x p
						Point3D o = Origin;
						Point3D p = ds1.SliceVector.VC_Mult(ds2.SliceVector);

						// Berechne g1  fr p0
						Point3D g1o = new Point3D(0, 0, 0);
						Point3D g1m = new Point3D(0, 0, 0);

						MathTools.calcIntersection_PP(ds1.ImagePosition, ds1.SliceVector, ds2.ImagePosition, ds2.ColVector, ds2.RowVector, g1o, g1m);

						// Berechne g2  fr p1
						Point3D g2o = new Point3D(0, 0, 0);
						Point3D g2m = new Point3D(0, 0, 0);

						MathTools.calcIntersection_PP(ds1.ImagePosition2, ds1.SliceVector, ds2.ImagePosition, ds2.ColVector, ds2.RowVector, g2o, g2m);

						// Berechne g3  fr p2
						Point3D g3o = new Point3D(0, 0, 0);
						Point3D g3m = new Point3D(0, 0, 0);

						MathTools.calcIntersection_PP(ds1.ImagePosition, ds1.SliceVector, ds2.ImagePosition2, ds2.ColVector, ds2.RowVector, g3o, g3m);

						// Berechne Punkte
						p0 = MathTools.calcIntersection_GP(g1o, g1m, Origin, p);
						Point3D ps1 = MathTools.calcIntersection_GP(g2o, g2m, Origin, p);
						Point3D ps2 = MathTools.calcIntersection_GP(g3o, g3m, Origin, p);

						Vector_1 = ps1.VC_Min(p0);
						Vector_2 = ps2.VC_Min(p0);
						thickness_1 = Vector_1.Length();
						thickness_2 = Vector_2.Length();
						Res_2 = thickness_2 / Global_Options.Matrix_Resolution;
						Res_1 = thickness_1 / Global_Options.Matrix_Resolution;
						Vector_1.norm();
						Vector_2.norm();
						trans_1 = Slice_Positions[x1] / ds1.SliceThicknes * thickness_1;
						trans_2 = Slice_Positions[x2] / ds2.SliceThicknes * thickness_2;

						// Berechne Flcheninhalt der Zellen (fl)
						Point3D cs = (Vector_1.Sk_Mult(Res_1)).VC_Mult((Vector_2.Sk_Mult(Res_2)));

						fl = (float) cs.Length();

						for (int mr = 0; mr < Global_Options.Matrix_Resolution; mr++) {

							Point3D p1 = p0.VC_Add(Vector_2.Sk_Mult(mr * Res_2));
							Point3D p2 = p0.VC_Add(Vector_1.Sk_Mult(mr * Res_1));

							try {
								Integral_1[mr] = getIntegral(p1, Res_2, 1, thickness_1, Vector_2, MathTools.Nullvector, Vector_1, Series_sel[x1]);
								Integral_2[mr] = getIntegral(p2, thickness_2, 1, Res_1, Vector_2, MathTools.Nullvector, Vector_1, Series_sel[x2]);
								sum_z += Integral_1[mr];
								sum_x += Integral_2[mr];
							} catch (Exception excep) {
								return val / count;
							} 

						}																// mr
 

						for (int i1 = 0; i1 < Global_Options.Matrix_Resolution; i1++) {
							for (int i2 = 0; i2 < Global_Options.Matrix_Resolution; i2++) {
								Point3D Org = p0.VC_Add(Vector_2.Sk_Mult(i2 * Res_2));

								Org = Org.VC_Add(Vector_1.Sk_Mult(i1 * Res_1));
								b[Global_Options.Matrix_Resolution * i1 + i2 + 2 * Global_Options.Matrix_Resolution] = getIntegral(Org, Res_2, 1.0f, Res_1, Vector_2, MathTools.Nullvector, Vector_1, Series_sel[x3]);

							} 
						} 
					}																	// tra / sag
 
					else if (ds1.isTransversal() && ds2.isCoronar() && Global_Options.TC_S) {

						if (Slice_Positions[x1] < 0 || Slice_Positions[x2] < 0) {
							return val / count;

							// Berechne Schnittebne E1
						} 
						factor_eq = Global_Options.Factor_eq_TC;

						// E1 = (x - o) x p
						Point3D o = Origin;
						Point3D p = ds1.SliceVector.VC_Mult(ds2.SliceVector);

						// Berechne g1  fr p0
						Point3D g1o = new Point3D(0, 0, 0);
						Point3D g1m = new Point3D(0, 0, 0);

						MathTools.calcIntersection_PP(ds1.ImagePosition, ds1.SliceVector, ds2.ImagePosition, ds2.ColVector, ds2.RowVector, g1o, g1m);

						// Berechne g2  fr p1
						Point3D g2o = new Point3D(0, 0, 0);
						Point3D g2m = new Point3D(0, 0, 0);

						MathTools.calcIntersection_PP(ds1.ImagePosition, ds1.SliceVector, ds2.ImagePosition2, ds2.ColVector, ds2.RowVector, g2o, g2m);

						// Berechne g3  fr p2
						Point3D g3o = new Point3D(0, 0, 0);
						Point3D g3m = new Point3D(0, 0, 0);

						MathTools.calcIntersection_PP(ds1.ImagePosition2, ds1.SliceVector, ds2.ImagePosition, ds2.ColVector, ds2.RowVector, g3o, g3m);

						// Berechne Punkte
						p0 = MathTools.calcIntersection_GP(g1o, g1m, Origin, p);
						Point3D ps1 = MathTools.calcIntersection_GP(g2o, g2m, Origin, p);
						Point3D ps2 = MathTools.calcIntersection_GP(g3o, g3m, Origin, p);

						Vector_1 = ps1.VC_Min(p0);
						Vector_2 = ps2.VC_Min(p0);
						thickness_1 = Vector_1.Length();
						thickness_2 = Vector_2.Length();
						Res_2 = thickness_2 / Global_Options.Matrix_Resolution;
						Res_1 = thickness_1 / Global_Options.Matrix_Resolution;
						Vector_1.norm();
						Vector_2.norm();
						trans_1 = Slice_Positions[x2] / ds2.SliceThicknes * thickness_1;
						trans_2 = Slice_Positions[x1] / ds1.SliceThicknes * thickness_2;

						// Berechne Flcheninhalt der Zellen (fl)
						Point3D cs = (Vector_1.Sk_Mult(Res_1)).VC_Mult((Vector_2.Sk_Mult(Res_2)));

						fl = (float) cs.Length();

						for (int mr = 0; mr < Global_Options.Matrix_Resolution; mr++) {

							Point3D p1 = p0.VC_Add(Vector_2.Sk_Mult(mr * Res_2));
							Point3D p2 = p0.VC_Add(Vector_1.Sk_Mult(mr * Res_1));

							try {
								Integral_1[mr] = getIntegral(p1, 1, thickness_1, Res_2, MathTools.Nullvector, Vector_1, Vector_2, Series_sel[x1]);
								Integral_2[mr] = getIntegral(p2, 1, Res_1, thickness_2, MathTools.Nullvector, Vector_1, Vector_2, Series_sel[x2]);
								sum_z += Integral_1[mr];
								sum_x += Integral_2[mr];
							} catch (Exception excep) {
								return val / count;
							} 

						}																// mr
 
						for (int i1 = 0; i1 < Global_Options.Matrix_Resolution; i1++) {
							for (int i2 = 0; i2 < Global_Options.Matrix_Resolution; i2++) {
								Point3D Org = p0.VC_Add(Vector_2.Sk_Mult(i2 * Res_2));

								Org = Org.VC_Add(Vector_1.Sk_Mult(i1 * Res_1));
								b[Global_Options.Matrix_Resolution * i1 + i2 + 2 * Global_Options.Matrix_Resolution] = getIntegral(Org, 1.0f, Res_1, Res_2, MathTools.Nullvector, Vector_1, Vector_2, Series_sel[x3]);

							} 
						} 
					}																	// tra / cor
 
					else if (ds1.isCoronar() && ds2.isSagittal() && Global_Options.SC_T) {

						if (Slice_Positions[x1] < 0 || Slice_Positions[x2] < 0) {
							return val / count;

							// Berechne Schnittebne E1
						} 
						factor_eq = Global_Options.Factor_eq_CS;

						// E1 = (x - o) x p
						Point3D o = Origin;
						Point3D p = ds1.SliceVector.VC_Mult(ds2.SliceVector);

						// Berechne g1  fr p0
						Point3D g1o = new Point3D(0, 0, 0);
						Point3D g1m = new Point3D(0, 0, 0);

						MathTools.calcIntersection_PP(ds1.ImagePosition, ds1.SliceVector, ds2.ImagePosition, ds2.ColVector, ds2.RowVector, g1o, g1m);

						// Berechne g2  fr p1
						Point3D g2o = new Point3D(0, 0, 0);
						Point3D g2m = new Point3D(0, 0, 0);

						MathTools.calcIntersection_PP(ds1.ImagePosition2, ds1.SliceVector, ds2.ImagePosition, ds2.ColVector, ds2.RowVector, g2o, g2m);

						// Berechne g3  fr p2
						Point3D g3o = new Point3D(0, 0, 0);
						Point3D g3m = new Point3D(0, 0, 0);

						MathTools.calcIntersection_PP(ds1.ImagePosition, ds1.SliceVector, ds2.ImagePosition2, ds2.ColVector, ds2.RowVector, g3o, g3m);

						// Berechne Punkte
						p0 = MathTools.calcIntersection_GP(g1o, g1m, Origin, p);
						Point3D ps1 = MathTools.calcIntersection_GP(g2o, g2m, Origin, p);
						Point3D ps2 = MathTools.calcIntersection_GP(g3o, g3m, Origin, p);

						Vector_1 = ps1.VC_Min(p0);
						Vector_2 = ps2.VC_Min(p0);
						thickness_1 = Vector_1.Length();
						thickness_2 = Vector_2.Length();
						Res_2 = thickness_2 / Global_Options.Matrix_Resolution;
						Res_1 = thickness_1 / Global_Options.Matrix_Resolution;
						Vector_1.norm();
						Vector_2.norm();
						trans_1 = Slice_Positions[x1] / ds1.SliceThicknes * thickness_1;
						trans_2 = Slice_Positions[x2] / ds2.SliceThicknes * thickness_2;

						// Berechne Flcheninhalt der Zellen (fl)
						Point3D cs = (Vector_1.Sk_Mult(Res_1)).VC_Mult((Vector_2.Sk_Mult(Res_2)));

						fl = (float) cs.Length();

						for (int mr = 0; mr < Global_Options.Matrix_Resolution; mr++) {

							Point3D p1 = p0.VC_Add(Vector_2.Sk_Mult(mr * Res_2));
							Point3D p2 = p0.VC_Add(Vector_1.Sk_Mult(mr * Res_1));

							try {
								Integral_1[mr] = getIntegral(p1, Res_2, thickness_1, 1, Vector_2, Vector_1, MathTools.Nullvector, Series_sel[x1]);
								Integral_2[mr] = getIntegral(p2, thickness_2, Res_1, 1, Vector_2, Vector_1, MathTools.Nullvector, Series_sel[x2]);
								sum_z += Integral_1[mr];
								sum_x += Integral_2[mr];
							} catch (Exception excep) {
								return val / count;
							} 

						}																// mr
 
						for (int i1 = 0; i1 < Global_Options.Matrix_Resolution; i1++) {
							for (int i2 = 0; i2 < Global_Options.Matrix_Resolution; i2++) {
								Point3D Org = p0.VC_Add(Vector_2.Sk_Mult(i2 * Res_2));

								Org = Org.VC_Add(Vector_1.Sk_Mult(i1 * Res_1));
								b[Global_Options.Matrix_Resolution * i1 + i2 + 2 * Global_Options.Matrix_Resolution] = getIntegral(Org, Res_2, Res_1, 1.0f, Vector_2, Vector_1, MathTools.Nullvector, Series_sel[x3]);

							} 
						} 
					}																	// cor / sag
 
					else															// keine Ansicht Zuordnung mglich
					 {
						continue;
					} 

					// Alles hat geklappt
					int split_2 = (int) (trans_2 * Global_Options.Matrix_Resolution / thickness_2);
					int split_1 = (int) (trans_1 * Global_Options.Matrix_Resolution / thickness_1);


					// b weiter auffllen
					for (int i = 0; i < Global_Options.Matrix_Resolution; i++) {
						b[i] = Integral_2[i];
						b[i + Global_Options.Matrix_Resolution] = Integral_1[i];
					} 

					// b und a sind berechnet !
					// Faktor der "wichtigeren" Zeilen bercksichtigen
					float				fac_imp = Global_Options.Factor_impRows;
					int					quad = Global_Options.Matrix_Resolution * Global_Options.Matrix_Resolution;
					double[][]	Matrix = new double[quad + 2 * Global_Options.Matrix_Resolution][quad];

					if (Global_Options.Matrix_Resolution == 3) {
						MathTools.copyMatrix(MathTools.Matrix3_3, Matrix);
					} else {
						MathTools.copyMatrix(MathTools.Matrix2_2, Matrix);
					} 

					for (int ip = 0; ip < 2 * Global_Options.Matrix_Resolution; ip++) {
						b[ip] *= fac_imp;
						for (int col = 0; col < quad; col++) {
							Matrix[ip][col] *= fac_imp;
						} 
					} 


					// lse Ax = b
					if (Global_Options.Matrix_Resolution == 3) {
						b = MathTools.solveLinEqu(Matrix, b, 15, 9);
					} else {
						b = MathTools.solveLinEqu(Matrix, b, 8, 4);
					} 


					// Faktor fr das Paar
					if (b == null) {
						continue;
					} 
					Matrix_values[Matrix_counter] = (float) b[split_1 * Global_Options.Matrix_Resolution + split_2];
					Matrix_values[Matrix_counter] *= factor_eq;
					Matrix_counter++;

					// return (vl1 + vl2) / (2* Res_2 * Res_1)  ;


				}																		// x1
 			 }																		// x2
 
			val = 0;
			for (int i = 0; i < Matrix_counter; i++) {
				val += Matrix_values[i];
			} 

			return val;

		// val = (val / Matrix_counter) ;

		// --------------------------------------------------------------------------


		// -----------------------------------------------------------------------------
		default:																// Wenn alles versagt
			return 0;


		}																				// switch


	} 


	/**
	 * Die Methode berechnet auf numerische Art und Weise ein dreifaches Integral mit Hilfe der Mittelpunktregel.
	 * Das zugehrige Integrationsvolumen wird durch drei Vektoren und einem Ankerpunkt aufgespannt.
	 * Die globale Variable "Integration_Number" gibt die Anzahl der Intervalle an.
	 * Als Basis dient der vorselektierte 3D-Datensatz.
	 * @param Origin Positionsvektor (Ankerpunkt)
	 * @param xval Lnge Vektor1
	 * @param yval Lnge Vektor2
	 * @param zval Lnge Vektor3
	 * @param xvec Vektor1
	 * @param yvec Vektor1
	 * @param zvec Vektor1
	 * 
	 * @return Wert des dreifachen Integrals ber das spezifizierte Volumen
	 * 
	 * 
	 */
	private float getIntegral_Midpoint_Sum(Point3D Origin, float xval, float yval, float zval, Point3D xvec, Point3D yvec, Point3D zvec) {

		// Integration nach der Mittelpunkt Regel
		// Global_Options_Integration_Number n
		float val = 0.0f;
		int		n = Global_Options.Integration_Number;

		if (n == 1) {
			return getFunktionValue(Origin.getX() + 0.5f * xval * xvec.getX(), Origin.getY() + 0.5f * yval * yvec.getY(), Origin.getZ() + 0.5f * zval * zvec.getZ()) * xval * yval * zval;
		} 

		for (int x = 0; x < n; x++) {
			for (int y = 0; y < n; y++) {
				for (int z = 0; z < n; z++) {
					{
						val += getFunktionValue(Origin.getX() + (x + 0.5f) * xval / n * xvec.getX(), Origin.getY() + (y + 0.5f) * yval / n * yvec.getY(), Origin.getZ() + (z + 0.5f) * zval / n * zvec.getZ());
					} 
				} 
			} 
		} 

		return val * xval * yval * zval / (n * n * n);
	} 


	/**
	 * Die Methode berechnet auf numerische Art und Weise ein dreifaches Integral mit Hilfe der Riemannsumme.
	 * Das zugehrige Integrationsvolumen wird durch drei Vektoren und einem Ankerpunkt aufgespannt.
	 * Die globale Variable "Integration_Number" gibt die Anzahl der Intervalle an.
	 * Als Basis dient der vorselektierte 3D-Datensatz.
	 * @param Origin Positionsvektor (Ankerpunkt)
	 * @param xval Lnge Vektor1
	 * @param yval Lnge Vektor2
	 * @param zval Lnge Vektor3
	 * @param xvec Vektor1
	 * @param yvec Vektor1
	 * @param zvec Vektor1
	 * 
	 * @return Wert des dreifachen Integrals ber das spezifizierte Volumen
	 * 
	 * 
	 */
	private float getIntegral_Riemann_Sum(Point3D Origin, float xval, float yval, float zval, Point3D xvec, Point3D yvec, Point3D zvec) {

		// Integration mit der Riemann Summe

		float val = 0.0f;
		int		n = Global_Options.Integration_Number;

		for (int x = 1; x <= n; x++) {
			for (int y = 1; y <= n; y++) {
				for (int z = 1; z <= n; z++) {
					{
						val += getFunktionValue(Origin.getX() + x * xval / n * xvec.getX(), Origin.getY() + y * yval / n * yvec.getY(), Origin.getZ() + z * zval / n * zvec.getZ());
					} 
				} 
			} 
		} 

		return val * xval * yval * zval / (n * n * n);
	} 


	/**
	 * Die Methode berechnet auf numerische Art und Weise ein dreifaches Integral mit Hilfe der Riemannsumme.
	 * Das zugehrige Integrationsvolumen wird durch drei Vektoren und einem Ankerpunkt aufgespannt.
	 * Die globale Variable "Integration_Number" gibt die Anzahl der Intervalle an.
	 * Als Basis dient die mit "Serie" spezifizierte Serie
	 * Mit dem Krzel "2d" wird angedeutet, da der 3.Vektor ein Nullvektor ist.
	 * Eine Ausnahme wird ausgelst, falls die Integrationsgrenzen ber das Serienvolumen hinazusgeht.
	 * 
	 * @param Origin Positionsvektor (Ankerpunkt)
	 * @param xval Lnge Vektor1
	 * @param yval Lnge Vektor2
	 * @param zval Lnge Vektor3 = 1 ;
	 * @param xvec Vektor1
	 * @param yvec Vektor1
	 * @param zvec Vektor3 = NULL
	 * @param serie Index der Serie
	 * 
	 * @return Das Dreifachintegral ber das spezifizierte Volumen
	 * 
	 * @throws IllegalArgumentException
	 */
	private float getIntegral_Riemann_Sum_2D(Point3D Origin, float xval, float yval, float zval, Point3D xvec, Point3D yvec, Point3D zvec, int Serie) throws IllegalArgumentException {

		// Integration mit der Riemann Sunne

		float val = 0.0f;
		int		n = Global_Options.Integration_Number;

		try {
			for (int x = 1; x <= n; x++) {
				for (int y = 1; y <= n; y++) {
					for (int z = 1; z <= n; z++) {
						{
							val += getFunktionValue(Origin.getX() + x * xval / n * xvec.getX(), Origin.getY() + y * yval / n * yvec.getY(), Origin.getZ() + z * zval / n * zvec.getZ(), Serie);
						} 
					} 
				} 
			} 

			return val * xval * yval * zval / (n * n * n);

		} catch (Exception e) {
			throw new IllegalArgumentException();
		} 
	} 




	/**
	 * Die Methode berechnet auf numerische Art und Weise ein dreifaches Integral mit Hilfe der Trapezrefel.
	 * Das zugehrige Integrationsvolumen wird durch drei Vektoren und einem Ankerpunkt aufgespannt.
	 * Die globale Variable "Integration_Number" gibt die Anzahl der Intervalle an.
	 * Als Basis dient die mit "Serie" spezifizierte Serie
	 * Mit dem Krzel "2d" wird angedeutet, da der 3.Vektor ein Nullvektor ist.
	 * Eine Ausnahme wird ausgelst, falls die Integrationsgrenzen ber das Serienvolumen hinazusgeht.
	 * 
	 * @param Origin Positionsvektor (Ankerpunkt)
	 * @param xval Lnge Vektor1
	 * @param yval Lnge Vektor2
	 * @param zval Lnge Vektor3 = 1 ;
	 * @param xvec Vektor1
	 * @param yvec Vektor1
	 * @param zvec Vektor3 = NULL
	 * @param serie Index der Serie
	 * 
	 * @return Das Dreifachintegral ber das spezifizierte Volumen
	 * 
	 * @throws IllegalArgumentException
	 */
	private float getIntegral_Trapezoidal_Sum_2D(Point3D Origin, float xval, float yval, float zval, Point3D xvec, Point3D yvec, Point3D zvec, int Serie) throws IllegalArgumentException {

		// Integration mit der Trapez Regel

		// Integration mit der Trapez Regel in drei Dimensionen
		// varibale nach n
		int		n = Global_Options.Integration_Number;
		float xfac = xval / n;
		float yfac = yval / n;
		float zfac = zval / n;
		float factor;
		float val = 0;

		for (int x = 0; x <= n; x++) {
			for (int y = 0; y <= n; y++) {
				for (int z = 0; z <= n; z++) {
					{
						factor = 1.0f;
						if (x == 0 || x == n) {
							factor *= 0.5f;
						} 
						if (y == 0 || y == n) {
							factor *= 0.5f;
						} 
						if (z == 0 || z == n) {
							factor *= 0.5f;

						} 
						val += factor * getFunktionValue(Origin.getX() + x * xfac * xvec.getX(), Origin.getY() + y * yfac * yvec.getY(), Origin.getZ() + z * zfac * zvec.getZ(), Serie);
					} 
				} 
			} 
		} 

		return (val * xfac * yfac * zfac);
	} 




	/**
	 * Die Methode berechnet auf numerische Art und Weise ein dreifaches Integral mit Hilfe der Simpson Regel.
	 * Das zugehrige Integrationsvolumen wird durch drei Vektoren und einem Ankerpunkt aufgespannt.
	 * Die globale Variable "Integration_Number" gibt die Anzahl der Intervalle an.
	 * Als Basis dient der gesamte 3D-Datensatz.
	 * @param Origin Positionsvektor (Ankerpunkt)
	 * @param xval Lnge Vektor1
	 * @param yval Lnge Vektor2
	 * @param zval Lnge Vektor3
	 * @param xvec Vektor1
	 * @param yvec Vektor1
	 * @param zvec Vektor1
	 * 
	 * @return Wert des dreifachen Integrals ber das spezifizierte Volumen
	 * 
	 * 
	 */
	private float getIntegral_Simpson_Sum(Point3D Origin, float xval, float yval, float zval, Point3D xvec, Point3D yvec, Point3D zvec) {

		// Integration mit der Simpson Regel in drei Dimensionen
		// varibale nach n
		int		n = Global_Options.Integration_Number;
		int		two_n = n << 1;
		float xfac = xval / two_n;
		float yfac = yval / two_n;
		float zfac = zval / two_n;
		float factor;
		float val = 0;

		for (int x = 0; x <= two_n; x++) {
			for (int y = 0; y <= two_n; y++) {
				for (int z = 0; z <= two_n; z++) {
					{

						factor = MathTools.getWeigth(x, n) * MathTools.getWeigth(y, n) * MathTools.getWeigth(z, n);

						val += factor * getFunktionValue(Origin.getX() + x * xfac * xvec.getX(), Origin.getY() + y * yfac * yvec.getY(), Origin.getZ() + z * zfac * zvec.getZ());
					} 
				} 
			} 
		} 

		return (val * xfac * yfac * zfac) / 27;

		// return (val*xval*yval*zval) / (27*8*n*n*n) ;


	} 


	/**
	 * Die Methode berechnet auf numerische Art und Weise ein dreifaches Integral mit Hilfe der Simpson Regel.
	 * Das zugehrige Integrationsvolumen wird durch drei Vektoren und einem Ankerpunkt aufgespannt.
	 * Die globale Variable "Integration_Number" gibt die Anzahl der Intervalle an.
	 * Als Basis dient die mit "Serie" spezifizierte Serie
	 * Mit dem Krzel "2d" wird angedeutet, da der 3.Vektor ein Nullvektor ist.
	 * Eine Ausnahme wird ausgelst, falls die Integrationsgrenzen ber das Serienvolumen hinazusgeht.
	 * 
	 * @param Origin Positionsvektor (Ankerpunkt)
	 * @param xval Lnge Vektor1
	 * @param yval Lnge Vektor2
	 * @param zval Lnge Vektor3 = 1 ;
	 * @param xvec Vektor1
	 * @param yvec Vektor1
	 * @param zvec Vektor3 = NULL
	 * @param serie Index der Serie
	 * 
	 * @return Das Dreifachintegral ber das spezifizierte Volumen
	 * 
	 * @throws IllegalArgumentException
	 */
	private float getIntegral_Simpson_Sum_2D(Point3D Origin, float xval, float yval, float zval, Point3D xvec, Point3D yvec, Point3D zvec, int Serie) throws IllegalArgumentException {

		// Integration mit der Simpson Regel in drei Dimensionen
		// varibale nach n
		int		n = Global_Options.Integration_Number;
		int		two_n = n << 1;
		float xfac = xval / two_n;
		float yfac = yval / two_n;
		float zfac = zval / two_n;
		float factor;
		float val = 0;

		try {
			for (int x = 0; x <= two_n; x++) {
				for (int y = 0; y <= two_n; y++) {
					for (int z = 0; z <= two_n; z++) {
						{

							factor = MathTools.getWeigth(x, n) * MathTools.getWeigth(y, n) * MathTools.getWeigth(z, n);

							val += factor * getFunktionValue(Origin.getX() + x * xfac * xvec.getX(), Origin.getY() + y * yfac * yvec.getY(), Origin.getZ() + z * zfac * zvec.getZ(), Serie);
						} 
					} 
				} 
			} 

			return (val * xfac * yfac * zfac) / 27;

		} catch (Exception e) {
			throw new IllegalArgumentException();
		} 

		// return (val*xval*yval*zval) / (27*8*n*n*n) ;


	} 


	/**
	 * Die Methode berechnet aus dem 3D-Datensatz einen neuen Schichtstapel.
	 * Das Schichtvolumen wird durch drei Vektoren aufgespannt, die in einem Ankerpunkt beginnen.
	 * Die Rasterung der korrespondierenden Bildmatrix ist durch die Anzahl der Reihen und Spalten gegeben.
	 * Mit dem Schichtabstand und der Anzahl der gewollten Schichten ist die Definition einer neuen Serie komplett.
	 * Der Berechnungsproze kann in einem Fortschrittsbalken mit eigenem Fenster dokumentiert werden.
	 * @param Origin Positionsvektor der ersten Schicht (Ankerpunkt)
	 * @param xvec Vektor1
	 * @param yvec Vektor2
	 * @param zvec Vektor3
	 * @param xval Lnge Vektor1
	 * @param yval Lnge Vektor2
	 * @param zval Lnge Vektor3
	 * @param rows Anzahl der Reihen in der Bildmatrix
	 * @param cols Anzahl der Spalten in der Bildmatrix
	 * @param mon Fortschrittsanzeige
	 * @param NumberSlices Anzahl der Schichten
	 * @param SliceDistant Schichtabstand
	 * 
	 * @return Ein Array der Bildmatrizen
	 * 
	 * 
	 */
	public short[][] getSlices(Point3D Origin, Point3D xvec, Point3D yvec, Point3D zvec, float xval, float yval, float zval, int rows, int cols, ProgressMonitor mon, int NumberSlices, float SliceDistant) {

		if (mon != null) {
			mon.setProgress(0);
		} 

		if (Global_Options.ApproxMethod == 2 && NumberOfSelectesSeries < 2) {
			IJ.error("Fr die Methode Gewichtete Summen werden mindestens zwei selektierte Serien gebraucht");
			return null;
		} 
		if (Global_Options.ApproxMethod == 3 && NumberOfSelectesSeries < 3) {
			IJ.error("Fr die Methode Lineare Gleichungen werden mindestens drei selektierte Serien gebraucht");
			return null;
		} 
		short		Slices[][] = new short[NumberSlices][cols * rows];

		Point3D Slice_Increment = zvec.Sk_Mult(SliceDistant + zval);

		for (int sl = 0; sl < NumberSlices; sl++) {

			short[] Pixels = new short[cols * rows];
			Point3D Col_Increment, Row_Increment;

			Col_Increment = yvec.Sk_Mult(yval);
			Row_Increment = xvec.Sk_Mult(xval);

			ColCount.set(Origin);
			for (int y = 0; y < rows; y++) {

				RowCount.set(ColCount);
				for (int x = 0; x < cols; x++) {
					float val = 0;

					val = getIntegral(RowCount, xval, yval, zval, xvec, yvec, zvec);

					// val = getIntegral_Simpson_Sum (RowCount,xval,yval,zval,xvec,yvec,zvec) ;

					if (val < 0) {
						val = 0;
					} 
					Pixels[x + y * cols] = (short) val;
					RowCount.VC_Add2(Row_Increment);
				}		// for x
				 ColCount.VC_Add2(Col_Increment);
				if (mon != null) {
					mon.setProgress(y + sl * rows);
				} 
			}			// for y
			 Slices[sl] = Pixels;
			if (mon != null) {
				if (mon.isCanceled()) {
					return null;
				} 
			} 
			Origin.VC_Add2(Slice_Increment);
		}				// for sl
 
		if (mon != null) {
			mon.close();
		} 
		return Slices;

	} 


	/**
	 * Die Methode setzt den Umgebungsquader und Schnittquader zurck auf die Standarwerte.
	 * 
	 * 
	 * 
	 */
	public void resetBB() {

		// zurcksetzen der Bounding Box

		MinZ = Float.POSITIVE_INFINITY;
		MinX = Float.POSITIVE_INFINITY;
		MinY = Float.POSITIVE_INFINITY;
		MaxX = Float.NEGATIVE_INFINITY;
		MaxY = Float.NEGATIVE_INFINITY;
		MaxZ = Float.NEGATIVE_INFINITY;

		// zurcksetzen der Intersection Box
		MinZ2 = Float.NEGATIVE_INFINITY;
		MinX2 = Float.NEGATIVE_INFINITY;
		MinY2 = Float.NEGATIVE_INFINITY;
		MaxX2 = Float.POSITIVE_INFINITY;
		MaxY2 = Float.POSITIVE_INFINITY;
		MaxZ2 = Float.POSITIVE_INFINITY;
	} 


	/**
	 * Die Methode aktualisiert die markierten Serien mit Hilfe eines Arrays, da die Indizes
	 * der markierten Serien beinhaltet.
	 * 
	 * @param Series Array mit markierten Serienindizes
	 * 
	 * 
	 */
	public void setSelectedSeries(int[] Series) {
		NumberOfSelectesSeries = Series.length;
		for (int i = 0; i < NumberOfSelectesSeries; i++) {
			SelectedSlices[i] = Series[i];
		} 
	} 


	/**
	 * Die Methode aktualisiert die markierten Serien, falls eine nderung durch den Benutzer vorgenommen wurde.
	 * 
	 * 
	 * 
	 */
	public void setSelectedSeries() {
		NumberOfSelectesSeries = 0;
		TreePath[]	paths = getSelectionPaths();

		if (paths == null || paths.length == 0) {
			return;

		} 
		for (int i = 0; i < paths.length; i++) {
			DefaultMutableTreeNode	node = (DefaultMutableTreeNode) paths[i].getLastPathComponent();
			Object									nodeinfo = node.getUserObject();
			String									classname = nodeinfo.getClass().getName();

			if (classname.compareTo("java.lang.String") == 0)		// alles markiert
			 {
				NumberOfSelectesSeries = root.getChildCount();
				for (int s = 0; s < NumberOfSelectesSeries; s++) {
					SelectedSlices[s] = s;
				} 
				return;
			} 


			if (classname.indexOf("Series_Node") > 0) {

				// Serie markiert
				NumberOfSelectesSeries++;
				SelectedSlices[i] = root.getIndex(node);
			} 

		}																											// for i
 
	}																												// set
 

	/**
	 * Die Methode entfernt smtliche Markierungen aus dem Schichtbaum.
	 * 
	 * 
	 * 
	 */
	public void clearSelection2() {

		NumberOfSelectesSeries = 0;

		TreeSelectionModel	tm = this.getSelectionModel();
		tm.clearSelection();
		this.model.reload();
		this.update();

	} 


	/**
	 * Die Methode berechnet auf numerische Art und Weise ein dreifaches Integral mit Hilfe der Mittelpunktregel.
	 * Das zugehrige Integrationsvolumen wird durch drei Vektoren und einem Ankerpunkt aufgespannt.
	 * Die globale Variable "Integration_Number" gibt die Anzahl der Intervalle an.
	 * Als Basis dient die mit "Serie" spezifizierte Serie
	 * Das Krzel "2d" rhrt daher, weil der 3.Vektor ein Nullvektor ist.
	 * Eine Ausnahme wird ausgelst, falls die Integrationsgrenzen ber das Serienvolumen hinazusgeht.
	 * 
	 * @param Origin Positionsvektor (Ankerpunkt)
	 * @param xval Lnge Vektor1
	 * @param yval Lnge Vektor2
	 * @param zval Lnge Vektor3 = 1 ;
	 * @param xvec Vektor1
	 * @param yvec Vektor1
	 * @param zvec Vektor3 = NULL
	 * @param serie Index der Serie
	 * 
	 * @return Das Dreifachintegral ber das spezifizierte Volumen
	 * 
	 * @throws IllegalArgumentException
	 * 
	 * 
	 */
	private float getIntegral_Midpoint_Sum_2D(Point3D Origin, float xval, float yval, float zval, Point3D xvec, Point3D yvec, Point3D zvec, int serie) throws IllegalArgumentException {

		// Integration nach der Mittelpunkt Regel
		// Global_Options_Integration_Number n
		float val = 0.0f;
		int		n = Global_Options.Integration_Number;

		try {
			if (n == 1) {
				return getFunktionValue(Origin.getX() + 0.5f * xval * xvec.getX(), Origin.getY() + 0.5f * yval * yvec.getY(), Origin.getZ() + 0.5f * zval * zvec.getZ(), serie) * xval * yval * zval;
			} 

			for (int x = 0; x < n; x++) {
				for (int y = 0; y < n; y++) {
					for (int z = 0; z < n; z++) {
						{
							val += getFunktionValue(Origin.getX() + (x + 0.5f) * xval / n * xvec.getX(), Origin.getY() + (y + 0.5f) * yval / n * yvec.getY(), Origin.getZ() + (z + 0.5f) * zval / n * zvec.getZ(), serie);
						} 
					} 
				} 
			} 

			return val * xval * yval * zval / (n * n * n);



		} catch (Exception e) {
			throw new IllegalArgumentException();
		} 

	} 


	/**
	 * Die Methode berechnet auf numerische Art und Weise ein dreifaches Integral.
	 * Das zugehrige Integrationsvolumen wird durch drei Vektoren und einem Ankerpunkt aufgespannt.
	 * Die globale Variable "Integration_Number" gibt die Anzahl der Intervalle an.
	 * Als Basis dient der gesamte 3D-Datensatz.
	 * Die Integrationsmethode wird global durch "IntegrationMethod" gesteuert.
	 * @param Origin Positionsvektor (Ankerpunkt)
	 * @param xval Lnge Vektor1
	 * @param yval Lnge Vektor2
	 * @param zval Lnge Vektor3
	 * @param xvec Vektor1
	 * @param yvec Vektor1
	 * @param zvec Vektor1
	 * 
	 * @return Wert des dreifachen Integrals ber das spezifizierte Volumen
	 * 
	 * 
	 */
	public float getIntegral(Point3D Origin, float xval, float yval, float zval, Point3D xvec, Point3D yvec, Point3D zvec) {
		switch (Global_Options.IntegrationMethod) {
		case 0:
			return getIntegral_Midpoint_Sum(Origin, xval, yval, zval, xvec, yvec, zvec);
		case 1:
			return getIntegral_Riemann_Sum(Origin, xval, yval, zval, xvec, yvec, zvec);
		case 2:
			return getIntegral_Simpson_Sum(Origin, xval, yval, zval, xvec, yvec, zvec);
		case 3:
			return getIntegral_Trapezoidal_Sum(Origin, xval, yval, zval, xvec, yvec, zvec);


		default:
			return 0;
		}
	} 


	/**
	 * * Die Methode berechnet auf numerische Art und Weise ein dreifaches Integral.
	 * Das zugehrige Integrationsvolumen wird durch drei Vektoren und einem Ankerpunkt aufgespannt.
	 * Die globale Variable "Integration_Number" gibt die Anzahl der Intervalle an.
	 * Als Basis dient die spezifizierte Serie.
	 * Die Integrationsmethode wird global durch "IntegrationMethod" gesteuert.
	 * Mit dem Krzel "2d" wird angedeutet, da der 3.Vektor ein Nullvektor ist.
	 * @param Origin Positionsvektor (Ankerpunkt)
	 * @param xval Lnge Vektor1
	 * @param yval Lnge Vektor2
	 * @param zval Lnge Vektor3
	 * @param xvec Vektor1
	 * @param yvec Vektor1
	 * @param zvec Vektor1
	 * 
	 * @return Wert des dreifachen Integrals ber das spezifizierte Volumen
	 * @param Serie Index der Serie
	 * 
	 * 
	 * 
	 * 
	 */
	public float getIntegral(Point3D Origin, float xval, float yval, float zval, Point3D xvec, Point3D yvec, Point3D zvec, int Serie) {
		switch (Global_Options.IntegrationMethod) {
		case 0:
			return getIntegral_Midpoint_Sum_2D(Origin, xval, yval, zval, xvec, yvec, zvec, Serie);
		case 1:
			return getIntegral_Riemann_Sum_2D(Origin, xval, yval, zval, xvec, yvec, zvec, Serie);
		case 2:
			return getIntegral_Simpson_Sum_2D(Origin, xval, yval, zval, xvec, yvec, zvec, Serie);
		case 3:
			return getIntegral_Trapezoidal_Sum_2D(Origin, xval, yval, zval, xvec, yvec, zvec, Serie);

		default:
			return 0;
		}
	} 


	/**
	 * Die Methode berechnet aus dem 3D-Datensatz einen neuen Schichtstapel.
	 * Das Schichtvolumen wird durch drei Vektoren aufgespannt, die in einem Ankerpunkt beginnen.
	 * Die Rasterung der korrespondierenden Bildmatrix ist durch die Anzahl der Reihen und Spalten gegeben.
	 * Mit dem Schichtabstand und der Anzahl der gewollten Schichten ist die Definition einer neuen Serie komplett.
	 * Der Berechnungsproze kann in einem Fortschrittsbalken dokumentiert werden.
	 * Der Unterschied zu "getSlices" liegt in dem Fortschrittsbalken.
	 * @param Origin Positionsvektor der ersten Schicht (Ankerpunkt)
	 * @param xvec Vektor1
	 * @param yvec Vektor2
	 * @param zvec Vektor3
	 * @param xval Lnge Vektor1
	 * @param yval Lnge Vektor2
	 * @param zval Lnge Vektor3
	 * @param rows Anzahl der Reihen in der Bildmatrix
	 * @param cols Anzahl der Spalten in der Bildmatrix
	 * @param mon Fortschrittsanzeige
	 * @param NumberSlices Anzahl der Schichten
	 * @param SliceDistant Schichtabstand
	 * 
	 * @return Ein Array der Bildmatrizen
	 * 
	 * 
	 */
	public short[][] getSlices_2(Point3D Origin, Point3D xvec, Point3D yvec, Point3D zvec, float xval, float yval, float zval, int rows, int cols, JProgressBar mon, int NumberSlices, float SliceDistant) {

		if (mon != null) {
			mon.setValue(0);
		} 
		if (Global_Options.ApproxMethod == 2 && NumberOfSelectesSeries < 2) {
			IJ.error("Fr die Methode Gewichtete Summen werden mindestens zwei selektierte Serien gebraucht");
			return null;
		} 
		if (Global_Options.ApproxMethod == 3 && NumberOfSelectesSeries < 3) {
			IJ.error("Fr die Methode Lineare Gleichungen werden mindestens drei selektierte Serien gebraucht");
			return null;
		} 

		short		Slices[][] = new short[NumberSlices][cols * rows];

		Point3D Slice_Increment = zvec.Sk_Mult(SliceDistant + zval);

		for (int sl = 0; sl < NumberSlices; sl++) {

			short[] Pixels = new short[cols * rows];
			Point3D Col_Increment, Row_Increment;

			Col_Increment = yvec.Sk_Mult(yval);
			Row_Increment = xvec.Sk_Mult(xval);

			ColCount.set(Origin);
			for (int y = 0; y < rows; y++) {

				RowCount.set(ColCount);
				for (int x = 0; x < cols; x++) {
					float val = 0;

					val = getIntegral(RowCount, xval, yval, zval, xvec, yvec, zvec);

					// val = getIntegral_Simpson_Sum (RowCount,xval,yval,zval,xvec,yvec,zvec) ;

					if (val < 0) {
						val = 0;
					} 
					Pixels[x + y * cols] = (short) val;
					RowCount.VC_Add2(Row_Increment);
				}		// for x
				 ColCount.VC_Add2(Col_Increment);
				if (mon != null) {
					mon.setValue(y + y * sl);
				} 
			}			// for y
			 Slices[sl] = Pixels;

			// if (mon != null) if (mon.isCanceled()) return null ;
			Origin.VC_Add2(Slice_Increment);
		}				// for sl
 
		// if (mon != null) mon.close();
		return Slices;

	} 


	/**
	 * Die Methode berechnet auf numerische Art und Weise ein dreifaches Integral mit Hilfe der Trapezregel.
	 * Das zugehrige Integrationsvolumen wird durch drei Vektoren und einem Ankerpunkt aufgespannt.
	 * Die globale Variable "Integration_Number" gibt die Anzahl der Intervalle an.
	 * Als Basis dient der vorselektierte 3D-Datensatz.
	 * @param Origin Positionsvektor (Ankerpunkt)
	 * @param xval Lnge Vektor1
	 * @param yval Lnge Vektor2
	 * @param zval Lnge Vektor3
	 * @param xvec Vektor1
	 * @param yvec Vektor1
	 * @param zvec Vektor1
	 * 
	 * @return Wert des dreifachen Integrals ber das spezifizierte Volumen
	 * 
	 * 
	 */

	private float getIntegral_Trapezoidal_Sum(Point3D Origin, float xval, float yval, float zval, Point3D xvec, Point3D yvec, Point3D zvec) {

		// Integration mit der Trapez Regel in drei Dimensionen
		// varibale nach n
		int		n = Global_Options.Integration_Number;
		float xfac = xval / n;
		float yfac = yval / n;
		float zfac = zval / n;
		float factor;
		float val = 0;

		for (int x = 0; x <= n; x++) {
			for (int y = 0; y <= n; y++) {
				for (int z = 0; z <= n; z++) {
					{
						factor = 1.0f;
						if (x == 0 || x == n) {
							factor *= 0.5f;
						} 
						if (y == 0 || y == n) {
							factor *= 0.5f;
						} 
						if (z == 0 || z == n) {
							factor *= 0.5f;

						} 
						val += factor * getFunktionValue(Origin.getX() + x * xfac * xvec.getX(), Origin.getY() + y * yfac * yvec.getY(), Origin.getZ() + z * zfac * zvec.getZ());
					} 
				} 
			} 
		} 

		return (val * xfac * yfac * zfac);

	} 


}




/*--- formatting done in "My Own Convention" style on 08-28-2000 ---*/

