/*--- formatted by Jindent 2.1, (www.c-lab.de/~jindent) ---*/

package mrcp.graphics;
import java.awt.*;
import ij.*;
import mrcp.tools.*;


/**
 * Die Klasse "Cube" beschreibt eine Festkrpermodellierung der Schichtgeometrie.
 * Dazu wird ein Drahtmodell verwaltet, das durch seine acht Eckpunkte definiert ist.
 * Als Funktionalitt wird die Berechnung von Projektionen und verdeckten Kanten angeboten.
 * Das Bezugssystem ist das MRT-Koordinatensystem.
 * 
 * @author Thomas Demuth
 * @version 2000.08.11
 */

public class Cube extends Vektor3DPoint {


	/**
	 * Name der zugehrigen Schicht
	 */
	String						filename = null;


	/**
	 * Nicht sichbarer Punkt des Drahtmodells
	 */
	int								HiddenPoint;


	/**
	 * Hilfsarray zum Auffinden der verdeckten kanten
	 */
	private boolean[] HiddenPoints = new boolean[9];


	/**
	 * Projektionsvektoren der Schicht
	 */
	Point[]						Points = new Point[8];	// 2d Punkte


	/**
	 * Tiefeninformation der Projektionspunkte
	 */
	float							depth;									// Tiefe der 2d Punkte


	/**
	 * Farbe des Drahtmodells
	 */
	Color							color;									// Farbe


	/**
	 * Projektionszentrum
	 */
	Point3D						BP;


	/**
	 * Mittelpunkt des Drahtmodells
	 */
	Point3D						Mid;										// Beobachtungspunkt, Mittelpunkt


	/**
	 * Projektionstyp (Schiefe Projektion, Auf-, Grund, Seitenri)
	 */
	byte							Projection_Type = 0;		// Kabinett voreingestellt


	/**
	 * Korrekturfaktor zur Orthogonalprojektion
	 */
	private float			factorY = 1;



	// Konstruktor


	/**
	 * Erzeugt ein neues Drahtmodell mit den angegebenen Punkten und Projektionszentrum.
	 * Die Projektionspunkte werden mit null initialisiert.
	 * 
	 * @param p1 Punkt1
	 * @param p2 Punkt2
	 * @param p3 Punkt3
	 * @param p4 Punkt4
	 * @param p5 Punkt5
	 * @param p6 Punkt6
	 * @param p7 Punkt7
	 * @param p8 Punkt8
	 * @param BP Projektionszentrum
	 * 
	 */
	public Cube(Point3D p1, Point3D p2, Point3D p3, Point3D p4, Point3D p5, Point3D p6, Point3D p7, Point3D p8, Point3D BP) {
		super(p1, p2, p3, p4, p5, p6, p7, p8);

		// Points initialisieren
		for (int i = 0; i <= 7; i++) {
			Points[i] = new Point(0, 0);
		} 

		// Projektionsrichtung
		// PR = new Point3D (-1 / Math.tan (Global_Options.Beta * Global_Options.RAD ) * Math.cos (Global_Options.Alpha * Global_Options.RAD ),
		// 1 / Math.tan (Global_Options.Beta * Global_Options.RAD ) * Math.sin (Global_Options.Alpha * Global_Options.RAD ),
		// -1) ;

		// Beobachtungspunkt
		this.BP = BP;
	}


	/**
	 * Erzeugt ein leeres Drahtmodell mit angegebenen Projektionszentrum.
	 * All Punkte werden auf (0,0,0) gesetzt.
	 * 
	 * 
	 * @param BP Projektionszentrum
	 * 
	 */
	public Cube(Point3D BP) {

		super(new Point3D(0, 0, 0), new Point3D(0, 0, 0), new Point3D(0, 0, 0), new Point3D(0, 0, 0), new Point3D(0, 0, 0), new Point3D(0, 0, 0), new Point3D(0, 0, 0), new Point3D(0, 0, 0));

		// Points initialisieren
		for (int i = 0; i <= 7; i++) {
			Points[i] = new Point(0, 0);
		} 

		// Projektionsrichtung
		// PR = new Point3D (-1 / Math.tan (Global_Options.Beta * Global_Options.RAD ) * Math.cos (Global_Options.Alpha * Global_Options.RAD ),
		// 1 / Math.tan (Global_Options.Beta * Global_Options.RAD ) * Math.sin (Global_Options.Alpha * Global_Options.RAD ),
		// -1) ;

		// Beobachtungspunkt
		this.BP = BP;
	}

	// Farbe setzen


	/**
	 * Die Methode setzt die Farbe das Drahtmodells.
	 * 
	 * 
	 * @param color1 Farbe des Drahtmodells
	 * 
	 * 
	 */
	public void setColor(Color color1) {
		this.color = color1;
	} 


	/**
	 * Die Methode bestimmt die Projektionsart.
	 * 
	 * 
	 * @param type Projektionstyp
	 * 
	 * 
	 */
	public void setProjection(byte type) {
		this.Projection_Type = type;
	} 


	/**
	 * Die Methode berechnet den Mittelpunkt des Drahtmodells.
	 * Der Mittelpunkt wird verwendet, um die Oberflchennormalen zu berechnen.
	 * 
	 * @return Mittelpunkt als Positionssvektor
	 * 
	 * 
	 */
	public Point3D getMid() {

		// Mittelpunkt berechenen
		Point3D ff = add(getPoint(1), getPoint(2), getPoint(3), getPoint(4));
		Point3D fb = add(getPoint(5), getPoint(6), getPoint(7), getPoint(8));

		Mid = ff.VC_Add(fb).Sk_Mult(0.125f);
		return Mid;
	} 


	/**
	 * Die Methode setzt einen neuen Projektionspunkt.
	 * 
	 * 
	 * @param number Nummer des zugehrigen Eckpunktes
	 * @param x x-Komponente
	 * @param y x-Komponente
	 * @param YConstant optionale Translation in y-Richtung
	 * @param XConstant optionale Translation in x-Richtung
	 * 
	 * 
	 */
	private void set2DPoint(int number, float x, float y, float YConstant, float XConstant) {

		// umgekehrtes Koordinatensystem bercksichtigen !
		Points[number - 1].x = (int) (Math.round(x) + XConstant);
		Points[number - 1].y = (int) (Math.round(y) + YConstant);
	} 


	/**
	 * Die Methode liest den Projektionspunkt mit Nummer "number" aus.
	 * 
	 * 
	 * @param number Nummer des zugehrigen Eckpunktes
	 * 
	 * @return Der spezifizierte Projektionspunk
	 * 
	 * 
	 */
	protected Point get2dPoint(int number) {
		return Points[number - 1];
	} 


	/**
	 * Die Methode liest den Projektionspunkt mit Nummer "number" aus.
	 * 
	 * 
	 * @param number Nummer des zugehrigen Eckpunktes
	 * 
	 * @return x-Komponente des spezifizierte Projektionspunktes
	 */
	protected int get2dPointX(int number) {
		return Points[number - 1].x;
	} 


	/**
	 * Die Methode liest den Projektionspunkt mit Nummer "number" aus.
	 * 
	 * 
	 * @param number Nummer des zugehrigen Eckpunktes
	 * 
	 * @return y-Komponente des spezifizierte Projektionspunkt
	 */
	protected int get2dPointY(int number) {
		return Points[number - 1].y;
	} 

	// berechnet die Punkte des Quader zum Zeichnen in einer Pseudo 3d Ansicht


	/**
	 * Die Methode berechnet die Projektionspunkte.
	 * Die Projektionsart ist in "Projection-type" festgelegt.
	 * Das Projektionszentrum liegt in "BP".
	 * Mit Hilfe eines Translationsvektors und einem Vergrerungsfaktor kann Einflu auf
	 * die Berechnung der Projektionspunkte genommen werden.
	 * 
	 * @param YConstant optionale Translation in y-Richtung
	 * @param XConstant optionale Translation in x-Richtung
	 * @param scale Vergrerungsfaktor
	 * 
	 * 
	 */
	protected void calculate2DPoints(float YConstant, float XConstant, float scale) {


		if (Projection_Type == Global_Options.P_Kabinett) {

			// Kabinettprojektion
			for (int i = 1; i <= 8; i++) {
				float alpha = Global_Options.Alpha * Global_Options.RAD;
				float beta = Global_Options.Beta * Global_Options.RAD;

				depth = (getPoint(i)).getZ() * scale;
				set2DPoint(i, (float) (((getPoint(i)).getX() * scale) - (depth * (1 / Math.tan(beta)) * Math.cos(alpha))), (float) (((getPoint(i)).getY() * scale) - (depth * (1 / Math.tan(beta)) * Math.sin(alpha))), YConstant, XConstant);	// => Pi
			} 
			return;
		} 

		if (Projection_Type == Global_Options.P_Parallel_Tra) {
			HiddenPoint = -1;
			for (int i = 1; i <= 8; i++) {
				set2DPoint(i, ((getPoint(i)).getX() * scale), ((getPoint(i)).getY() * scale), YConstant, XConstant);
			} 
			return;
		} 

		// Achtung gendert !

		if (Projection_Type == Global_Options.P_Parallel_Sag) {
			HiddenPoint = -1;
			for (int i = 1; i <= 8; i++) {
				set2DPoint(i, ((getPoint(i)).getZ() * scale), ((getPoint(i)).getY() * scale), YConstant, XConstant);
			} 
			return;
		} 

		if (Projection_Type == Global_Options.P_Parallel_Cor) {
			HiddenPoint = -1;
			for (int i = 1; i <= 8; i++) {
				set2DPoint(i, ((getPoint(i)).getX() * scale), ((getPoint(i)).getZ() * scale), YConstant, XConstant);
			} 
			return;
		} else if (Projection_Type == Global_Options.P_Parallel_Tra2) {
			HiddenPoint = -1;
			for (int i = 1; i <= 8; i++) {
				set2DPoint(i, ((getPoint(i)).getX() * scale), ((getPoint(i)).getY() * factorY), YConstant, XConstant);
			} 
			return;
		} else if (Projection_Type == Global_Options.P_Parallel_Sag2) {
			HiddenPoint = -1;
			for (int i = 1; i <= 8; i++) {
				set2DPoint(i, ((getPoint(i)).getY() * scale), ((getPoint(i)).getZ() * factorY), YConstant, XConstant);
			} 
			return;
		} 

		if (Projection_Type == Global_Options.P_Parallel_Cor2) {
			HiddenPoint = -1;
			for (int i = 1; i <= 8; i++) {
				set2DPoint(i, ((getPoint(i)).getX() * scale), ((getPoint(i)).getZ() * factorY), YConstant, XConstant);
			} 
			return;
		} 
	} 


	/**
	 * Die Methode berechechnet den verdeckten Punkt bei schiefer Projektion.
	 * Als Algorithmus kommt hier eine Rckseiteneliminierung zur Anwendung.
	 * 
	 * @return Nummer des verdeckten Punktes
	 * 
	 * 
	 */
	public int calculateHidden() {
		float beta = Global_Options.Beta * Global_Options.RAD;
		float alpha = Global_Options.Alpha * Global_Options.RAD;

		// Mittelpunkt berechenen

		Mid = getMid();


		// Berechne normalenvektoren als Differenz zu Mittelpunkt
		// F1 = {1,2,3,4}

		Point3D F1 = add(sub(getPoint(1), Mid), sub(getPoint(2), Mid), sub(getPoint(3), Mid), sub(getPoint(4), Mid));

		F1.norm();

		// F2 = 5,6,7,8
		Point3D F2 = add(sub(getPoint(5), Mid), sub(getPoint(6), Mid), sub(getPoint(7), Mid), sub(getPoint(8), Mid));

		F2.norm();

		// F3 = 2,4,6,8
		Point3D F3 = add(sub(getPoint(2), Mid), sub(getPoint(4), Mid), sub(getPoint(6), Mid), sub(getPoint(8), Mid));

		F3.norm();

		// F4
		Point3D F4 = add(sub(getPoint(1), Mid), sub(getPoint(3), Mid), sub(getPoint(5), Mid), sub(getPoint(7), Mid));

		F4.norm();

		// F5 = {1,2,5,6}

		Point3D F5 = add(sub(getPoint(1), Mid), sub(getPoint(2), Mid), sub(getPoint(5), Mid), sub(getPoint(6), Mid));

		F5.norm();

		// F6 = 3,4,7,8
		Point3D F6 = add(sub(getPoint(3), Mid), sub(getPoint(4), Mid), sub(getPoint(7), Mid), sub(getPoint(8), Mid));

		F6.norm();



		// Rckfacetteneliminierung
		float[] dummy = new float[6];

		for (int i = 1; i <= 8; i++) {
			HiddenPoints[i] = true;
		} 

		Point3D temp;
		float		test;

		// F1
		temp = sub(BP, getPoint(1));
		temp.norm();
		test = F1.VC_SK_Mult(temp);
		dummy[0] = test;
		if (test > 0) {
			setHiddenpoint(1, 2, 3, 4);
		} 

		// F2
		temp = sub(BP, getPoint(5));
		temp.norm();
		test = F2.VC_SK_Mult(temp);
		dummy[1] = test;
		if (test > 0) {
			setHiddenpoint(5, 6, 7, 8);
		} 

		// F3 = 2,4,6,8
		temp = sub(BP, getPoint(2));
		temp.norm();
		test = F3.VC_SK_Mult(temp);
		dummy[2] = test;
		if (test > 0) {
			setHiddenpoint(2, 4, 6, 8);
		} 

		// F4 = 1,3,5,7
		temp = sub(BP, getPoint(1));
		temp.norm();
		test = F4.VC_SK_Mult(temp);
		dummy[3] = test;
		if (test > 0) {
			setHiddenpoint(1, 3, 5, 7);
		} 

		// F5 = {1,2,5,6}
		temp = sub(BP, getPoint(1));
		temp.norm();
		test = F5.VC_SK_Mult(temp);
		dummy[4] = test;
		if (test > 0) {
			setHiddenpoint(1, 2, 5, 6);
		} 

		// F6 = 3,4,7,8
		temp = sub(BP, getPoint(3));
		temp.norm();
		test = F6.VC_SK_Mult(temp);
		dummy[5] = test;
		if (test > 0) {
			setHiddenpoint(3, 4, 7, 8);
		} 

		for (int i = 1; i <= 8; i++) {
			if (HiddenPoints[i]) {
				HiddenPoint = i;

			} 
		} 
		if (HiddenPoint == 3) {
			HiddenPoint = 3;

		} 
		return HiddenPoint;

	} 


	/**
	 * Die Methode zeichnet die gewhlte Projektion in den angegebenen grafischen Kontext.
	 * Zustzlich kann ein Translationsvektor und ein Vergrerungsfaktor angegeben werden.
	 * Falls eine schiefe Projektion mit verdeckten Kanten gezeichnet werden soll, dann kann
	 * global ber "showHiddenLines" eingestellt werden, ob diese weggelassen oder durch
	 * ein Strichmuster angedeutet werden sollen.
	 * 
	 * @param g grafischer kontext
	 * @param YConstant y-Komponente des Translationsvektors
	 * @param XConstant x-Komponente des Translationsvektors
	 * @param scale Vergrerungsfaktor
	 * 
	 * 
	 */
	public void drawCube(Graphics2D g, float YConstant, float XConstant, float scale) {

		// Beochbachtungspunkt an die Translationskoordinaten anpassen !
		// this.BP = Global_Options.BP ;

		if (Projection_Type == Global_Options.P_Kabinett) {

			HiddenPoint = calculateHidden();	// verdeckte Kanten berechnen


			// Projektion berechnen
		} 
		calculate2DPoints(YConstant, XConstant, scale);

		// Farbe bestimmen
		g.setColor(color);


		// p1 - p2
		float				dash1[] = {
			5.0f
		};
		BasicStroke bs2 = new BasicStroke(1.0f, BasicStroke.CAP_BUTT, BasicStroke.JOIN_MITER, 10.0f, dash1, 0.0f);
		BasicStroke bs1 = new BasicStroke(1.0f);

		g.setStroke(bs1);


		if (HiddenPoint != 1 && HiddenPoint != 2) {
			g.drawLine(get2dPointX(1), get2dPointY(1), get2dPointX(2), get2dPointY(2));
		} else if (Global_Options.showHiddenLines) {
			g.setStroke(bs2);
			g.drawLine(get2dPointX(1), get2dPointY(1), get2dPointX(2), get2dPointY(2));
			g.setStroke(bs1);
		} 

		// p1 - p3
		if (HiddenPoint != 1 && HiddenPoint != 3) {
			g.drawLine(get2dPointX(1), get2dPointY(1), get2dPointX(3), get2dPointY(3));
		} else if (Global_Options.showHiddenLines) {
			g.setStroke(bs2);
			g.drawLine(get2dPointX(1), get2dPointY(1), get2dPointX(3), get2dPointY(3));
			g.setStroke(bs1);
		} 

		// p2 - p4
		if (HiddenPoint != 4 && HiddenPoint != 2) {
			g.drawLine(get2dPointX(2), get2dPointY(2), get2dPointX(4), get2dPointY(4));
		} else if (Global_Options.showHiddenLines) {
			g.setStroke(bs2);
			g.drawLine(get2dPointX(2), get2dPointY(2), get2dPointX(4), get2dPointY(4));
			g.setStroke(bs1);
		} 

		// p4 - p3
		if (HiddenPoint != 4 && HiddenPoint != 3) {
			g.drawLine(get2dPointX(4), get2dPointY(4), get2dPointX(3), get2dPointY(3));
		} else if (Global_Options.showHiddenLines) {
			g.setStroke(bs2);
			g.drawLine(get2dPointX(4), get2dPointY(4), get2dPointX(3), get2dPointY(3));
			g.setStroke(bs1);
		} 

		// p1 - p5
		if (HiddenPoint != 1 && HiddenPoint != 5) {
			g.drawLine(get2dPointX(1), get2dPointY(1), get2dPointX(5), get2dPointY(5));
		} else if (Global_Options.showHiddenLines) {
			g.setStroke(bs2);
			g.drawLine(get2dPointX(1), get2dPointY(1), get2dPointX(5), get2dPointY(5));
			g.setStroke(bs1);
		} 


		// p2 - p6
		if (HiddenPoint != 2 && HiddenPoint != 6) {
			g.drawLine(get2dPointX(2), get2dPointY(2), get2dPointX(6), get2dPointY(6));
		} else if (Global_Options.showHiddenLines) {
			g.setStroke(bs2);
			g.drawLine(get2dPointX(2), get2dPointY(2), get2dPointX(6), get2dPointY(6));
			g.setStroke(bs1);
		} 

		// p5 - p6
		if (HiddenPoint != 5 && HiddenPoint != 6) {
			g.drawLine(get2dPointX(5), get2dPointY(5), get2dPointX(6), get2dPointY(6));
		} else if (Global_Options.showHiddenLines) {
			g.setStroke(bs2);
			g.drawLine(get2dPointX(5), get2dPointY(5), get2dPointX(6), get2dPointY(6));
			g.setStroke(bs1);
		} 

		// p4 - p8
		if (HiddenPoint != 4 && HiddenPoint != 8) {
			g.drawLine(get2dPointX(4), get2dPointY(4), get2dPointX(8), get2dPointY(8));
		} else if (Global_Options.showHiddenLines) {
			g.setStroke(bs2);
			g.drawLine(get2dPointX(4), get2dPointY(4), get2dPointX(8), get2dPointY(8));
			g.setStroke(bs1);
		} 

		// p6 - p8
		if (HiddenPoint != 6 && HiddenPoint != 8) {
			g.drawLine(get2dPointX(6), get2dPointY(6), get2dPointX(8), get2dPointY(8));
		} else if (Global_Options.showHiddenLines) {
			g.setStroke(bs2);
			g.drawLine(get2dPointX(6), get2dPointY(6), get2dPointX(8), get2dPointY(8));
			g.setStroke(bs1);
		} 

		// p3 - p7
		if (HiddenPoint != 3 && HiddenPoint != 7) {
			g.drawLine(get2dPointX(3), get2dPointY(3), get2dPointX(7), get2dPointY(7));
		} else if (Global_Options.showHiddenLines) {
			g.setStroke(bs2);
			g.drawLine(get2dPointX(3), get2dPointY(3), get2dPointX(7), get2dPointY(7));
			g.setStroke(bs1);
		} 

		// p7 - p8
		if (HiddenPoint != 7 && HiddenPoint != 8) {
			g.drawLine(get2dPointX(7), get2dPointY(7), get2dPointX(8), get2dPointY(8));
		} else if (Global_Options.showHiddenLines) {
			g.setStroke(bs2);
			g.drawLine(get2dPointX(7), get2dPointY(7), get2dPointX(8), get2dPointY(8));
			g.setStroke(bs1);
		} 

		// p5 - p7
		if (HiddenPoint != 5 && HiddenPoint != 7) {
			g.drawLine(get2dPointX(5), get2dPointY(5), get2dPointX(7), get2dPointY(7));
		} else if (Global_Options.showHiddenLines) {
			g.setStroke(bs2);
			g.drawLine(get2dPointX(5), get2dPointY(5), get2dPointX(7), get2dPointY(7));
			g.setStroke(bs1);
		} 


	} 


	/**
	 * Die Methode hilft den verdeckten Punkt einzuschrnken.
	 * Als Parameter werden 4 Indizes der Punkte angegeben, die nicht als verdeckte Punkte in Frage kommen.
	 * 
	 * @param a Index Punkt1
	 * @param b Index Punkt2
	 * @param c Index Punkt3
	 * @param d Index Punkt4
	 * 
	 */
	private void setHiddenpoint(int a, int b, int c, int d) {
		HiddenPoints[a] = false;
		HiddenPoints[b] = false;
		HiddenPoints[c] = false;
		HiddenPoints[d] = false;
	} 


	/**
	 * Die Methode setzt einen Korrekturfactor zur optimalen Projektion der
	 * Drahtmodelle bei Orthogonalprojektion (Mastabanpassung).
	 * 
	 * 
	 * @param factor2 Korrekturfaktor
	 * 
	 */
	public void setCorrectionFactor(float factor2) {
		this.factorY = factor2;
	} 

}








/*--- formatting done in "My Own Convention" style on 08-28-2000 ---*/

