/*--- formatted by Jindent 2.1, (www.c-lab.de/~jindent) ---*/

/*
 * Copyright (C) 2000 Thomas Hacklaender, e-mail: reply@thomas-hacklaender.de
 * Christian Schalla
 * Andreas Truemper
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License 2
 * as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * http://www.gnu.org/copyleft/copyleft.html
 */
package rad.dicom.dcm;

import java.util.*;
import java.io.*;


/**
 * Die Klasse repraesentiert ein DICOM Data Element (entspricht einem abstrakten
 * "Attribute"). Die Daten des Daten Elementes werden in einem Byte-Array
 * gespeichert. Darin ist die Bytefolge "little Endian". Fuer ein Element
 * ohne Daten hat das Byte Array die Laenge 0.
 * Erst beim I/O in <code>rad.dicom.dcm.io</code> wird ggf. eine Konvertierung
 * entsprechend der Transfer Syntax vorgenommen. Die dafuer notwendige
 * Methode <code>swapBuf(byte[], int VR)</code> ist dort als static definiert.
 * <DL><DT><B>Modifications: </B><DD>
 * tha 2000.07.06: In toString() trim eingefuegt. <br>
 * ans 2000.10.28: Cloneable/Serializable hinzugefgt, CopyConstructor, equals(), getDataAsCopy()
 * ans 2000.11.07: bufToMultipleXXXX() Diese Methoden prfen auf Multiplicity und liefern (wo vorhanden) mehrere Elemente zurck
 * </DD></DL>
 * @author   Thomas Hacklaender
 * @author   Christian Schalla
 * @author   Andreas Truemper
 * @version  2000.07.06
 */
public class DcmValue implements Cloneable, Serializable {


	/**
	 * Gruppennummer des Daten-Elements
	 */
	private int			group;


	/**
	 * Elementnummer des Daten-Elements
	 */
	private int			element;


	/**
	 * Datenbereich des Daten-Elements
	 */
	private byte[]	data;


	/**
	 * Typ des Daten-Elements
	 */
	private int			type = DcmDDE.DD_UNDEFINED;


	/**
	 * Typcode des Daten-Elements
	 */
	private int			vr = DcmDDE.VRCODE_UNDEFINED;


	/**
	 * Standardkonstruktor. Initialisiert eine Daten-Element mit Gruppennummer 0,
	 * Elementnummer 0 und 0 Byte Laenge.
	 */
	public DcmValue() {
		group = 0;
		element = 0;
		data = new byte[0];
	}


	/**
	 * Copy-Konstruktor. Erzeugt sich als Kopie des geg. DcmValue-Objektes
	 * @param o zu kopierendes DcmValue-Objekt
	 */
	public DcmValue(DcmValue o) {
		group = o.getGroup();
		element = o.getElement();
		data = o.getDataAsCopy();
		type = o.getType();
		vr = o.getVR();
	}


	/**
	 * Der Konstruktor initialisiert ein Daten-Element mit den uebergebenen Parametern.
	 * Ist d = null, so wird sein Inhalt auf Byte Array der Laenge 0 gesetzt.
	 * @param type  Der Typecode des Daten-Elements.
	 * @param d     Die Daten des Daten-Elements.
	 */
	public DcmValue(int type, byte[] d) {
		this(DcmDDE.getDictGroup(type), DcmDDE.getDictElement(type), d);
	}


	/**
	 * Der Konstruktor initialisiert ein Daten-Element mit den uebergebenen Parametern.
	 * Ist d = null, so wird sein Inhalt auf Byte Array der Laenge 0 gesetzt.
	 * @param g Der Gruppencode des Daten-Elements.
	 * @param e Der Elementcode des Daten-Elements.
	 * @param d Die Daten des Daten-Elements.
	 */
	public DcmValue(int g, int e, byte[] d) {
		group = g;
		element = e;
		if (d != null) {
			data = d;
		} else {
			data = new byte[0];
		}
	}


	/**
	 * Die Methode liefert den Gruppencode des Daten-Elements zurueck.
	 * @return Der Gruppencode des Daten-Elements.
	 */
	public int getGroup() {
		return group;
	}


	/**
	 * Die Methode liefert den Elementcode des Daten-Elements zurueck.
	 * @return Der Elementcode des Daten-Elements.
	 */
	public int getElement() {
		return element;
	}


	/**
	 * Die Methode liefer eine Referenz auf den Datenbereich des Daten-Elements
	 * zurueck. Die Bytefolge ist standardmaessig "little Endian". Es wird nie
	 * der Wert null zurueckgegeben, sondern mindestens ein Byte Array der
	 * Laenge 0.
	 * @return Die Referenz auf den Datenbereich des Daten-Elements.
	 */
	public byte[] getData() {
		if (data != null) {
			return data;
		} else {
			return new byte[0];
		}
	}


	/**
	 * Im Gegensatz zur Methode <code>getData()</code>, die eine Referenz auf
	 * die Daten zurckgibt, liefert diese Methode eine echte Kopie oder null,
	 * falls das Array nicht existiert.
	 */
	public byte[] getDataAsCopy() {
		if (data == null) {
			return null;
		}
		byte[]	res = new byte[data.length];

		for (int i = 0; i < data.length; i++) {
			res[i] = data[i];
		}
		return res;
	}


	/**
	 * Die Methode liefert den Typ des Daten-Elements zurueck. Ist der Type
	 * DcmDDE.DD_UNDEFINED, so wird der Defaultwert im Dictionary nachgeschlagen.
	 * @return Der Typ des Daten-Elements.
	 */
	public int getType() {
		if (type != DcmDDE.DD_UNDEFINED) {
			return type;
		} else {
			return DcmDDE.getDictType(group, element);
		}
	}


	/**
	 * Die Methode liefert den VR-Code des Daten-Elements zurueck. Ist der Code
	 * DcmDDE.VRCODE_UNDEFINED, so wird der Defaultwert im Dictionary nachgeschlagen.
	 * @return Der Typcode des Daten-Elements.
	 * @author Thomas Hacklaender
	 */
	public int getVR() {
		if (vr != DcmDDE.VRCODE_UNDEFINED) {
			return vr;
		} else {
			return DcmDDE.getDictVRCode(group, element);
		}
	}


	/**
	 * Die Methode setzt den VR-Code des Daten-Elements auf einen neuen
	 * Wert. Dies ist nur fuer die Faelle notwendig, in denen die Norm
	 * nicht eindeutig ist.
	 * @see DcmDDE
	 * @param vr Der Typecode des Daten-Elements.
	 * @author Thomas Hacklaender
	 */
	public void setVR(int v) {
		vr = v;
	}


	/**
	 * Clone-Methode.
	 * @return Liefert eine Kopie von sich selber.
	 */
	public Object clone() {
		return new DcmValue(this);
	}


	/**
	 * Vergleicht das DcmValue Objekt mit einem anderen.
	 * @param o		 Das zu vergleichende DcmValue Objekt.
	 * @return     Liefert TRUE, wenn beide Attribute in allen Belangen bereinstimmen,
   *             d.h. auch die selbe Endian-Kodierung besitzen.
	 */
	public boolean equals(Object o) {
		if ((o instanceof DcmValue) == false) {
			throw new IllegalArgumentException();
		}
		DcmValue	e = (DcmValue) o;

		if (e.getGroup() != getGroup()) {
			return false;
		}
		if (e.getElement() != getElement()) {
			return false;
		}
		if (e.getType() != getType()) {
			return false;
		}
		if (e.getVR() != getVR()) {
			return false;

			// Daten vergleichen
		}
		if ((e.getData() != null) && (getData() != null)) {
			if (e.getData().length != data.length) {
				return false;
			}
			int			n = data.length;
			byte[]	b1 = e.getData();
			byte[]	b2 = getData();

			for (int i = 0; i < n; i++) {
				if (b1[i] != b2[i]) {
					return false;

					// stimmen bis aufs letze Byte berein.
				}
			}
			return true;
		}

		// Mind. einer von beiden Daten ist == null
		return false;

	}


	/**
	 * Generiert einen String Darstellung des DcmValue. Eine Reihe von
	 * Formatparametern koennen festgelegt werden.
	 * Binaere Inhalte werden in Hexadecimalform umgesetzt.
	 * @param		withDes  Gibt an, ob der Name des Daten Elementes in den
	 * Ausgabestring mit aufgenommen werden sollen
	 * @param		desLen   Die im Ausgabestring fuer den Namen reservierte Laenge
	 * @param		withTag  Gibt an, ob Gruppennummer, Elementnummer,
	 * Value Representation und Tag Laenge in den
	 * Ausgabestring mit aufgenommen werden sollen
	 * @param		valueLen Die im Ausgabestring fuer den Inhalt reservierte Laenge
	 * @param		withQuotes Gibt an, ob Strings mit Anfuehrungszeichen eingefasst
	 * werden sollen
	 * @return	String Darstellung des Objektes
	 * @author	Thomas Hacklaender
	 */
	public String toString(boolean withDes, int desLen, boolean withTag, int valueLen, boolean withQuotes) {
		int			index;
		byte[]	buf;
		byte[]	newdata;
		String	theStr = "";
		String	endStr = "";
		int			t;
		int			vr;
		String	vrName;
		String	description;

		// Erzwingt die Suche im Dictionary, falls unbekannt
		t = getType();

		// Erzwingt die Suche im Dictionary, falls unbekannt
		vr = getVR();
		vrName = DcmDDE.vrCodeToString(vr);
		description = DcmDDE.getDictName(t);

		if (withDes) {
			theStr += padOrTruncate(description, desLen) + " ";
		}

		if (withTag) {
			theStr += "[" + toHexString((short) getGroup()) + ", " + toHexString((short) getElement()) + ", " + toHexString((int) getData().length) + ", " + padOrTruncate(vrName, 2) + "] ";
		}

		if (withDes || withTag) {
			theStr += "= ";
		}

		buf = getData();
		if (buf.length < 1) {
			theStr += "<null>";
			return theStr;
		}
		if (buf.length > valueLen) {
			newdata = new byte[valueLen];
			for (index = 0; index < valueLen; index++) {
				newdata[index] = buf[index];
			}
			buf = newdata;
			endStr = "<...>";
		}

		switch (vr) {

		case DcmDDE.VRCODE_AE:
		case DcmDDE.VRCODE_AS:
		case DcmDDE.VRCODE_CS:
		case DcmDDE.VRCODE_DA:
		case DcmDDE.VRCODE_DS:
		case DcmDDE.VRCODE_DT:
		case DcmDDE.VRCODE_IS:
		case DcmDDE.VRCODE_LO:
		case DcmDDE.VRCODE_LT:
		case DcmDDE.VRCODE_PN:
		case DcmDDE.VRCODE_SH:
		case DcmDDE.VRCODE_ST:
		case DcmDDE.VRCODE_TM:
		case DcmDDE.VRCODE_UI:
		case DcmDDE.VRCODE_UT:
			String	s = new String(buf);

			if (withQuotes) {
				theStr += "\"" + s + "\"";
			} else {
				theStr += s;
			}
			break;

		case DcmDDE.VRCODE_AT:
			int g = bufToUS(buf, 0);
			int e = bufToUS(buf, 4);

			theStr += "(" + toHexString((short) g) + ",";
			theStr += toHexString((short) e) + ")";
			break;

		case DcmDDE.VRCODE_OB:
		case DcmDDE.VRCODE_UN:
			for (index = 0; index < buf.length; index++) {
				theStr += toHexString((byte) buf[index]) + " ";
			}
			break;

		case DcmDDE.VRCODE_OW:
			for (index = 0; index < buf.length; index += 2) {
				short d = (short) bufToUS(buf, index);

				theStr += toHexString((short) d) + " ";
			}
			break;

		case DcmDDE.VRCODE_SQ:
			theStr += "<Sequence Item>";
			break;

		case DcmDDE.VRCODE_US:
			for (index = 0; index < buf.length; index += 2) {
				theStr += String.valueOf(bufToUS(buf, index)) + " ";
			}
			break;

		case DcmDDE.VRCODE_UL:
			for (index = 0; index < buf.length; index += 4) {
				theStr += String.valueOf(bufToUL(buf, index)) + " ";
			}
			break;

		case DcmDDE.VRCODE_SS:
			for (index = 0; index < buf.length; index += 2) {
				theStr += String.valueOf(bufToSS(buf, index)) + " ";
			}
			break;

		case DcmDDE.VRCODE_SL:
			for (index = 0; index < buf.length; index += 4) {
				theStr += String.valueOf(bufToSL(buf, index)) + " ";
			}
			break;

		case DcmDDE.VRCODE_FL:
			for (index = 0; index < buf.length; index += 4) {
				theStr += String.valueOf(bufToFL(buf, index)) + " ";
			}
			break;

		case DcmDDE.VRCODE_FD:
			for (index = 0; index < buf.length; index += 8) {
				theStr += String.valueOf(bufToFD(buf, index)) + " ";
			}
			break;

		default:
			for (index = 0; index < buf.length; index++) {
				theStr += toHexString(buf[index]) + " ";
			}
			break;
		}		// switch

		if (theStr.length() > valueLen) {
			theStr = theStr.substring(0, valueLen - 5);
			endStr = "<...>";
		}

		// tha 2000.07.06 eingefuegt
		theStr = theStr.trim();

		theStr += endStr;

		return theStr;
	}

	// ans 2000.10.28


	/**
	 * Method declaration
	 *
	 *
	 * @return
	 *
	 * @see
	 */
	public String toString() {
		return toString(true, 25, true, 70, true);
	}


	/**
	 * Wandelt ein 8 Bit Byte in einen hexadezimalen String der Form 0x00
	 * @author  Thomas Hacklaender
	 * @param  data   Das zu konvertierende Byte
	 * @return        Der Ergebnis-String
	 * @version 2000.1.30
	 */
	private String toHexString(byte data) {
		return ("0x" + addLeadingZeros(Integer.toHexString(data), 2));
	}


	/**
	 * Wandelt ein Byte-Array der Laenge 2 in einen hexadezimalen String
	 * der Form 0x0000
	 * @param  data   Das zu konvertierende Byte-Array
	 * @return        Der Ergebnis-String
	 * @author  CS
	 * @version 990619
	 */
	private String toHexString(short data) {
		return ("0x" + addLeadingZeros(Integer.toHexString(data), 4));
	}


	/**
	 * Wandelt ein Byte-Array der Laenge 4 in einen hexadezimalen String
	 * der Form 0x00000000
	 * @param  data   Das zu konvertierende Byte-Array
	 * @return        Der Ergebnis-String
	 * @author  CS
	 * @version 990619
	 */
	private String toHexString(int data) {
		return ("0x" + addLeadingZeros(Integer.toHexString(data), 8));
	}


	/**
	 * Formatiert einen beliebigen String so, dass seine Laenge genau
	 * length Zeichen betraegt. Ist der urspruengliche String kuerzer,
	 * wird er mit Leerzeichen aufgefuellt, anderenfalls abgeschnitten.
	 * @param   s       Der String der bearbeitet werden soll
	 * @param   length  Die gewuenschte Laenge
	 * @return          Der modifizierte String s
	 * @author  CS
	 * @version 990619
	 */
	private String padOrTruncate(String s, int length) {
		if (s == null) {
			return (s);
		}

		while (s.length() < length) {
			s = s + " ";
		}
		return (truncate(s, length));
	}


	/**
	 * Formatiert einen beliebigen String so, dass seine Laenge maximal
	 * length Zeichen betraegt.
	 * @param   s       Der String der bearbeitet werden soll
	 * @param   length  Die gewuenschte Laenge
	 * @return          Der modifizierte String s
	 * @author  CS
	 * @version 990619
	 */
	private String truncate(String s, int length) {
		if (s == null) {
			return s;
		}

		if (s.length() > length) {
			s = s.substring(0, length);
		}
		return s;
	}


	/**
	 * Fuegt vor beliebigen String soviele "0", dass seine Laenge
	 * len Zeichen betraegt.
	 * @param   str   Der String der bearbeitet werden soll
	 * @param   len   Die gewuenschte Laenge
	 * @return        Der neue String der Laenge len
	 * @author  CS
	 * @version 990619
	 */
	private String addLeadingZeros(String str, int len) {
		String	s = "0000000000000000000" + str;

		return (s.substring(s.length() - len));
	}


	/**
	 * Konvertiert 2 Bytes eines little Endian Byte-Arrays in ein
	 * 16 Bit signed Integer (ein Java short).
	 * @param buf		 Der Buffer mit den Rohdaten.
	 * @param offset Der Index, ab dem konvertiert werden soll.
	 * @return       Das Ergebnis als Integer.
	 */
	static public short bufToSS(byte[] buf, int offset) {
		return (short) ((((short) buf[offset++]) & 0xff) | (((short) buf[offset]) & 0xff) << 8);
	}


	/**
	 * Konvertiert 4 Bytes eines little Endian Byte-Arrays in ein
	 * 32 Bit signed Integer (ein Java int).
	 * @param buf		 Der Buffer mit den Rohdaten.
	 * @param offset Der Index, ab dem konvertiert werden soll.
	 * @return       Das Ergebnis als Integer.
	 */
	static public int bufToSL(byte[] buf, int offset) {
		return ((((int) buf[offset++]) & 0xff)) | ((((int) buf[offset++]) & 0xff) << 8) | ((((int) buf[offset++]) & 0xff) << 16) | ((((int) buf[offset]) & 0xff) << 24);
	}


	/**
	 * Konvertiert 4 Bytes eines little Endian Byte-Arrays in ein
	 * 32 Bit Floating-Point (ein Java float).
	 * @param buf		 Der Buffer mit den Rohdaten.
	 * @param offset Der Index, ab dem konvertiert werden soll.
	 * @return       Das Ergebnis als Floating-Point.
	 */
	static public float bufToFL(byte[] buf, int offset) {
		int bits = ((((int) buf[offset++]) & 0xff)) | ((((int) buf[offset++]) & 0xff) << 8) | ((((int) buf[offset++]) & 0xff) << 16) | ((((int) buf[offset]) & 0xff) << 24);

		return Float.intBitsToFloat(bits);
	}


	/**
	 * Konvertiert 8 Bytes eines little Endian Byte-Arrays in ein
	 * 64 Bit Floating-Point (ein Java double).
	 * @param buf		 Der Buffer mit den Rohdaten.
	 * @param offset Der Index, ab dem konvertiert werden soll.
	 * @return       Das Ergebnis als Floating-Point.
	 */
	static public double bufToFD(byte[] buf, int offset) {
		long	bits = ((((long) buf[offset++]) & 0xff)) | ((((long) buf[offset++]) & 0xff) << 8) | ((((long) buf[offset++]) & 0xff) << 16) | ((((long) buf[offset++]) & 0xff) << 24) | ((((long) buf[offset++]) & 0xff) << 32) | ((((long) buf[offset++]) & 0xff) << 40) | ((((long) buf[offset++]) & 0xff) << 48) | ((((long) buf[offset]) & 0xff) << 56);

		return Double.longBitsToDouble(bits);
	}


	/**
	 * Konvertiert 2 Bytes eines little Endian Byte-Arrays in ein
	 * 16 Bit unsigned Integer (ein Java int).
	 * @param buf		 Der Buffer mit den Rohdaten.
	 * @param offset Der Index, ab dem konvertiert werden soll.
	 * @return       Das Ergebnis als Integer.
	 */
	static public int bufToUS(byte[] buf, int offset) {
		return ((((int) buf[offset++]) & 0xff)) | ((((int) buf[offset]) & 0xff) << 8);
	}


	/**
	 * Konvertiert 4 Bytes eines little Endian Byte-Arrays in ein
	 * 32 Bit unsigned Integer (ein Java long).
	 * @param buf		 Der Buffer mit den Rohdaten.
	 * @param offset Der Index, ab dem konvertiert werden soll.
	 * @return       Das Ergebnis als Integer.
	 */
	static public long bufToUL(byte[] buf, int offset) {
		return ((((long) buf[offset++]) & 0xff)) | ((((long) buf[offset++]) & 0xff) << 8) | ((((long) buf[offset++]) & 0xff) << 16) | ((((long) buf[offset]) & 0xff) << 24);
	}


	/**
	 * Konvertiert ein 16 Bit signed Integer (ein Java short) in eine
	 * 2 Bytes little Endian Bytefolge und legt sie in einem Byte-Array
	 * ab einem definierten Offset ab.
	 * @param int		 Der zu konvertierende Integer Wert.
	 * @param buf		 Der Buffer mit den Rohdaten.
	 * @param offset Der Index, ab dem konvertiert werden soll.
	 */
	static public void ssToBuf(short n, byte[] buf, int offset) {
		buf[offset++] = (byte) (n & 0xff);
		buf[offset] = (byte) ((n >> 8) & 0xff);
	}


	/**
	 * Konvertiert ein 32 Bit signed Integer (ein Java int) in eine
	 * 4 Bytes little Endian Bytefolge und legt sie in einem Byte-Array
	 * ab einem definierten Offset ab.
	 * @param int		 Der zu konvertierende Integer Wert.
	 * @param buf		 Der Buffer mit den Rohdaten.
	 * @param offset Der Index, ab dem konvertiert werden soll.
	 */
	static public void slToBuf(int n, byte[] buf, int offset) {
		buf[offset++] = (byte) (n & 0xff);
		buf[offset++] = (byte) ((n >> 8) & 0xff);
		buf[offset++] = (byte) ((n >> 16) & 0xff);
		buf[offset] = (byte) ((n >> 24) & 0xff);
	}


	/**
	 * Konvertiert ein 16 Bit unsigned Integer (ein Java int) in eine
	 * 2 Bytes little Endian Bytefolge und legt sie in einem Byte-Array
	 * ab einem definierten Offset ab.
	 * @param int		 Der zu konvertierende Integer Wert.
	 * @param buf		 Der Buffer mit den Rohdaten.
	 * @param offset Der Index, ab dem konvertiert werden soll.
	 */
	static public void usToBuf(int n, byte[] buf, int offset) {
		buf[offset++] = (byte) (n & 0xff);
		buf[offset] = (byte) ((n >>> 8) & 0xff);
	}


	/**
	 * Konvertiert ein 32 Bit unsigned Integer (ein Java long) in eine
	 * 4 Bytes little Endian Bytefolge und legt sie in einem Byte-Array
	 * ab einem definierten Offset ab.
	 * @param int		 Der zu konvertierende Integer Wert.
	 * @param buf		 Der Buffer mit den Rohdaten.
	 * @param offset Der Index, ab dem konvertiert werden soll.
	 */
	static public void ulToBuf(long n, byte[] buf, int offset) {
		buf[offset++] = (byte) (n & 0xff);
		buf[offset++] = (byte) ((n >>> 8) & 0xff);
		buf[offset++] = (byte) ((n >>> 16) & 0xff);
		buf[offset] = (byte) ((n >>> 24) & 0xff);
	}


	/**
	 * Konvertiert ein 32 Bit Floating-Point (ein Java float) in eine
	 * 4 Bytes little Endian Bytefolge und legt sie in einem Byte-Array
	 * ab einem definierten Offset ab.
	 * @param f   	 Der zu konvertierende Floating-Point Wert.
	 * @param buf		 Der Buffer mit den Rohdaten.
	 * @param offset Der Index, ab dem konvertiert werden soll.
	 */
	static public void flToBuf(float f, byte[] buf, int offset) {
		int n = Float.floatToIntBits(f);

		buf[offset++] = (byte) (n & 0xff);
		buf[offset++] = (byte) ((n >>> 8) & 0xff);
		buf[offset++] = (byte) ((n >>> 16) & 0xff);
		buf[offset] = (byte) ((n >>> 24) & 0xff);
	}


	/**
	 * Konvertiert ein 64 Bit Floating-Point (ein Java double) in eine
	 * 8 Bytes little Endian Bytefolge und legt sie in einem Byte-Array
	 * ab einem definierten Offset ab.
	 * @param d   	 Der zu konvertierende Floating-Point Wert.
	 * @param buf		 Der Buffer mit den Rohdaten.
	 * @param offset Der Index, ab dem konvertiert werden soll.
	 */
	static public void fdToBuf(double d, byte[] buf, int offset) {
		long	n = Double.doubleToLongBits(d);

		buf[offset++] = (byte) (n & 0xff);
		buf[offset++] = (byte) ((n >>> 8) & 0xff);
		buf[offset++] = (byte) ((n >>> 16) & 0xff);
		buf[offset++] = (byte) ((n >>> 24) & 0xff);
		buf[offset++] = (byte) ((n >>> 32) & 0xff);
		buf[offset++] = (byte) ((n >>> 40) & 0xff);
		buf[offset++] = (byte) ((n >>> 48) & 0xff);
		buf[offset] = (byte) ((n >>> 56) & 0xff);
	}


	/**
	 * Die Methode swappt die Bytefolge des Buffers zwischen little und big
	 * Endian. Dabei wird der Typ des Daten-Elementes bercksichtigt.
	 * @param buf		Der zu veraendernde Buffer
	 * @param vr		Der Typecode fuer die Daten im Buffer
	 * @author	Thomas Hacklaender
	 */
	public static void swapBuf(byte[] buf, int vr) {
		int		i;
		byte	temp;

		switch (vr) {

		case DcmDDE.VRCODE_AT:
		case DcmDDE.VRCODE_OW:
		case DcmDDE.VRCODE_SS:
		case DcmDDE.VRCODE_US:	// Swap 2 Byte
			for (i = 0; i < buf.length; i += 2) {
				temp = buf[i];
				buf[i] = buf[i + 1];
				buf[i + 1] = temp;
			}
			break;

		case DcmDDE.VRCODE_FL:
		case DcmDDE.VRCODE_SL:
		case DcmDDE.VRCODE_UL:	// Swap 4 Byte
			for (i = 0; i < buf.length; i += 4) {
				temp = buf[i];
				buf[i] = buf[i + 3];
				buf[i + 3] = temp;
				temp = buf[i + 1];
				buf[i + 1] = buf[i + 2];
				buf[i + 2] = temp;
			}
			break;

		case DcmDDE.VRCODE_FD:	// Swap 8 Byte
			for (i = 0; i < buf.length; i += 8) {
				temp = buf[i];
				buf[i] = buf[i + 7];
				buf[i + 7] = temp;
				temp = buf[i + 1];
				buf[i + 1] = buf[i + 6];
				buf[i + 6] = temp;
				temp = buf[i + 2];
				buf[i + 2] = buf[i + 5];
				buf[i + 5] = temp;
				temp = buf[i + 3];
				buf[i + 3] = buf[i + 4];
				buf[i + 4] = temp;
			}
			break;

		case DcmDDE.VRCODE_AE:
		case DcmDDE.VRCODE_AS:
		case DcmDDE.VRCODE_CS:
		case DcmDDE.VRCODE_DA:
		case DcmDDE.VRCODE_DS:
		case DcmDDE.VRCODE_DT:
		case DcmDDE.VRCODE_IS:
		case DcmDDE.VRCODE_LO:
		case DcmDDE.VRCODE_LT:
		case DcmDDE.VRCODE_OB:
		case DcmDDE.VRCODE_PN:
		case DcmDDE.VRCODE_SH:
		case DcmDDE.VRCODE_SQ:	// siehe PS 3.5-7.5
		case DcmDDE.VRCODE_ST:
		case DcmDDE.VRCODE_TM:
		case DcmDDE.VRCODE_UI:

		case DcmDDE.VRCODE_UN:
		case DcmDDE.VRCODE_UNDEFINED:
		default:
			break;								// Daten-Elemente werden nicht geswappt

		}												// switch
	}


	/**
	 * Wandelt einen String in ein Longinteger um.
	 * @param String  Die String-Darstellung der Longinteger-Zahl.
	 * @param def     Ein Default-Wert.
	 * @return        Das Ergebnis der Umwandlung. Falls diese nicht erfolgreich
	 * durchgefuehrt werden kann, wird der Default-Wert
	 * zurueckgegeben.
	 */
	static public long str2Long(String s, long def) {
		try {
			return Long.parseLong(s.trim());
		} catch (Exception err) {
			return def;
		}
	}


	/**
	 * Wandelt einen String in ein Double um.
	 * @param String  Die String-Darstellung der Double-Zahl.
	 * @param def     Ein Default-Wert.
	 * @return        Das Ergebnis der Umwandlung. Falls diese nicht erfolgreich
	 * durchgefuehrt werden kann, wird der Default-Wert
	 * zurueckgegeben.
	 */
	static public double str2Double(String s, double def) {
		try {
			return Double.valueOf(s.trim()).doubleValue();
		} catch (Exception err) {
			return def;
		}
	}


	/**
	 * Zerlegt einen String in ein Array von Teilstrings, die im Ursprungsstring
	 * durch definierte Zeichen voneinander getrennt sind. DICOM Elemente vom
	 * Typ String benutzen dazu meist das Zeichen '\'. Um diese Zeichen als
	 * Delimiter verwenden zu koennen, muss man es als "\\" uebergeben. Die
	 * Trennzeichen selbst sind in den Unterstrings nicht enthalten.
	 * @param s   	Der zu zerlegende Ursprungsstring.
	 * @param delim Ein String von Trennzeicehn.
	 * @return      Ein String-Array mit den Teilstrigs. beinhaltet der Ursprungs-
	 * string keine Zeichen wird ein String-Array der Laenge 0
	 * zurueckgegen.
	 */
	static public String[] str2StringArray(String s, String delim) {
		StringTokenizer st;
		String[]				array;
		int							n;

		int							index = -1;
		String					str;
		Vector					values = new Vector();

		if ((s == null) | (s.length() == 0)) {
			return new String[0];
		}

		st = new StringTokenizer(s, delim);
		n = st.countTokens();
		array = new String[n];

		for (int i = 0; i < n; i++) {
			array[i] = st.nextToken();
		}

		return array;
	}

}




/*--- formatting done in "My Own Convention" style on 11-11-2000 ---*/

