/*--- formatted by Jindent 2.1, (www.c-lab.de/~jindent) ---*/

/*
 * Copyright (C) 2000 Thomas Hacklaender, e-mail: reply@thomas-hacklaender.de
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License 2
 * as published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * 
 * http://www.gnu.org/copyleft/copyleft.html
 */
package rad.dicom.ima;

import java.awt.*;
import java.awt.image.*;
import java.awt.event.*;
import javax.swing.*;
import javax.swing.border.*;

import rad.dicom.dcm.*;


/**
 * Mit dieser Klasse koennen die Bildinformationen einer
 * <code>GeneralImageIOD</code> als JPanel dargestellt werden. Das Bild
 * wird entweder auf die Groesse des Panels skaliert, oder erhaelt ggf.
 * horizontale und vertikale Scrollbalken zur Darstellung in der
 * Originalgroesse. Falls gewuenscht, koennen Center und Width ueber
 * Mausbewegungen bei gedrueckter Muastaste veraendert werden. <br>
 * Die 16 Bit Pixelinformationen werden entsprechend den Center/Width
 * Werten in ein 8 Bit Pixelarray umgerechnet. Aus diesem und dem ColorModel
 * des GeneralImageIOD wird daraus eine MemoryImageSource erstellt. Zur
 * Darstellung wird daraus ein ImageIcon generiert, das als JLabel auf dem
 * ImagePanel dargestellt wird.
 * <DL><DT><B>Modifications: </B><DD>
 * tha 2000.02.29: iconLabel und scrollPane initialisiert. <br>
 * tha 2000.03.05: Window Eistellungen beziehen sich immer auf Pixelwerte. Auch
 * bei CT Bildern. Houndsfieldeinheiten werden bereits in
 * GeneralImageIOD in Pixelwerte umgerechnet. <br>
 * tha 2000.04.02: Bei setImage die Einstellung von Window auch in
 * GeneralImageIOD eintragen. <br>
 * tha 2000.04.02: calcAutomaticCW in GeneralImageIOD verlagert. <br>
 * tha 2000.04.02: Die Methode getImageIcon implementiert. <br>
 * </DD></DL>
 * @author  Thomas Hacklaender
 * @version 2000.05.04
 */
public class ImagePanel extends JPanel {

	// Constants for Center/Width setup
	public static final int		CW_DEFAULT = 0;
	public static final int		CW_AUTO = 1;

	/* Property "scrollable" */
	private boolean						scrollable = false;

	/* Property "windowable" */
	private boolean						windowable = true;

	/*
	 * Property "preferredSize"
	 * Die Default Bildgre ist 256*256 Pixel. Hinzu kommt:
	 * - ein 1 Pixel breiter Rand fuer die Selektion des Bildes
	 * - ein 2 Pixel breiter Rand fuer die ScrollPane.
	 */
	private Dimension					preferredSize = new Dimension(256 + 2 * 1 + 2 * 2, 256 + 2 * 1 + 2 * 2);

	/*
	 * Property "minimumSize"
	 * Die kleinste Bildgroesse ist 32 * 32 Pixel.
	 */
	private Dimension					minimumSize = new Dimension(32 + 2 * 1 + 2 * 2, 32 + 2 * 1 + 2 * 2);

	/* Property "background". Standardfarbe ist darkGray. */
	private Color							background = Color.darkGray;

	/* Die GeneralImageIOD des gerade dargestellten Bildes */
	private GeneralImageIOD		genIma = null;

	/* Das von GeneralImageIOD abgeleitete Image */
	private Image							img = null;

	/* Das von dem Image abgeleitete ImageIcon */
	private ImageIcon					iIcon = null;

	/* Zeitpunkt des Mouse Pressed Events */
	private long							lastWhen;

	/* Koordinaten der Mouse beim Mouse Pressed Event */
	private int								lastX, lastY;

	/*
	 * Defaultwert fuer Center/Width. Wird aus den Feldern windowCenter und
	 * windowWidth des aktuelle GeneralImageIOD uebernommen. Die Wert
	 * entsprechen Pixelwerten.
	 */
	private int								defCenter;
	private int								defWidth;

	/* Die aktuell auf dem Bild dargestellten Wert von Center/Width. */
	private int								theCenter;
	private int								theWidth;

	/* Allgemeine globale Felder der Instanz */

	// tha	2000.2.29: geaendert
	// private JLabel        	iconLabel   = null;
	JLabel										iconLabel = new JLabel();

	// tha	2000.2.29: geaendert
	// private JScrollPane   	scrollPane  = null;
	JScrollPane								scrollPane = new JScrollPane();
	Border										grayBorder = new LineBorder(Color.darkGray, 1);
	Border										greenBorder = new LineBorder(Color.green, 1);
	BorderLayout							borderLayout1 = new BorderLayout();

	/* Center/Window Einstellung beim Mouse Pressed Event */
	private int								lastCenter, lastWidth;


	/**
	 * Erzeugt ein ImageBtnPanel ohne Bildinformation.
	 * Hier wird die darzustellende GeneralImageIOD noch nicht angegeben, da sie
	 * moeglicherweise noch nicht bekannt ist und das Panel die Moeglichkeit
	 * bieten muss, nacheinander verschiede Pixelmatrizen anzuzeigen.
	 */
	public ImagePanel() {
		try {
			jbInit();
		} catch (Exception ex) {
			ex.printStackTrace();
		} 
	}


	/**
	 * Es wird eine neue GeneralImageIOD zur Darstellung uebergeben.
	 * Das neue Bild wird mit diesen Daten neu dargestellt.
	 * @param gi	Die neue GeneralImageIOD des Bildes.
	 */
	public void setImage(GeneralImageIOD gi) {
		genIma = gi;

		if (genIma.pixel16.length == 0) {
			return;

		} 
		if (genIma.isCenterWidth) {
			if (genIma.classUID.compareTo(DcmUID.CT_STORAGE_UID) == 0) {

				// Bei CT-Bildern Houndsfield-Einheiten in Pixelwerte umrechnen
				int[] window = new int[2];

				window[0] = genIma.windowCenter;
				window[1] = genIma.windowWidth;
				genIma.hu2pv(window);
				defCenter = window[0];
				defWidth = window[1];
			} else {
				defCenter = genIma.windowCenter;
				defWidth = genIma.windowWidth;
			} 
		} else {
			defCenter = genIma.autoCenter;
			defWidth = genIma.autoWidth;
		} 
		theCenter = defCenter;
		theWidth = defWidth;

		// Aenderungen auch in GeneralImageIOD eintragen
		genIma.lastCenter = theCenter;
		genIma.lastWidth = theWidth;

		iconLabel = new JLabel();
		iconLabel.setOpaque(true);
		iconLabel.setBackground(background);
		iconLabel.setHorizontalAlignment(SwingConstants.CENTER);
		iconLabel.setVerticalAlignment(SwingConstants.CENTER);
		updateIconLabel();

		if (scrollPane != null) {
			this.remove(scrollPane);
		} 
		scrollPane = new JScrollPane(iconLabel);
		scrollPane.setBackground(background);
		this.add(scrollPane, BorderLayout.CENTER);

		this.validate();
	} 


	/**
	 * Center und Width des Bildes werden auf Standardwerte gesetzt:
	 * Das Bild wird anschliessend mit den Werten neu gezeichnet.
	 * @param	type	Standardwerte fuer die Bildeinstellung:
	 * CW_DEFAULT = Das Wertepaar, das fuer das Bild als
	 * Defaultwert festgelegt wurde. Normalerweise werden das die
	 * Werte aus der DICOM "VOI LUT Module" sein.
	 * CW_AUTO = Die Werte werden automatisch so berechnet, dass
	 * alle Pixelwerte zwischen der 10ten und 90ten Perzentiele
	 * dargestellt werden.
	 */
	public void setCW(int type) {
		switch (type) {
		case CW_DEFAULT:
			theCenter = defCenter;
			theWidth = defWidth;
			break;

		case CW_AUTO:
			theCenter = genIma.autoCenter;
			theWidth = genIma.autoWidth;
			break;
		}

		// Aenderungen auch in GeneralImageIOD eintragen
		genIma.lastCenter = theCenter;
		genIma.lastWidth = theWidth;

		updateIconLabel();
	} 


	/**
	 * Center und Width des Bildes werden auf numerische Werte gesetzt. Das
	 * Bild wird anschliessend mit den Werten neu gezeichnet.
	 * @param	center	Center Einstellung des Bildes.
	 * @param	window	Width Einstellung des Bildes.
	 */
	public void setCW(int center, int width) {
		theCenter = center;
		theWidth = width;

		// Aenderungen auch in GeeralImageIOD eintragen
		genIma.lastCenter = theCenter;
		genIma.lastWidth = theWidth;

		updateIconLabel();
	} 


	/**
	 * Liefert den aktuellen Center Wert des Bildes.
	 * @return  Der aktuelle Center Wert des Bildes.
	 */
	public int getCenter() {
		return theCenter;
	} 


	/**
	 * Liefert den aktuellen Width Wert des Bildes.
	 * @return  Der aktuelle Width Wert des Bildes.
	 */
	public int getWidth() {
		return theWidth;
	} 


	/**
	 * Liefert den aktuellen Wert des dargestellten ImageIcon.
	 * @return  Der aktuelle Width Wert des ImageIcon.
	 */
	public ImageIcon getImageIcon() {
		return iIcon;
	} 


	/**
	 * Die Property legt fest, wie ein Bild dargestellt wird, das groesser als
	 * das Panel ist: Entweder wird es auf die Groesse des Panels skaliert,
	 * oder das Panel erhaelt Scollbalken.
	 * @param	scr	false: Bild wird skaliert. true: ggf. Darstellung mit Scrollbalken.
	 */
	public void setScrollable(boolean scr) {
		scrollable = scr;
		if (genIma != null) {
			updateIconLabel();
		} 
	} 


	/**
	 * Liefert den aktuellen Wert der Property "scrollable"
	 */
	public boolean isScrollable() {
		return scrollable;
	} 


	/**
	 * Die Property legt fest, ob Cernter und Window Einstellungen des Bildes
	 * ueber Mausbewegungen geaendert werden duerfen.
	 * @param	win		true: Einstellung ueber die Mause erlaubt.
	 */
	public void setWindowable(boolean win) {
		windowable = win;
	} 


	/**
	 * Liefert den aktuellen Wert der Property "windowable"
	 */
	public boolean isWindowable() {
		return windowable;
	} 


	/**
	 * Das Bild wird als IconLabel auf dem Panel dargestellt. Mit dieser Methode
	 * wird das IconLabel entprechend den aktuellen Einstellungen fuer
	 * Pixelmatrix, Center und Window neu generiert und dann dargestellt.
	 */
	private void updateIconLabel() {
		double	fac;
		double	imW, imH;

		if (genIma == null) {
			return;

		} 

		if (img != null) {
			img.flush();
		}
    img = ImaUtil.GIODtoImage(genIma, theCenter, theWidth);

		imW = (double) img.getWidth(this);
		imH = (double) img.getHeight(this);
		if (scrollable) {
			iIcon = new ImageIcon(img);
		} else {

			// Die Groesse des Panels ist die eigentliche Bildgroesse plus ein
			// - ein 1 Pixel breiter Rand fuer die Selektion des Bildes
			// - ein 2 Pixel breiter Rand fuer die ScrollPane.
			Dimension size = this.getSize();

			double		wFac = imW / (size.width - 6);
			double		hFac = imH / (size.height - 6);

			if (wFac >= hFac) {
				fac = wFac;
			} else {
				fac = hFac;
			}
			iIcon = new ImageIcon(img.getScaledInstance((int) (imW / fac), (int) (imH / fac), Image.SCALE_DEFAULT));
		} 
		iconLabel.setIcon(iIcon);
		iconLabel.setSize(iIcon.getIconWidth(), iIcon.getIconHeight());
		this.validate();
	} 


	/**
	 * Das Panel wird neu gezeichnet.
	 */
	public void repaint() {
		updateIconLabel();
		super.repaint();
	} 


	/**
	 * Eventhandler fuer Mouse Pressed Events (Name von JBuielder generiert).
	 * Falls die Einstellungen fuer das Bild veraenderbar sind:
	 * Statt des grauen Rahmens wird ein gruener um das Bild gezeichnet.
	 * @param	e		Das Event.
	 */
	void this_mousePressed(MouseEvent e) {

		if (!windowable) {
			return;

		} 
		lastX = e.getX();
		lastY = e.getY();
		lastCenter = getCenter();
		lastWidth = getWidth();
		lastWhen = e.getWhen();
		this.setBorder(greenBorder);
		this.repaint();
	} 


	/**
	 * Eventhandler fuer Mouse Released Events (Name von JBuielder generiert).
	 * Falls die Einstellungen fuer das Bild veraenderbar sind:
	 * Es wird ein grauer Rahmen um das Bild gezeichnet.
	 * @param	e		Das Event.
	 */
	void this_mouseReleased(MouseEvent e) {

		if (!windowable) {
			return;

		}
		this.setBorder(grayBorder);
		this.repaint();
	}


	/**
	 * Eventhandler fuer Mouse Dragged Events (Name von JBuielder generiert).
	 * Falls die Einstellungen fuer das Bild veraenderbar sind:
	 * Solange die Mousetaste gedrueckt bleibt, werden Center und Window des
	 * Bildes ueber die Mousebewegung geaendert. Window = links/rechts,
	 * Center = oben/unten.
	 * @param	e		Das Event.
	 */
	void this_mouseDragged(MouseEvent e) {
		int		actX, actY;
		int		deltaX, deltaY;
		long	actWhen;
		int		deltaT;

		if (!windowable) {
			return;

		} 
		actWhen = e.getWhen();
		deltaT = (int) (actWhen - lastWhen);
		if (deltaT < 100) {
			return;		// Max 10 updates/sec

		} 
		actX = e.getX();
		actY = e.getY();
		deltaX = actX - lastX;
		deltaY = lastY - actY;
		if (Math.abs(deltaX) > 10) {
			deltaX = deltaX * 5;
		} 
		if (Math.abs(deltaY) > 10) {
			deltaY = deltaY * 5;
		} 
		setCW(lastCenter + deltaY, lastWidth + deltaX);
		updateIconLabel();

		lastX = actX;
		lastY = actY;
		lastCenter = getCenter();
		lastWidth = getWidth();
		lastWhen = actWhen;
	} 


	/**
	 * Von JBuilder automatisch generierte Initialisierungsmethode fuer die
	 * grafischen Elemente.
	 */
	private void jbInit() throws Exception {

		// tha	2000.2.29: eingefuegt:
		iconLabel.setOpaque(true);
		iconLabel.setBackground(background);
		scrollPane.setBackground(background);

		// the	end

		this.setMinimumSize(minimumSize);				// Von Hand eingefuegt
		this.setPreferredSize(preferredSize);		// Von Hand eingefuegt

		this.setBorder(grayBorder);

		this.addMouseMotionListener(new java.awt.event.MouseMotionAdapter() {

			/**
			 */
			public void mouseDragged(MouseEvent e) {
				this_mouseDragged(e);
			} 

		});

		this.addMouseListener(new java.awt.event.MouseAdapter() {

			/**
			 */
			public void mouseReleased(MouseEvent e) {
				this_mouseReleased(e);
			} 

			/**
			 */
			public void mousePressed(MouseEvent e) {
				this_mousePressed(e);
			} 

		});
		
		this.setLayout(borderLayout1);
		iconLabel.setMinimumSize(new Dimension(32, 32));
		iconLabel.setPreferredSize(new Dimension(256, 256));
		scrollPane.setMinimumSize(new Dimension(37, 37));
		scrollPane.setPreferredSize(new Dimension(261, 261));
		scrollPane.getViewport().add(iconLabel, null);
		this.add(scrollPane, BorderLayout.CENTER);
	} 

}

/*--- formatting done in "My Own Convention" style on 04-30-2000 ---*/

