/*--- formatted by Jindent 2.1, (www.c-lab.de/~jindent) ---*/

/*
 * Copyright (C) 2000 Thomas Hacklaender, e-mail: reply@thomas-hacklaender.de
 * Christian Schalla
 * Andreas Truemper
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License 2
 * as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * http://www.gnu.org/copyleft/copyleft.html
 */
package rad.dicom.dcm;

import java.io.*;
import java.util.*;


/**
 * Die Klasse repraesentiert ein DICOM Information Object Definition (IOD).
 * Dazu verwaltet sie die verschiedenen Gruppen von DICOM-Elementen in einer
 * Hashtabelle.
 * Die Elemente jeder Gruppe werden wiederum in einer Hashtabelle verwaltet.
 * Die Klasse bietet Methoden, um neue DICOM-Elemente hinzuzufuegen und
 * zurueckzuliefern.
 *
 * <DL><DT><B>Modifications: </B><DD>
 * tha  2000.11.12: getMultiplicity() implementiert.<br>
 * tha  2000.11.12: getXXArray() implementiert. Liefert ein Array von Ergebnissen
 * zurueck<br>
 * tha  2000.11.12: hasValue() verbessert: Ueberprueft jetzt, ob eine korrekte
 * Anzahl von Bytes im Datenteil vorhanden ist.<br>
 * ans  2000.11.07: extract, merge, clear implementiert, aus zuknftigem DcmTransform.
 * vorgezogen.<br>
 * ans  2000.10.28: Cloneable/Serializable hinzugefgt, CopyConstructor, equals()<br>
 * tha	2000.1.29: Transfersyntax wird fuer jede Instanz festgelegt.
 * </DD></DL>
 * @author   Thomas Hacklaender
 * @author   Christian Schalla
 * @author   Andreas Truemper
 * @version  2000.11.12
 */
public class DcmDataObject implements Cloneable, Serializable {


	/**
	 * Konstante zur Repraesentation des little Endian Encodings
	 */
	public static final int		LITTLE_ENDIAN = 1;


	/**
	 * Konstante zur Repraesentation des big Endian Encodings
	 */
	public static final int		BIG_ENDIAN = 2;


	/**
	 * Konstante zur Anforderung der automatischen Endian Erkennung
	 */
	public static final int		AUTO_ENDIAN = 4;


	/**
	 * Konstante zur Repraesentation der implicite VR Structure
	 */
	public static final int		IMPLICITE_VR = 8;


	/**
	 * Konstante zur Repraesentation der explicite VR Structure
	 */
	public static final int		EXPLICITE_VR = 16;


	/**
	 * Konstante zur Anforderung der automatischen Structure Erkennung
	 */
	public static final int		AUTO_VR = 32;


	/**
	 * Konstante zur Festlegung des externen Speicherformates:
	 * Ohne vorangestellten DICOM File Meta Information.
	 */
	public static final int		PLAIN_STORAGE = 64;


	/**
	 * Konstante zur Festlegung des externen Speicherformates:
	 * Mit vorangestellten DICOM File Meta Information entsprechend
	 * PS 3.10-7.1 DICOM.
	 */
	public static final int		META_STORAGE = 128;


	/**
	 * Konstante zur Anforderung der automatischen Erkennung des externen
	 * Speicherforamtes
	 */
	public static final int		AUTO_STORAGE = 256;


	/**
	 * Die Hashtabelle, die die Gruppen verwaltet
	 */
	Hashtable									grouptable = new Hashtable();


	/**
	 * Ist true, wenn eine neue Gruppe gelesen wurde. Global fuer Method pop
	 */
	private transient boolean popNewGroup = true;

	/*
	 * Achtung: Alle globalen Variablen mussen in der Methode getCopyOfMe
	 * und getHeaderCopyOfMe mit kopiert werde!
	 */

	/* Das Encoding der Trasfersyntax fuer die Instanz der Klasse */
	private int								encoding = LITTLE_ENDIAN;

	/* Die Structre der Trasfersyntax fuer die Instanz der Klasse */
	private int								structure = EXPLICITE_VR;

	/* Das Speicherformt des externen Speicher fuer die Instanz der Klasse */
	private int								storage = META_STORAGE;


	/**
	 * Standardkonstruktor. Die Transfersyntax wird als Defaultwert auf
	 * little Endian, implicite VR gesetzt.
	 */
	public DcmDataObject() {}


	/**
	 * Konstruktor. Die Transfersyntax wird angegeben.
	 * @param enc	Encoding der Transfersyntax
	 * @param str	Structure der Transfersyntax
	 * @param sto	Speicherformat des externen Speichers
	 */
	public DcmDataObject(int enc, int str, int sto) {
		encoding = enc;
		structure = str;
		storage = sto;
	}


	/**
	 * Copy-Konstruktor. Alle Datenelemente, Encoding, Structure
	 * und Speicherformat werden aus dem geg. DcmDataObject kopiert.
	 * @param o 	zu kopierendes DcmDataObject
	 */
	public DcmDataObject(DcmDataObject o) {
		grouptable = (Hashtable) o.grouptable.clone();
		encoding = o.getEncoding();
		structure = o.getStructure();
		storage = o.getStorage();
	}


	/**
	 * Liefert die Bytefolge der Transfersyntax. Diese kann nach Erzeugung
	 * des DcmDataObjects nicht mehr veraendert werde, da sonst Inkonsistenzen
	 * in der Darstellung der DcmValues und auch der Laenge der Gruppen
	 * entstehen koennten (setEncoding() darf nicht implementiert werden!).
	 * @return	Die Bytefolge
	 * @author	Thomas Hacklaender
	 */
	public int getEncoding() {
		return encoding;
	}


	/**
	 * Liefert die Struktur der Transfersyntax. Diese kann nach Erzeugung
	 * des DcmDataObjects nicht mehr veraendert werde, da sonst Inkonsistenzen
	 * in der Darstellung der DcmValues und auch der Laenge der Gruppen
	 * entstehen koennten (setStructure() darf nicht implementiert werden!).
	 * @return	Die Struktur
	 * @author	Thomas Hacklaender
	 */
	public int getStructure() {
		return structure;
	}


	/**
	 * Liefert das Speicherformt fuer den externen Speicher.
	 * @return	Das Speicherformat
	 * @author	Thomas Hacklaender
	 */
	public int getStorage() {
		return storage;
	}


	/**
	 * Das Speicherformt fuer den externen Speicher wird neu definiert.
	 * @param sto	Das Speicherformat
	 * @author		Thomas Hacklaender
	 */
	public void setStorage(int sto) {
		storage = sto;
	}


	/**
	 * Die Clone-Methode (von Interface Cloneable) liefert
	 * eine Kopie des DataObjects
	 * @see #getCopyOfMe
	 */
	public Object clone() {
		return getCopyOfMe();
	}


	/**
	 * Vergleicht zwei DcmDataObjects
	 * @return TRUE, falls bei beiden Objekten alles bereinstimmt (encoding .. + grouptable)
	 */
	public boolean equals(Object o) {
		if ((o instanceof DcmDataObject) == false) {
			throw new IllegalArgumentException();
		}
		DcmDataObject e = (DcmDataObject) o;

		if (getStorage() != e.getStorage()) {
			return false;
		}
		if (getEncoding() != e.getEncoding()) {
			return false;
		}
		if (getStructure() != e.getStructure()) {
			return false;
		}
		return (grouptable.equals(grouptable));		// evtl. nicht korrekt so..
	}


	/**
	 * Die Methode liefer eine Kopie des DICOM-Daten-Objektes zurueck.
	 * @return Eine Kopie des durch die aktuelle Instanz dieser Klasse repraesentierten
	 * DICOM-Daten-Objektes.
	 */
	public DcmDataObject getCopyOfMe() {

		// newSDDO nimmt das neue DcmDataObject auf
		DcmDataObject newSDDO = new DcmDataObject(encoding, structure, storage);

		// saveSDDO rettet das aktuelle DcmDataObject waehrend des leeren des Stacks
		DcmDataObject saveSDDO = new DcmDataObject();

		// Nimmt waehrend des Umkopierens des Stacks ein einzelnes DICOM-Element auf
		DcmValue			dcmval;

		// Die Elemente des DcmDataObjects werden solange vom Stapel genommen,
		// bis keines mehr vorhanden ist. Die Heruntergenommenen Elemente werden
		// gleichzeitig auf zwei neue Stapel gelegt, damit nicht nur umgeschichtet,
		// sondern wirklich kopiert wird
		while ((dcmval = pop()) != null) {
			newSDDO.push(dcmval);
			saveSDDO.push(dcmval);
		}

		// Die Elemente des Sicherungs-Stapels werden nun wieder auf den Originalstapel
		// umgelagert
		while ((dcmval = saveSDDO.pop()) != null) {
			push(dcmval);
		}

		// Das neue DcmDataObject wird zurueckgeliefert
		return newSDDO;
	}


	/**
	 * Die Methode liefer eine Kopie des DICOM-Daten-Objektes ohne die
	 * Pixeldaten (Gruppe 0x7FE0) zurueck.
	 * @return Eine Kopie des durch die aktuelle Instanz dieser Klasse repraesentierten
	 * DICOM-Daten-Objektes.
	 */
	public DcmDataObject getHeaderCopyOfMe() {

		// newSDDO nimmt das neue DcmDataObject auf
		DcmDataObject newSDDO = new DcmDataObject(encoding, structure, storage);

		// saveSDDO rettet das aktuelle DcmDataObject waehrend des leeren des Stacks
		DcmDataObject saveSDDO = new DcmDataObject();

		// Nimmt waehrend des Umkopierens des Stacks ein einzelnes DICOM-Element auf
		DcmValue			dcmval;

		// Die Elemente des DcmDataObjects werden solange vom Stapel genommen,
		// bis keines mehr vorhanden ist. Die Heruntergenommenen Elemente werden
		// gleichzeitig auf zwei neue Stapel gelegt, damit nicht nur umgeschichtet,
		// sondern wirklich kopiert wird. Die Elemente der Gruppe 0x7FE0 (Pixel Daten)
		// werden nicht kopiert.
		while ((dcmval = pop()) != null) {
			saveSDDO.push(dcmval);

			if (dcmval.getGroup() != 0x7FE0) {
				newSDDO.push(dcmval);
			}
		}

		// Die Elemente des Sicherungs-Stapels werden nun wieder auf den Originalstapel
		// umgelagert
		while ((dcmval = saveSDDO.pop()) != null) {
			push(dcmval);
		}

		// Das neue DcmDataObject wird zurueckgeliefert
		return newSDDO;
	}


	/**
	 * Die Methode legt ein neues DICOM-Element auf den Stapel, der durch dieses
	 * DcmDataObject repraesentiert wird.
	 * @param dcmval Das auf den Stapel zu legende DcmValue Objekt
	 */
	public void push(DcmValue dcmval) {
		DcmValue	oldDcmValue;

		if (dcmval != null) {
			if (dcmval.getElement() != 0x0000) {

				// Gruppen- und Elementcode des Dicom-Objekts besorgen
				Integer			group = new Integer(dcmval.getGroup());
				Integer			element = new Integer(dcmval.getElement());

				// Besorgen der Gruppe, in die das Element gehoert, aus der Hashtabelle
				DcmElements dcmElements;

				dcmElements = (DcmElements) grouptable.get(group);

				// Wenn die Gruppe noch nicht existiert, erzeuge einene neuen
				// Hashtabelleneintrag und fuege diesen der Hashtabelle zur Gruppenverwaltung
				// hinzu
				if (dcmElements == null) {
					dcmElements = new DcmElements();
					grouptable.put(group, dcmElements);
				}

				// Schaue nun nach, ob schon ein Eintrag mit dem einzufuegenden
				// Element existiert
				oldDcmValue = (DcmValue) dcmElements.get(element);

				// ... wenn das der Fall ist, schaue nach, ob die Date eine gerade
				// Laenge haben. Ist das nicht der Fasll, schneide das letzte Zeichen ab.
				// (Das letzte Zeichen ist dann laut Dicom-Standard nur Fuellzeichen
				if (oldDcmValue != null) {
					dcmElements.subLength(oldDcmValue, structure);
				}

				// Fuege das neue Element zu seiner Grppenhashtabelle hinzu
				dcmElements.put(element, dcmval);
				dcmElements.addLength(dcmval, structure);

			}
		}
	}


	/**
	 * Die Methode liefert das oberste DICOM-Element vom Stapel zurueck.
	 * @return Das DcmValue Objekt des obersten Elementes vom Stapel
	 */
	public DcmValue pop() {
		DcmElements myGroup;

		// Hilfvariable zum Finden der kleinsten Gruppe bzw. des kleinsten Elements
		int					hlp;

		// kleinster Gruppen- und Elementwert wird zunaechst auf einen groesseren Wert
		// gesetzt, als der groesstmoegliche
		int					mingroup = 0x10000;
		int					minelement = 0x10000;

		// Wenn die Hashtabelle mit den Gruppen leer ist, liefer null zurueck
		if (grouptable.size() == 0) {
			popNewGroup = true;
			return null;
		}

		// ... ansonsten gehe die Hashtabelle durch und suche die Gruppe mit dem
		// kleinsten Wert
		for (Enumeration e = grouptable.keys(); e.hasMoreElements(); ) {
			hlp = ((Integer) e.nextElement()).intValue();
			if (hlp < mingroup) {
				mingroup = hlp;
			}
		}

		// Besorgen der Gruppe mit kleinstem Wert aus der Hashtabelle
		myGroup = (DcmElements) grouptable.get(new Integer(mingroup));

		// Wenn eine neue Gruppe gelesen wurde, wird ein neues DcmValue-Objekt
		// mit dieser Gruppe erzeugt und zurueckgeliefert
		if (popNewGroup == true) {
			popNewGroup = false;

			// tha	2000.1.29
			// return (new DcmValue(mingroup, 0x0000, myGroup.length, 4));
			byte[]	buf = new byte[4];

			DcmValue.ulToBuf(myGroup.getLength(), buf, 0);
			return (new DcmValue(mingroup, 0x0000, buf));
		}

		// ... ansonsten wird das Element mit kleinstem Wert in der Gruppe gesucht
		// und zurueckgeliefert
		for (Enumeration e = myGroup.keys(); e.hasMoreElements(); ) {
			hlp = ((Integer) e.nextElement()).intValue();
			if (hlp < minelement) {
				minelement = hlp;
			}
		}

		// Besorgen des Elements mit kleinstem Wert
		DcmValue	myElement = (DcmValue) myGroup.get(new Integer(minelement));

		// Entfernen des kleinsten Elementes aus der Gruppe
		myGroup.remove(new Integer(minelement));

		// Und wenn die Gruppe nun auch leer ist, entferne auch die Gruppe
		if (myGroup.size() == 0) {
			grouptable.remove(new Integer(mingroup));
			popNewGroup = true;
		}

		// Zurueckliefern des Elementes
		return myElement;
	}


	/**
	 * Die Methode prueft, ob ein DICOM-Elements verfuegbar ist.
	 * @param type  Der Type des DICOM-Elements.
	 * @return      true, falls das Objekt verfuegbar ist, sonst false.
	 */
	public boolean isAvailable(int type) {
		return isAvailable(DcmDDE.getDictGroup(type), DcmDDE.getDictElement(type));
	}		// Methode isAvailable


	/**
	 * Die Methode prueft, ob ein DICOM-Elements verfuegbar ist.
	 * @param group   Die Gruppennummer des DICOM-Elements.
	 * @param element Die Elementnummer des DICOM-Elements.
	 * @return        true, falls das Objekt verfuegbar ist, sonst false.
	 */
	public boolean isAvailable(int group, int element) {
		if (getDcmValue(group, element) != null) {
			return true;
		} else {
			return false;
		}
	}		// Methode isAvailable


	/**
	 * Die Methode prueft, ob ein DICOM-Elements einen Wert enthaelt.
	 * @param type  Der Type des DICOM-Elements.
	 * @return      true, falls das Objekt nicht leer ist, sonst false.
	 */
	public boolean hasValue(int type) {
		return hasValue(DcmDDE.getDictGroup(type), DcmDDE.getDictElement(type));
	}		// Methode hasValue


	/**
	 * Die Methode prueft, ob ein DICOM-Elements einen Wert enthaelt.
	 * @param group   Die Gruppennummer des DICOM-Elements.
	 * @param element Die Elementnummer des DICOM-Elements.
	 * @return        true, falls das Objekt nicht leer ist, sonst false.
	 */
	public boolean hasValue(int group, int element) {
		if (!isAvailable(group, element)) {
			return false;
		}

		DcmValue	dcmval = getDcmValue(group, element);
    int length = dcmval.getData().length;

		if (dcmval != null) {
      switch (dcmval.getVR()) {
        case DcmDDE.VRCODE_SS:
        case DcmDDE.VRCODE_US:
          if ((length >= 2) & ((length % 2) == 0)) return true;
          break;

        case DcmDDE.VRCODE_FL:
        case DcmDDE.VRCODE_SL:
        case DcmDDE.VRCODE_UL:
          if ((length >= 4) & ((length % 4) == 0)) return true;
          break;

        case DcmDDE.VRCODE_FD:
          if ((length >= 8) & ((length % 8) == 0)) return true;
          break;

        default:
          if (length > 0) return true;
      }
		}

		return false;
	}


	/**
	 * Die Methode liefert die Value Multiplicity des Elementes.
	 * @param type    Der Typ des DICOM-Elements.
	 * @param element Die Elementnummer des DICOM-Elements.
	 * @return        Die Multiplicity.
	 */
	public int getMultiplicity(int type) {
		return getMultiplicity(DcmDDE.getDictGroup(type), DcmDDE.getDictElement(type));
  }


	/**
	 * Die Methode liefert die Value Multiplicity des Elementes.
	 * @param group   Die Gruppennummer des DICOM-Elements.
	 * @param element Die Elementnummer des DICOM-Elements.
	 * @return        Die Multiplicity.
	 */
	public int getMultiplicity(int group, int element) {
		if (!isAvailable(group, element)) {
			return 0;
		}

		DcmValue	dcmval = getDcmValue(group, element);
    int length = dcmval.getData().length;

		if (dcmval != null) {
      switch (dcmval.getVR()) {
        case DcmDDE.VRCODE_SS:
        case DcmDDE.VRCODE_US:
          if ((length % 2) != 0) return 0;
          return length / 2;

        case DcmDDE.VRCODE_FL:
        case DcmDDE.VRCODE_SL:
        case DcmDDE.VRCODE_UL:
          if ((length % 4) != 0) return 0;
          return length / 4;

        case DcmDDE.VRCODE_FD:
          if ((length % 8) != 0) return 0;
          return length / 8;

        default:
          int n = 1;
          for (int i = 0; i < length; i++) {
            if (dcmval.getData()[i] == '\\') n++;
          }
      }
		}

		return 0;
	}


	/**
	 * Die Methode liefert das DICOM Data-Element als DcmValue Objekt zurueck.
	 * @param type  Der Type des DICOM-Elements.
	 * @result		  Das DcmValue Objekt.
	 */
	public DcmValue getDcmValue(int type) {
		return getDcmValue(DcmDDE.getDictGroup(type), DcmDDE.getDictElement(type));
	}


	/**
	 * Die Methode liefert das DICOM Data-Element als DcmValue Objekt zurueck.
	 * @param group   Die Gruppennummer des DICOM-Elements.
	 * @param element Die Elementnummer des DICOM-Elements.
	 * @result		 		Das DcmValue Objekt.
	 */
	public DcmValue getDcmValue(int group, int element) {
		DcmElements dcmElems;

		// Besorgen der Hashtabelle mit den Elementen der uebergenen Gruppe
		dcmElems = (DcmElements) grouptable.get(new Integer(group));

		if (dcmElems != null) {
			if (element == 0x000) {

				// tha	2000.1.29
				// return (new DcmValue(group, element, dcmElems.length, 4));
				byte[]	buf = new byte[4];

				DcmValue.ulToBuf(dcmElems.getLength(), buf, 0);
				return (new DcmValue(group, element, buf));
			}

			// Auslesen des Dicom-Elements aud der Hashtabelle
			return (DcmValue) dcmElems.get(new Integer(element));
		} else {
			return (null);
		}
	}


	/**
	 * Die Methode liefert den Wert eines beliebigen DICOM-Elements zurueck.
	 * @param type  Der Type des DICOM-Elements.
	 * @return 			Die Daten, die das DICOM-Element enthaelt.
	 * Byte Array der Laenge 0, wenn das Element nicht existiert.
	 */
	public byte[] getData(int type) {
		return getData(DcmDDE.getDictGroup(type), DcmDDE.getDictElement(type));
	}


	/**
	 * Die Methode liefert den Wert eines beliebigen DICOM-Elements zurueck.
	 * @param group   Die Gruppennummer des DICOM-Elements.
	 * @param element Die Elementnummer des DICOM-Elements.
	 * @return        Die Daten, die das DICOM-Element enthaelt.
	 * Byte Array der Laenge 0, wenn das Element nicht existiert.
	 */
	public byte[] getData(int group, int element) {
		DcmValue	dcmval = getDcmValue(group, element);

		if (dcmval != null) {
			return dcmval.getData();
		} else {
			return new byte[0];
		}
	}


	/**
	 * Die Methode loescht ein Daten Element aus dem DcmDataObject.
	 * @param type  Der Type des DICOM-Elements.
	 */
	public void removeObject(int type) {
		removeObject(DcmDDE.getDictGroup(type), DcmDDE.getDictElement(type));
	}


	/**
	 * Die Methode loescht ein Daten Element aus dem DcmDataObject.
	 * @param group   Die Gruppennummer des DICOM-Elements.
	 * @param element Die Elementnummer des DICOM-Elements.
	 */
	public void removeObject(int group, int element) {
		DcmElements dcmElems = (DcmElements) grouptable.get(new Integer(group));

		dcmElems.remove(new Integer(element));
	}


	/**
	 * Liefert die String Darstellung des DICOM-Elements zurueck.
	 * @param type  Der Type des DICOM-Elements.
	 * @return      Die String Darstellung des DICOM-Elements.
	 * "", wenn das gesuchte Element nicht vorhanden ist.
	 */
	public String getString(int type) {
		return getString(DcmDDE.getDictGroup(type), DcmDDE.getDictElement(type));
	}


	/**
	 * Liefert die String Darstellung des DICOM-Elements zurueck.
	 * @param group   Die Gruppennummer des DICOM-Elements.
	 * @param element Die Elementnummer des DICOM-Elements.
	 * @return        Die String Darstellung des DICOM-Elements.
	 * "", wenn das gesuchte Element nicht vorhanden ist.
	 */
	public String getString(int group, int element) {
		String		result = "";

		DcmValue	dcmval = getDcmValue(group, element);

		if (dcmval != null) {

			// Auslesen der Daten aus dem DICOM-Element
			byte[]	data;

			data = dcmval.getData();

			if (data.length > 0) {

				// Wenn die Daten gerade Laenge haben, wird das letzte Zeichen der
				// Daten mit einem Fuellzeichen gefuellt (Dicom-standard)
				if ((data.length % 2) == 0) {
					int last = data.length - 1;

					if ((data[last] == 0) || (data[last] == ' ')) {
						result = new String(data, 0, last);
					} else {
						result = new String(data);
					}
				} else {
					result = new String(data);
				}
			}
		}

		// Das Datenelement wird zurueckgeliefert
		return result;
	}


	/**
	 * Liefert die String Darstellung des DICOM-Elements zurueck.
	 * Diese Variante prft die Multiplicity und liefert ggf. mehrere Elemente als Array zurck.
	 * @param typ     Der Typ des DICOM-Elements.
	 * @return        Die String Darstellung der DICOM-Elemente, null, wenn das gesuchte Element nicht vorhanden ist.
	 */
	public String[] getStringArray(int type) {
		return getStringArray(DcmDDE.getDictGroup(type), DcmDDE.getDictElement(type));
	}


	/**
	 * Liefert die String Darstellung des DICOM-Elements zurueck.
	 * Diese Variante prft die Multiplicity und liefert ggf. mehrere Elemente als Array zurck.
	 * @param group   Die Gruppennummer des DICOM-Elements.
	 * @param element Die Elementnummer des DICOM-Elements.
	 * @return        Die String Darstellung der DICOM-Elemente, null, wenn das gesuchte Element nicht vorhanden ist.
	 */
	public String[] getStringArray(int group, int element) {
		String	s = this.getString(group, element);

		if (s == null) {
			return null;
		}
		return DcmValue.str2StringArray(s, "\\");
	}


	/**
	 * Liefert die signed Integer Darstellung (16 Bit) des DICOM-Elements zurueck.
	 * @param type  Der Type des DICOM-Elements.
	 * @return      Die Ganzzahlendarstellung DICOM-Elements.
	 * 0, wenn das gesuchte Element nicht vorhanden ist.
	 */
	public short getSS(int type) {
		return getSS(DcmDDE.getDictGroup(type), DcmDDE.getDictElement(type));
	}


	/**
	 * Liefert die signed Integer Darstellung (16 Bit) des DICOM-Elements zurueck.
	 * @param group   Die Gruppennummer des DICOM-Elements.
	 * @param element Die Elementnummer des DICOM-Elements.
	 * @return        Die Ganzzahlendarstellung DICOM-Elements.
	 * 0, wenn das gesuchte Element nicht vorhanden ist.
	 */
	public short getSS(int group, int element) {
		short			result = 0;

		DcmValue	dcmval = getDcmValue(group, element);

		if (dcmval != null) {
			byte[]	data = dcmval.getData();

			if (data.length >= 2) {
				return DcmValue.bufToSS(data, 0);
			}
		}
		return result;
	}


	/**
	 * Liefert die signed Integer Darstellung (16 Bit) des DICOM-Elements zurueck.
	 * Diese Variante prft die Multiplicity und liefert ggf. mehrere Elemente als Array zurck.
	 * @param type    Der Type des DICOM-Elements.
	 * @return        Die Ganzzahlendarstellung der DICOM-Elemente, null, wenn das Element nicht vorhanden ist.
	 */
	public short[] getSSArray(int type) {
		return getSSArray(DcmDDE.getDictGroup(type), DcmDDE.getDictElement(type));
	}


	/**
	 * Liefert die signed Integer Darstellung (16 Bit) des DICOM-Elements zurueck.
	 * Diese Variante prft die Multiplicity und liefert ggf. mehrere Elemente als Array zurck.
	 * @param group   Die Gruppennummer des DICOM-Elements.
	 * @param element Die Elementnummer des DICOM-Elements.
	 * @return        Die Ganzzahlendarstellung der DICOM-Elemente, null, wenn das Element nicht vorhanden ist.
	 */
	public short[] getSSArray(int group, int element) {
		DcmValue	dcmval = getDcmValue(group, element);

		if (dcmval != null) {
			byte[]	data = dcmval.getData();

      if ((data.length < 2) | ((data.length % 2) != 0)) return null;

      short[] values = new short[data.length / 2];
      int i = 0;
      for (int offset = 0; offset < data.length; offset += 2) {
        values[i++] = DcmValue.bufToSS(data, offset);
      }
      return values;
    }

		return null;
	}


	/**
	 * Liefert die signed Integer Darstellung (32 Bit) des DICOM-Elements zurueck.
	 * @param type  Der Type des DICOM-Elements.
	 * @return      Die Ganzzahlendarstellung DICOM-Elements.
	 * 0, wenn das gesuchte Element nicht vorhanden ist.
	 */
	public int getSL(int type) {
		return getSL(DcmDDE.getDictGroup(type), DcmDDE.getDictElement(type));
	}


	/**
	 * Liefert die signed Integer Darstellung (16 Bit) des DICOM-Elements zurueck.
	 * @param group   Die Gruppennummer des DICOM-Elements.
	 * @param element Die Elementnummer des DICOM-Elements.
	 * @return        Die Ganzzahlendarstellung DICOM-Elements.
	 * 0, wenn das gesuchte Element nicht vorhanden ist.
	 */
	public int getSL(int group, int element) {
		int				result = 0;

		DcmValue	dcmval = getDcmValue(group, element);

		if (dcmval != null) {
			byte[]	data = dcmval.getData();

			if (data.length >= 4) {
				return DcmValue.bufToSL(data, 0);
			}
		}
		return result;
	}


	/**
	 * Liefert die signed Integer Darstellung (16 Bit) des DICOM-Elements zurueck.
	 * Diese Variante prft die Multiplicity und liefert ggf. mehrere Elemente als Array zurck.
	 * @param typ     Der Typ des DICOM-Elements.
	 * @return        Die Ganzzahlendarstellung der DICOM-Elemente, null, wenn das gesuchte Element nicht vorhanden ist.
	 */
	public int[] getSLArray(int type) {
		return getSLArray(DcmDDE.getDictGroup(type), DcmDDE.getDictElement(type));
	}


	/**
	 * Liefert die signed Integer Darstellung (16 Bit) des DICOM-Elements zurueck.
	 * Diese Variante prft die Multiplicity und liefert ggf. mehrere Elemente als Array zurck.
	 * @param group   Die Gruppennummer des DICOM-Elements.
	 * @param element Die Elementnummer des DICOM-Elements.
	 * @return        Die Ganzzahlendarstellung der DICOM-Elemente, null, wenn das gesuchte Element nicht vorhanden ist.
	 */
	public int[] getSLArray(int group, int element) {
		DcmValue	dcmval = getDcmValue(group, element);

		if (dcmval != null) {
			byte[]	data = dcmval.getData();

      if ((data.length < 4) | ((data.length % 4) != 0)) return null;

      int[] values = new int[data.length / 4];
      int i = 0;
      for (int offset = 0; offset < data.length; offset += 4) {
        values[i++] = DcmValue.bufToSL(data, offset);
      }
      return values;
		}

		return null;
	}


	/**
	 * Liefert die unsigned Integer Darstellung (16 Bit) des DICOM-Elements zurueck.
	 * @param type  Der Type des DICOM-Elements.
	 * @return      Die Ganzzahlendarstellung DICOM-Elements.
	 * 0, wenn das gesuchte Element nicht vorhanden ist.
	 */
	public int getUS(int type) {
		return getUS(DcmDDE.getDictGroup(type), DcmDDE.getDictElement(type));
	}


	/**
	 * Liefert die unsigned Integer Darstellung (16 Bit) des DICOM-Elements zurueck.
	 * @param group   Die Gruppennummer des DICOM-Elements.
	 * @param element Die Elementnummer des DICOM-Elements.
	 * @return        Die Ganzzahlendarstellung DICOM-Elements.
	 * 0, wenn das gesuchte Element nicht vorhanden ist.
	 */
	public int getUS(int group, int element) {
		int				result = 0;

		DcmValue	dcmval = getDcmValue(group, element);

		if (dcmval != null) {
			byte[]	data = dcmval.getData();

			if (data.length >= 2) {
				return DcmValue.bufToUS(data, 0);
			}
		}
		return result;
	}


	/**
	 * Method declaration
	 *
	 *
	 * @param type
	 *
	 * @return
	 *
	 * @see
	 */
	public int[] getUSArray(int type) {
		return getUSArray(DcmDDE.getDictGroup(type), DcmDDE.getDictElement(type));
	}


	/**
	 * Liefert die unsigned Integer Darstellung (16 Bit) des DICOM-Elements zurueck.
	 * Diese Variante prft die Multiplicity und liefert ggf. mehrere Elemente als Array zurck.
	 * @param group   Die Gruppennummer des DICOM-Elements.
	 * @param element Die Elementnummer des DICOM-Elements.
	 * @return        Die Ganzzahlendarstellung DICOM-Elements, null, wenn das gesuchte Element nicht vorhanden ist.
	 */
	public int[] getUSArray(int group, int element) {
		DcmValue	dcmval = getDcmValue(group, element);

		if (dcmval != null) {
			byte[]	data = dcmval.getData();

      if ((data.length < 2) | ((data.length % 2) != 0)) return null;

      int[] values = new int[data.length / 2];
      int i = 0;
      for (int offset = 0; offset < data.length; offset += 2) {
        values[i++] = DcmValue.bufToUS(data, offset);
      }
      return values;
		}

		return null;
	}


	/**
	 * Liefert die unsigned Integer Darstellung (32 Bit) des DICOM-Elements zurueck.
	 * @param type  Der Type des DICOM-Elements.
	 * @return      Die Ganzzahlendarstellung DICOM-Elements.
	 * 0, wenn das gesuchte Element nicht vorhanden ist.
	 */
	public long getUL(int type) {
		return getUL(DcmDDE.getDictGroup(type), DcmDDE.getDictElement(type));
	}


	/**
	 * Liefert die unsigned Integer Darstellung (32 Bit) des DICOM-Elements zurueck.
	 * @param group   Die Gruppennummer des DICOM-Elements.
	 * @param element Die Elementnummer des DICOM-Elements.
	 * @return        Die Ganzzahlendarstellung DICOM-Elements.
	 * 0, wenn das gesuchte Element nicht vorhanden ist.
	 */
	public long getUL(int group, int element) {
		long			result = 0;

		DcmValue	dcmval = getDcmValue(group, element);

		if (dcmval != null) {
			byte[]	data = dcmval.getData();

			if (data.length >= 4) {
				return DcmValue.bufToUL(data, 0);
			}
		}
		return result;
	}


	/**
	 * Liefert die unsigned Integer Darstellung (32 Bit) des DICOM-Elements zurueck.
	 * Diese Variante prft die Multiplicity und liefert ggf. mehrere Elemente als Array zurck.
	 * @param typ     Der Typ des DICOM-Elements.
	 * @return        Die Ganzzahlendarstellung der DICOM-Elemente, null, wenn das gesuchte Element nicht vorhanden ist.
	 */
	public long[] getULArray(int type) {
		return getULArray(DcmDDE.getDictGroup(type), DcmDDE.getDictElement(type));
	}


	/**
	 * Liefert die unsigned Integer Darstellung (32 Bit) des DICOM-Elements zurueck.
	 * Diese Variante prft die Multiplicity und liefert ggf. mehrere Elemente als Array zurck.
	 * @param group   Die Gruppennummer des DICOM-Elements.
	 * @param element Die Elementnummer des DICOM-Elements.
	 * @return        Die Ganzzahlendarstellung der DICOM-Elemente, null, wenn das gesuchte Element nicht vorhanden ist.
	 */
	public long[] getULArray(int group, int element) {
		DcmValue	dcmval = getDcmValue(group, element);

		if (dcmval != null) {
			byte[]	data = dcmval.getData();

      if ((data.length < 4) | ((data.length % 4) != 0)) return null;

      long[] values = new long[data.length / 4];
      int i = 0;
      for (int offset = 0; offset < data.length; offset += 4) {
        values[i++] = DcmValue.bufToUL(data, offset);
      }
      return values;
		}

		return null;
	}


	/**
	 * Liefert die float Darstellung (32 Bit) des DICOM-Elements zurueck.
	 * @param type  Der Type des DICOM-Elements.
	 * @return      Die Floating-Point Darstellung des DICOM-Elements.
	 * 0.0, wenn das gesuchte Element nicht vorhanden ist.
	 */
	public float getFL(int type) {
		return getFL(DcmDDE.getDictGroup(type), DcmDDE.getDictElement(type));
	}


	/**
	 * Liefert die float Darstellung (32 Bit) des DICOM-Elements zurueck.
	 * @param group   Die Gruppennummer des DICOM-Elements.
	 * @param element Die Elementnummer des DICOM-Elements.
	 * @return        Die Floating-Point Darstellung des DICOM-Elements.
	 * 0.0, wenn das gesuchte Element nicht vorhanden ist.
	 */
	public float getFL(int group, int element) {
		float			result = (float) 0.0;

		DcmValue	dcmval = getDcmValue(group, element);

		if (dcmval != null) {
			byte[]	data = dcmval.getData();

			if (data.length >= 4) {
				return DcmValue.bufToFL(data, 0);
			}
		}
		return result;
	}


	/**
	 * Liefert die float Darstellung (32 Bit) des DICOM-Elements zurueck.
	 * Diese Variante prft die Multiplicity und liefert ggf. mehrere Elemente als Array zurck.
	 * @param typ     Der Typ des DICOM-Elements.
	 * @return        Die Floating-Point Darstellung der DICOM-Elemente, null, wenn das gesuchte Element nicht vorhanden ist.
	 */
	public float[] getFLArray(int type) {
		return getFLArray(DcmDDE.getDictGroup(type), DcmDDE.getDictElement(type));
	}


	/**
	 * Liefert die float Darstellung (32 Bit) des DICOM-Elements zurueck.
	 * Diese Variante prft die Multiplicity und liefert ggf. mehrere Elemente als Array zurck.
	 * @param group   Die Gruppennummer des DICOM-Elements.
	 * @param element Die Elementnummer des DICOM-Elements.
	 * @return        Die Floating-Point Darstellung der DICOM-Elemente, null, wenn das gesuchte Element nicht vorhanden ist.
	 */
	public float[] getFLArray(int group, int element) {
		DcmValue	dcmval = getDcmValue(group, element);

		if (dcmval != null) {
			byte[]	data = dcmval.getData();

      if ((data.length < 4) | ((data.length % 4) != 0)) return null;

      float[] values = new float[data.length / 4];
      int i = 0;
      for (int offset = 0; offset < data.length; offset += 4) {
        values[i++] = DcmValue.bufToFL(data, offset);
      }
      return values;
		}

		return null;
	}


	/**
	 * Liefert die double Darstellung (64 Bit) des DICOM-Elements zurueck.
	 * @param type  Der Type des DICOM-Elements.
	 * @return      Die Floating-Point Darstellung des DICOM-Elements.
	 * 0.0, wenn das gesuchte Element nicht vorhanden ist.
	 */
	public double getFD(int type) {
		return getFD(DcmDDE.getDictGroup(type), DcmDDE.getDictElement(type));
	}


	/**
	 * Liefert die double Darstellung (64 Bit) des DICOM-Elements zurueck.
	 * @param group   Die Gruppennummer des DICOM-Elements.
	 * @param element Die Elementnummer des DICOM-Elements.
	 * @return        Die Floating-Point Darstellung des DICOM-Elements.
	 * 0.0, wenn das gesuchte Element nicht vorhanden ist.
	 */
	public double getFD(int group, int element) {
		double		result = 0.0;

		DcmValue	dcmval = getDcmValue(group, element);

		if (dcmval != null) {
			byte[]	data = dcmval.getData();

			if (data.length >= 8) {
				return DcmValue.bufToFD(data, 0);
			}
		}
		return result;
	}


	/**
	 * Liefert die double Darstellung (64 Bit) des DICOM-Elements zurueck.
	 * @param typ     Der Typ des DICOM-Elements.
	 * @return        Die Floating-Point Darstellung des DICOM-Elements.
	 *                null, wenn das gesuchte Element nicht vorhanden ist.
	 */
	public double[] getFDArray(int type) {
		return getFDArray(DcmDDE.getDictGroup(type), DcmDDE.getDictElement(type));
	}


	/**
	 * Liefert die double Darstellung (64 Bit) des DICOM-Elements zurueck.
	 * @param group   Die Gruppennummer des DICOM-Elements.
	 * @param element Die Elementnummer des DICOM-Elements.
	 * @return        Die Floating-Point Darstellung des DICOM-Elements.
	 *                null, wenn das gesuchte Element nicht vorhanden ist.
	 */
	public double[] getFDArray(int group, int element) {
		DcmValue	dcmval = getDcmValue(group, element);

		if (dcmval != null) {
			byte[]	data = dcmval.getData();

      if ((data.length < 8) | ((data.length % 8) != 0)) return null;

      double[] values = new double[data.length / 8];
      int i = 0;
      for (int offset = 0; offset < data.length; offset += 8) {
        values[i++] = DcmValue.bufToFD(data, offset);
      }
      return values;
		}
		return null;
	}


	/**
	 * Erzeugt ein DICOM-Element mit Typecode String (beinhaltet alle DICOM
	 * Stringarten).
	 * @param type  Der Type des DICOM-Elements.
	 * @param str   Der String.
	 */
	public void setString(int type, String str) {
		setString(DcmDDE.getDictGroup(type), DcmDDE.getDictElement(type), str);
	}


	/**
	 * Erzeugt ein DICOM-Element mit Typecode String (beinhaltet alle DICOM
	 * Stringarten).
	 * @param group   Die Gruppennummer des DICOM-Elements.
	 * @param element Die Elementnummer des DICOM-Elements.
	 * @param str     Der String.
	 */
	public void setString(int group, int element, String str) {
		push(new DcmValue(group, element, str.getBytes()));
	}


	/**
	 * Erzeugt ein DICOM-Element mit Typecode SS.
	 * @param type  Der Type des DICOM-Elements.
	 * @param n     Der Integerwert.
	 */
	public void setSS(int type, short n) {
		setSS(DcmDDE.getDictGroup(type), DcmDDE.getDictElement(type), n);
	}


	/**
	 * Erzeugt ein DICOM-Element mit Typecode SS.
	 * @param group   Die Gruppennummer des DICOM-Elements.
	 * @param element Die Elementnummer des DICOM-Elements.
	 * @param n       Der Integerwert.
	 */
	public void setSS(int group, int element, short n) {
		byte[]	buf = new byte[2];

		DcmValue.ssToBuf(n, buf, 0);
		push(new DcmValue(group, element, buf));
	}


	/**
	 * Erzeugt ein DICOM-Element mit Typecode US.
	 * @param type  Der Type des DICOM-Elements.
	 * @param n     Der Integerwert.
	 */
	public void setUS(int type, int n) {
		setUS(DcmDDE.getDictGroup(type), DcmDDE.getDictElement(type), n);
	}


	/**
	 * Erzeugt ein DICOM-Element mit Typecode US.
	 * @param group   Die Gruppennummer des DICOM-Elements.
	 * @param element Die Elementnummer des DICOM-Elements.
	 * @param n       Der Integerwert.
	 */
	public void setUS(int group, int element, int n) {
		byte[]	buf = new byte[2];

		DcmValue.usToBuf(n, buf, 0);
		push(new DcmValue(group, element, buf));
	}


	/**
	 * Erzeugt ein DICOM-Element mit Typecode SL.
	 * @param type  Der Type des DICOM-Elements.
	 * @param n     Der Integerwert.
	 */
	public void setSL(int type, int n) {
		setSL(DcmDDE.getDictGroup(type), DcmDDE.getDictElement(type), n);
	}


	/**
	 * Erzeugt ein DICOM-Element mit Typecode SS.
	 * @param group   Die Gruppennummer des DICOM-Elements.
	 * @param element Die Elementnummer des DICOM-Elements.
	 * @param n       Der Integerwert.
	 */
	public void setSL(int group, int element, int n) {
		byte[]	buf = new byte[4];

		DcmValue.slToBuf(n, buf, 0);
		push(new DcmValue(group, element, buf));
	}


	/**
	 * Erzeugt ein DICOM-Element mit Typecode UL.
	 * @param type  Der Type des DICOM-Elements.
	 * @param n     Der Integerwert.
	 */
	public void setUL(int type, long n) {
		setUL(DcmDDE.getDictGroup(type), DcmDDE.getDictElement(type), n);
	}


	/**
	 * Erzeugt ein DICOM-Element mit Typecode UL.
	 * @param group   Die Gruppennummer des DICOM-Elements.
	 * @param element Die Elementnummer des DICOM-Elements.
	 * @param n       Der Integerwert.
	 */
	public void setUL(int group, int element, long n) {
		byte[]	buf = new byte[4];

		DcmValue.ulToBuf(n, buf, 0);
		push(new DcmValue(group, element, buf));
	}


	/**
	 * Erzeugt ein DICOM-Element mit Typecode FL.
	 * @param type  Der Type des DICOM-Elements.
	 * @param n     Der Floating-Point Wert.
	 */
	public void setFL(int type, float n) {
		setFL(DcmDDE.getDictGroup(type), DcmDDE.getDictElement(type), n);
	}


	/**
	 * Erzeugt ein DICOM-Element mit Typecode FL.
	 * @param group   Die Gruppennummer des DICOM-Elements.
	 * @param element Die Elementnummer des DICOM-Elements.
	 * @param n       Der Floating-Point Wert.
	 */
	public void setFL(int group, int element, float n) {
		byte[]	buf = new byte[4];

		DcmValue.flToBuf(n, buf, 0);
		push(new DcmValue(group, element, buf));
	}


	/**
	 * Erzeugt ein DICOM-Element mit Typecode FD.
	 * @param type  Der Type des DICOM-Elements.
	 * @param n     Der Floating-Point Wert.
	 */
	public void setFD(int type, double n) {
		setFD(DcmDDE.getDictGroup(type), DcmDDE.getDictElement(type), n);
	}


	/**
	 * Erzeugt ein DICOM-Element mit Typecode FD.
	 * @param group   Die Gruppennummer des DICOM-Elements.
	 * @param element Die Elementnummer des DICOM-Elements.
	 * @param n       Der Floating-Point Wert.
	 */
	public void setFD(int group, int element, double n) {
		byte[]	buf = new byte[8];

		DcmValue.fdToBuf(n, buf, 0);
		push(new DcmValue(group, element, buf));
	}


	/**
	 * Generiert einen String Darstellung des DcmDataObjekt's. Dabei werden
	 * folgende Formatparametern benutzt:
	 * - Anzeige des Description-Strings mit 32 Stellen
	 * - Anzeige des Tags
	 * - Anzeige des Inhaltes mit 256 Stellen
	 * - String Elemente werden ohne umschliessende Anfuehrungszeichen dargestellt.
	 */
	public String toString() {
		DcmDataObject ddo;
		DcmValue			dv;
		String				theStr = "";

		ddo = getCopyOfMe();

		while ((dv = ddo.pop()) != null) {
			theStr += dv.toString(true, 32, true, 256, false) + "\n";
		}

		return theStr;
	}


	/**
	 * Nur zum Debuggen
	 */
	public void dump(PrintStream ps) {
		Enumeration e = grouptable.elements();

		while (e.hasMoreElements()) {
			DcmElements v = (DcmElements) e.nextElement();

			Enumeration e2 = v.elements();

			while (e2.hasMoreElements()) {
				DcmValue	vv = (DcmValue) e2.nextElement();

				ps.println(vv.toString());
			}
		}
	}

	// ans
	// ab hier folgen die Filtermethoden


	/**
	 * extract() ist eine Filtermethode, die aus dem aktuellen DcmDataObject ein
	 * neues kreiert, welches nur noch Tags der geg. Typeliste enthlt. Damit
	 * lassen sich einzelne Attribute automatisch herausgreifen (z.b. alle
	 * Patienteninformationen)
	 * @param types Typecodes aller Tags, die herausgefiltert werden sollen
	 * @return neues, gefiltertes DcmDataObject
	 */
	public DcmDataObject extract(int[] types) {
		DcmDataObject localCopy = (DcmDataObject) this.clone();
		DcmDataObject res = new DcmDataObject(this.getEncoding(), this.getStructure(), this.getStorage());

		DcmValue			dcmval;

		while ((dcmval = localCopy.pop()) != null) {
			boolean bOk = false;
			int			t = dcmval.getType();

			for (int i = 0; i < types.length; i++) {
				if (types[i] == t) {
					bOk = true;
				}
			}

			// war der typ in der typliste vorhanden ? -> im Resultat einfgen
			if (bOk) {
				res.push(dcmval);
			}
		}

		return res;
	}


	/**
	 * clear() ist eine Filtermethode, die aus dem aktuellen DcmDataObject ein
	 * neues kreiert, welches die Tags der geg. Typeliste NICHT enthlt. Damit
	 * lassen sich einzelne Attribute gezielt lschen
	 * @param types Typecodes aller Tags, die gelscht werden sollen
	 * @return neues, gefiltertes DcmDataObject
	 */
	public DcmDataObject clear(int[] types) {
		DcmDataObject localCopy = (DcmDataObject) this.clone();
		DcmDataObject res = new DcmDataObject(this.getEncoding(), this.getStructure(), this.getStorage());

		DcmValue			dcmval;

		while ((dcmval = localCopy.pop()) != null) {
			boolean bOk = true;
			int			t = dcmval.getType();

			for (int i = 0; i < types.length; i++) {
				if (types[i] == t) {
					bOk = false;
				}
			}

			// war der typ in der typliste vorhanden ? -> im Resultat einfgen
			if (bOk) {
				res.push(dcmval);
			}
		}

		return res;
	}


	/**
	 * merge() ist eine Filtermethode, die aus dem aktuellen DcmDataObject ein
	 * neues kreiert, welches die zustzlich zu den eigenen Tags die Tags der geg. Typeliste aus
	 * einem zweiten DcmDataObject integriert. Damit lassen sich zwei DcmDataObjekte "mischen"
	 * @param types Typecodes aller Tags, die gemischt werden sollen
	 * @param ddo2 zweites DcmDataObject zum Mischen
	 * @return neues, gefiltertes DcmDataObject
	 */
	public DcmDataObject merge(int[] types, DcmDataObject ddo2) {
		DcmDataObject localCopy = (DcmDataObject) this.clone();
		DcmDataObject res = new DcmDataObject(this.getEncoding(), this.getStructure(), this.getStorage());

		DcmValue			dcmval;

		while ((dcmval = localCopy.pop()) != null) {
			boolean bOk = false;
			int			t = dcmval.getType();

			for (int i = 0; i < types.length; i++) {
				if (types[i] == t) {
					bOk = true;
				}
			}

			// war der typ in der typliste vorhanden ? -> aus zweitem DDO nehmen, sonst normal bernehmen
			if (bOk) {
				DcmValue	vv2 = ddo2.getDcmValue(t);

				if (vv2 != null) {
					res.push(vv2);
				} else {	// in der zweiten Liste nicht vorhanden, dann aus der ersten nehmen
					res.push(dcmval);
				}
			} else {
				res.push(dcmval);
			}
		}

		return res;
	}

}



/*--- formatting done in "My Own Convention" style on 11-11-2000 ---*/

