/*--- formatted by Jindent 2.1, (www.c-lab.de/~jindent) ---*/

/*
 * Copyright (C) 2000 Thomas Hacklaender, e-mail: reply@thomas-hacklaender.de
 * Christian Schalla
 * Andreas Schmidt
 * Andreas Truemper
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License 2
 * as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * http://www.gnu.org/copyleft/copyleft.html
 */
package rad.dicom.dcm;

import java.io.*;
import java.net.*;


/**
 * Diese Klasse stellt Methoden zum Einlesen von DcmDataObjects aus
 * Streams zur Verfuegung. Die Klasse ist Tochterklasse von FilterInputStream.
 * Mit folgendem Code Fragment kann man eine DcmDataObject aus einem
 * File einlesen:
 * <code><pre>
 * try {
 * dis = new DcmInputStream(new BufferedInputStream(new FileInputStream(File f)));
 * DcmDataObject ddo = dis.readDDO();
 * } catch (Exception ex) {
 * ex.printStackTrace(System.err);
 * }
 * </pre></code>
 * <br><br>
 * Alternativ kann z.b. direkt von einem Webserver geladen werden
 * <code><pre>
 * URL url = new URL( "http://www.xyz.de/down/slice1.dc3");
 * DcmDataObject ddo = DataInputStream.loadDDO(url);
 * </pre></code>
 * <DL><DT><B>Modifications: </B><DD>
 * tha	2000.3.1:   Patch der Methode readData fuer Java 1.1.8.<br>
 * tha	2000.4.30:  Neue statische Methode loadDDO definiert.<br>
 * tha	2000.4.30:  Fr die statische Methode isDicomStream neue Varianten
 * definiert.<br>
 * ans  2000.10.28: loadDDO() mit URL, 1. Versuch.<br>
 * tha  2000.11.13: getMetaInfo() implementiert. Kiefert den File Meta Information
 * Block, sofern einer in dem Stream vorhanden war.<br>
 * </DD></DL>
 * @author   Thomas Hacklaender
 * @author   Andreas Schmidt
 * @version  2000.11.13
 */
public class DcmInputStream extends FilterInputStream {

	/* Der BufferedInputStream, aus dem die Daten gelesen werden. */
	private BufferedInputStream binStream = null;

	/* Zeigt an, ob das End Of File des Streams erreicht wurde */
	private boolean							isEOF = false;

	/*
	 * Gibt an, ob nur bis zu einer maximalen Gruppennummer (einschliesslich)
	 * eingelesen werden soll
	 */
	private boolean							isMaxGroupNumber = false;

	/*
	 * Die maximalen Gruppennummer bis zu der einschliesslich eingelesen
	 * werden soll
	 */
	private int									maxGroupNumber = 0x7FFFFFFF;

	/* Bytefolge des externen Speichers */
	private int									extEncoding;

	/* Struktur des externen Speichers */
	private int									extStructure;

	/* Speicherformat des externen Speichers */
	private int									extStorage;

	/* Testergebnis der Bytefolge des externen Speichers */
	private int									testEncoding;

	/* Testergebnis der Struktur des externen Speichers */
	private int									testStructure;

	/* Testergebnis der Speicherformat des externen Speichers */
	private int									testStorage;

  /* Das DcmDataObject des File Meta Information Blocks */
  private DcmDataObject       metaInfoDDO = null;


	/**
	 * Ueberprueft, ob der File der DICOM Syntax entspricht.
	 * Statische Methode. Fuer die Instanz der Klasse existiert eine
	 * identische Methode, bei der der File nicht angegeben werden muss.
	 * @param f 	Der File, der ueberprueft werden sollen.
	 * @return   	True, wenn DICOM konform, sonst false.
	 */
	public static boolean isDicomStream(File f) {
		boolean							valid = false;
		FileInputStream			fis = null;
		BufferedInputStream bis = null;

		try {
			fis = new FileInputStream(f);
			bis = new BufferedInputStream(fis);
			valid = isDicomStream(bis);
		} catch (Exception e) {
			return false;
		}
		finally {
			if (fis != null) {
				try {
					fis.close();
				} catch (Exception e) {}
			}
			bis = null;		// BufferedInputStream.close() erst ab Java 1.2 definiert
		}

		return valid;
	}


	/**
	 * Ueberprueft, ob der InputStream der DICOM Syntax entspricht.
	 * Statische Methode. Fuer die Instanz der Klasse existiert eine
	 * identische Methode, bei der der InputStream nicht angegeben werden muss.
	 * @param bis 	Der InputStream, der ueberprueft werden sollen.
	 * @return   		True, wenn DICOM konform, sonst false.
	 */
	public static boolean isDicomStream(InputStream is) {
		boolean							valid = false;
		BufferedInputStream bis = null;

		try {
			bis = new BufferedInputStream(is);
			valid = isDicomStream(bis);
		}
		finally {
			bis = null;		// BufferedInputStream.close() erst ab Java 1.2 definiert
		}

		return valid;
	}


	/**
	 * Ueberprueft, ob der BufferedInputStream der DICOM Syntax entspricht.
	 * Statische Methode. Fuer die Instanz der Klasse existiert eine
	 * identische Methode, bei der der InputStream nicht angegeben werden muss.
	 * @param bis 	Der BufferedInputStream, der ueberprueft werden sollen.
	 * @return   		True, wenn DICOM konform, sonst false.
	 */
	public static boolean isDicomStream(BufferedInputStream bis) {
		bis.mark(128 + 4);
		byte[]	buf = new byte[132];

		try {
			bis.read(buf, 0, buf.length);
			bis.reset();
		} catch (Exception ex) {
			return false;
		}

		// ans: Bei der Angabe in '...' hat jikes gemeckert. Angaben in Hochkomma sind
		// eigentlich Unicode, d.h. nicht unbedint byte. Beizeiten sollten wir das
		// folgende austesten:
		// if ((buf[128] == (byte)86) & (buf[129] == (byte)73) & (buf[130] == (byte)67) & (buf[131] == (byte)77)) {
    // tha 2000.11.12: Unicode Zeichen sind zuweisungskompatibel zu Integer.
    // Deshalb jetzt ein Caste auf Byte.

		if ((buf[128] == (byte) 'D') & (buf[129] == (byte) 'I') & (buf[130] == (byte) 'C') & (buf[131] == (byte) 'M')) {
			return true;
		}
		if ((buf[0] == 0x08) & (buf[1] == 0x00)) {
			return true;
		}
		if ((buf[0] == 0x00) & (buf[1] == 0x08)) {
			return true;
		}

		return false;
	}


	/**
	 * Liest ein DcmDataObject aus einem File. Die Transfersyntax wird
	 * automatisch ermittelt.
	 * @param f 	Der File, aus dem gelesen werden sollen.
	 * @return   	Das DcmDataObject.
	 */
	public static DcmDataObject loadDDO(File f) throws Exception {
		DcmDataObject				ddo = null;
		FileInputStream			fis = null;
		BufferedInputStream bis = null;
		DcmInputStream			dis = null;

		try {
			fis = new FileInputStream(f);
			bis = new BufferedInputStream(fis);
			dis = new DcmInputStream(bis);
			ddo = dis.readDDO();
		} catch (Exception e) {
			throw e;
		}
		finally {
			if (fis != null) {
				fis.close();
			}
			bis = null;		// BufferedInputStream.close() erst ab Java 1.2 definiert
			dis = null;
		}

		return ddo;
	}


	/**
	 * Liest ein DcmDataObject aus einer URL.
	 * @param	u   URL, z.b. per HTTP
	 * @return   	Das DcmDataObject.
	 */
	public static DcmDataObject loadDDO(URL u) throws Exception {
		URLConnection uc = u.openConnection();

		InputStream		is = uc.getInputStream();
		DcmDataObject ddo = loadDDO(is);

		is.close();

		return ddo;
	}


	/**
	 * Liest ein DcmDataObject aus einem Stream. Die Transfersyntax des input
	 * Streams wird automatisch ermittelt.
	 * @param is 	Der InputStream, aus dem gelesen werden sollen.
	 * @return   	Das DcmDataObject.
	 */
	public static DcmDataObject loadDDO(InputStream is) throws Exception {
		DcmDataObject				ddo = null;
		BufferedInputStream bis = null;
		DcmInputStream			dis = null;

		try {
			bis = new BufferedInputStream(is);
			dis = new DcmInputStream(bis);
			ddo = dis.readDDO();
		} catch (Exception e) {
			throw e;
		}
		finally {
			bis = null;		// BufferedInputStream.close() erst ab Java 1.2 definiert
			dis = null;
		}

		return ddo;
	}


	/**
	 * Erzeugt einen neuen neuen DcmInputStream fuer den BufferedInputStream in.
	 * @param bis 	Der BufferedInputStream, aus dem die Daten gelesen werden sollen.
	 */
	public DcmInputStream(BufferedInputStream bis) {
		super(bis);
		binStream = bis;
	}


	/**
	 * Dieses Methode liest ein DcmDataObject aus dem Stream. Die Transfersyntax
	 * des externen Speichermediums wird automatisch ermittelt.
	 */
	public DcmDataObject readDDO() throws Exception {
		return readDDO(DcmDataObject.AUTO_ENDIAN, DcmDataObject.AUTO_VR, DcmDataObject.AUTO_STORAGE);
	}


	/**
	 * Die Methode liefert das DcmDataObject des File Meta Information Blocks.
   * Diese Methode liefert erst dann ein von null verschiedenes Ergebnis, wenn
   * vorher mit readDDO ein DcmDataObject eingelesen wurde.
	 * @return  Das DcmDataObject. null, wenn nicht vorhanden.
	 */
  public DcmDataObject getMetaInfo() {
    return metaInfoDDO;
  }


	/**
	 * Dieses Methode liest ein DcmDataObject aus dem Stream. Dabei wird
	 * eine vorgegebene Transfersyntax verwendet.
	 * @param encoding  Der zu verwendende Encoding der Transfersyntax.
	 * @param structure Die zu verwendende Structure der Transfersyntax.
	 * @param storage   Das zu verwendende Speicherformat.
	 */
	public DcmDataObject readDDO(int encoding, int structure, int storage) throws Exception {

		// Das DicomDataObject
		DcmDataObject ddo;

		// Ein einzelnes Dicom Daten Element
		DcmValue			dcmvalue;

		// Der Gruppencode des Dicom Daten Elements
		int						group;

		// Der Elementcode des Dicom Daten Elements
		int						element;

		// Die Laenge des Dicom Daten Elements (in Bytes)
		long					length;

		// Der Type Code des Dicom Daten Elements (bei explicit VR)
		int						vr;

		// Ein allgemeiner Byte Buffer
		byte[]				buf;

		// Das Format des externen Speichermediums fuer alle Methoden der Klassen
		// bekannt machen
		extEncoding = encoding;
		extStructure = structure;
		extStorage = storage;

		// Testen der Transfer Syntax des Speichermediums
		// ans 14.2.00 nur testen , wenn ein AUTO-Wert dabei ist
		if ((extEncoding == DcmDataObject.AUTO_ENDIAN) || (extStructure == DcmDataObject.AUTO_VR) || (extStorage == DcmDataObject.AUTO_STORAGE)) {
			testTransferSyntax();

		}
		if (extEncoding == DcmDataObject.AUTO_ENDIAN) {
			extEncoding = testEncoding;
		}
		if (extStructure == DcmDataObject.AUTO_VR) {
			extStructure = testStructure;
		}
		if (extStorage == DcmDataObject.AUTO_STORAGE) {
			extStorage = testStorage;
		}

		// Im DcmDataObject die gewuenschten Werte hinterlegen
		ddo = new DcmDataObject(extEncoding, extStructure, extStorage);

		// Einlesen des Streams bis zum End Of File.
		while (true) {

			// Auslesen des Gruppen- und Elementcodes und der Datenlaenge
			group = readUInt16();

			// Soll nur bis zu einer bestimmten Grupennummer ausgelesen werden ?
			if (isMaxGroupNumber) {

				// Ist die groesste Gruppennummer ueberschritten ?
				if (group > maxGroupNumber) {
					break;
				}
			}

			// Stream koplett eingelesen ?
			if (isEOF) {
				break;

			}
			element = readUInt16();

			if (extStructure == DcmDataObject.IMPLICITE_VR) {
				vr = DcmDDE.VRCODE_UNDEFINED;
				length = readUInt32();
			} else {
				buf = new byte[2];
				readData(buf);
				vr = DcmDDE.vrStringToCode(new String(buf));
				if ((vr == DcmDDE.VRCODE_OB) || (vr == DcmDDE.VRCODE_OW) || (vr == DcmDDE.VRCODE_SQ) || (vr == DcmDDE.VRCODE_UN)) {

					// ans 14.2.00 Bei diesen VRs und expliziter Darstellung
					// folgen vor der 32-bit-Lnge 2 Bytes mit Nullen. Die
					// berspringen wir.
					byte	dat[] = new byte[2];

					readData(dat);

					// und lesen jetzt die Lnge
					length = readUInt32();
				} else {
					length = readUInt16();
				}

			}
			buf = new byte[(int) length];
			readData(buf);

			// Intern werden Daten in little Endian gespeichert
			if (extEncoding == DcmDataObject.BIG_ENDIAN) {
				DcmValue.swapBuf(buf, vr);
			}

			dcmvalue = new DcmValue(group, element, buf);
			dcmvalue.setVR(vr);

			// Einfuegen des Dicom-Daten-Elements in das Dicom-Daten-Objekt
			ddo.push(dcmvalue);
		}		// while

		// Das DcmDataObject zurueckgeben
		return ddo;
	}		// Methode RawToDcm


	/**
	 * Erlaubt das Einlesen der DICOM Elemente nur bis einschliesslich einer
	 * maximalen Gruppennummer
	 * @param	mgm		Die maximale Gruppennummer
	 */
	public void setMaxGroupNumber(int mgn) {
		isMaxGroupNumber = true;
		maxGroupNumber = mgn;
	}


	/**
	 * Erlaubt das Einlesen der DICOM Elemente bis zu jeder beliebigen
	 * Gruppennummer (entspricht der Voreinstellung der Klasse).
	 */
	public void clearMaxGroupNumber() {
		isMaxGroupNumber = false;
	}


	/**
	 * Die Methode liesst Daten aus dem Dateipuffer in das uebergebene Array.
	 * Die Anzahk der eingelesenen Bytes entspricht der Laenge des Arrays.
	 * @param dat	Das Array, in das die Daten eingelesen werden sollen..
	 */
	private void readData(byte[] dat) throws Exception {

		// tha	2000.3.1: Patch fuer Java 1.1.8
		// Der im Folgenden auskommentierte Code funktioniert in Java 1.2.x
		// problemlos.
		// if (binStream.read(dat, 0, dat.length) != dat.length) {
		// isEOF = true;
		// }
		// Unter Java 1.1.8 kommt es jedoch vor, dass die Methode binStream.read
		// weniger Bytes einliest als gewuenscht, ohne dass EOF erreicht ist.
		// Dies scheint immer dann aufzutreten, wenn der interne Buffer des
		// BufferedInputStream nicht mehr ausreicht, die notwendige Anzahl der
		// Bytes zu liefern, und die Anzahl der eizulesenden Bytes relativ
		// gross ist (wurde nur beobachtet, wenn n>200 war).
		// Der jetzt ersetzte Code lauft unter Java 1.1.8 und 1.2.x:
		int idx = 0;
		int l = 0;

		int nBytes = dat.length;

		while (nBytes > 0) {
			l = binStream.read(dat, idx, nBytes);
			if (l == -1) {
				isEOF = true;
				break;
			}
			idx += l;
			nBytes -= l;
		}
	}


	/**
	 * Die Methode liesst einen 16-Bit Ganzzahlwert (ohne Vorzeichen) aus dem
	 * Dateipuffer.
	 * @return	Der 16-Bit Ganzzahlwert (ohne Vorzeichen).
	 */
	private int readUInt16() throws Exception {

		// Einlesen von 2 Bytes
		byte	dat[] = new byte[2];

		readData(dat);

		if (extEncoding == DcmDataObject.LITTLE_ENDIAN) {
			return (int) ((((int) dat[0]) & 0xff)) | ((((int) dat[1]) & 0xff) << 8);
		} else {
			return (int) ((((int) dat[1]) & 0xff)) | ((((int) dat[0]) & 0xff) << 8);
		}
	}


	/**
	 * Die Methode liesst einen 32-Bit Ganzzahlwert (ohne Vorzeichen) aus dem
	 * Dateipuffer.
	 * @return	Der 32-Bit Ganzzahlwert (ohne Vorzeichen).
	 */
	private long readUInt32() throws Exception {

		// Einlesen von 4 Bytes
		byte	dat[] = new byte[4];

		readData(dat);

		if (extEncoding == DcmDataObject.LITTLE_ENDIAN) {
			return (long) ((((long) dat[0]) & 0xff)) | ((((long) dat[1]) & 0xff) << 8) | ((((long) dat[2]) & 0xff) << 16) | ((((long) dat[3]) & 0xff) << 24);
		} else {
			return (long) ((((long) dat[3]) & 0xff)) | ((((long) dat[2]) & 0xff) << 8) | ((((long) dat[1]) & 0xff) << 16) | ((((long) dat[0]) & 0xff) << 24);
		}
	}


	/**
	 * Ueberprueft, ob der InputStream der DICOM Syntax entspricht.
	 * @param in 	Der InputStream, der ueberprueft werden sollen.
	 * @return    True, wenn DICOM konform, sonst false.
	 */
	public boolean isDicomStream() {
		return DcmInputStream.isDicomStream(binStream);
	}


	/**
	 * Ueberprueft die Transfer-Syntax und setzt die Felder testStorage,
	 * testEncoding, testStructure entsprechend den gefundenen Werten.
	 */
	private void testTransferSyntax() throws Exception {
		byte[]	buf;

		binStream.mark(512);

		// ans 14.2.00 skip() hat hier wohl die Eigenschaft, die mark()-Position zu
		// zerstren, sodas das reset() unten nichts bringt. deshalb lesen wir hier
		// 128 byte in den dummy-Buffer.
		// binStream.skip(128);
		buf = new byte[128];
		readData(buf);

		buf = new byte[4];
		readData(buf);
		binStream.reset();

		if ((buf[0] == 'D') & (buf[1] == 'I') & (buf[2] == 'C') & (buf[3] == 'M')) {
			testMetaStorage();
			testStorage = DcmDataObject.META_STORAGE;
			return;
		}
		testStorage = DcmDataObject.PLAIN_STORAGE;

		testEncoding = DcmDataObject.LITTLE_ENDIAN;
		testStructure = DcmDataObject.IMPLICITE_VR;

		binStream.mark(512);
		buf = new byte[16];
		readData(buf);
		binStream.reset();

		// Unter der Annahme, dass die Gruppennummer des ersten Daten Elements
		// <= 0x00FF ist (normalerweise ist sie fuer Bilddateien 0x0008)
		if ((buf[0] == 0) & (buf[1] != 0)) {
			testEncoding = DcmDataObject.BIG_ENDIAN;
		}

		// Test auf explizite Structure: Daten Element vom Typ OB, OW, SQ, UN
		if ((buf[6] == 0) & (buf[7] == 0)) {
			if (((buf[4] == 'O') & (buf[5] == 'B')) | ((buf[4] == 'O') & (buf[5] == 'W')) | ((buf[4] == 'S') & (buf[5] == 'Q')) | ((buf[4] == 'U') & (buf[5] == 'N'))) {
				testStructure = DcmDataObject.EXPLICITE_VR;
				return;
			}
		}

		// Unter der Annahme, dass die Elementnummer des ersten Daten Elements
		// der Gruppe <= 0x00FF ist (normalerweise ist sie fuer Bilddateien
		// entweder 0x0000 = Group Length oder 0x0008 = Image Type)
		if ((buf[4] >= 'A') & (buf[5] >= 'A')) {
			testStructure = DcmDataObject.EXPLICITE_VR;
		}
	}


	/**
	 * Ueberprueft, ob der Stream einen Meta-File Header besitzt. Setzt das
	 * Felder testStorage, testEncoding, testStructure entsprechend den
	 * im Header gefundenen Werten.
	 */
	private void testMetaStorage() throws Exception {
		byte[]					buf;
		long						lmib;
		DcmInputStream	dis;
		int							groupLength;
		String					transferUID;

		binStream.mark(2048);

		// Berechnung der Laenge des Meta Information Blocks
		// 128 Byte File Preambel + 4 Byte DICOM Prefix + 8 Byte Header
		// tha 2000.2.20: skip durch read ersetzt.
		// binStream.skip(128+4+8);
		buf = new byte[128 + 4 + 8];
		readData(buf);

		buf = new byte[4];
		readData(buf);
		binStream.reset();

		// Die "Group Length" der Gruppe 0x0002 einlesen: little endian,
		// explicit transfer systax: (0x0002, 0x0000) hat VR = UL.
		// Das gesamte Daten Element ist 12 Byte lang.
		// lmib = Anzahl der Bytes in Gruppe 0x0002, die dem letzten Byte
		// des Elementes (0x0002, 0x0000) folgen.
		lmib = (long) ((((long) buf[0]) & 0xff)) | ((((long) buf[1]) & 0xff) << 8) | ((((long) buf[2]) & 0xff) << 16) | ((((long) buf[3]) & 0xff) << 24);

		// Laenge des Datenelementes (0x0002, 0x0000) addieren.
		lmib += 12;

		// Einlesen des Meta Information Blocks
		// 128 Byte File Preambel + 4 Byte DICOM Prefix
		// tha 2000.2.20: skip durch read ersetzt.
		// binStream.skip(128+4);
		buf = new byte[128 + 4];
		readData(buf);

		// DICOM File Meta Information in buf einlesen
		buf = new byte[(int) lmib];
		readData(buf);
		binStream.reset();

		// DICOM File Meta Information in ddo umwandeln
		dis = new DcmInputStream(new BufferedInputStream(new ByteArrayInputStream(buf)));
		metaInfoDDO = dis.readDDO(DcmDataObject.LITTLE_ENDIAN, DcmDataObject.EXPLICITE_VR, DcmDataObject.PLAIN_STORAGE);

		transferUID = metaInfoDDO.getString(DcmDDE.DD_TransferSyntaxUID);

		// Setzt die Transfersyntax auf die Werte des Meta Info Blocks
		if (transferUID.compareTo(DcmUID.LI_IM_TRANSFER_UID) == 0) {
			testEncoding = DcmDataObject.LITTLE_ENDIAN;
			testStructure = DcmDataObject.IMPLICITE_VR;
		}
		if (transferUID.compareTo(DcmUID.LI_EX_TRANSFER_UID) == 0) {
			testEncoding = DcmDataObject.LITTLE_ENDIAN;
			testStructure = DcmDataObject.EXPLICITE_VR;
		}
		if (transferUID.compareTo(DcmUID.BI_EX_TRANSFER_UID) == 0) {
			testEncoding = DcmDataObject.BIG_ENDIAN;
			testStructure = DcmDataObject.EXPLICITE_VR;
		}

		// Setze die Lesemarker auf den Beginn des eigentlichen Inhalts
		// tha 2000.2.20: skip durch read ersetzt.
		// binStream.skip(128+4+lmib);
		buf = new byte[128 + 4 + (int) lmib];
		readData(buf);

	}

}



/*--- formatting done in "My Own Convention" style on 11-11-2000 ---*/

