/*--- formatted by Jindent 2.1, (www.c-lab.de/~jindent) ---*/

/*
 * Copyright (C) 2000 Thomas Hacklaender, e-mail: reply@thomas-hacklaender.de
 * Christian Schalla
 * Andreas Schmidt
 * Andreas Truemper
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License 2
 * as published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * 
 * http://www.gnu.org/copyleft/copyleft.html
 */
package rad.dicom.dcm;

import java.io.*;


/**
 * Diese Klasse stellt Methoden zum Schreiben von DcmDataObjects in einen
 * Stream zur Verfuegung. Die Klasse ist Tochterklasse von FilterOutputStream.
 * Mit folgendem Code Fragment kann man eine DcmDataObject in einem
 * File speichern:
 * <code><pre>
 * try {
 * dos = new DcmOutputStream(new FileOutputStream(f));
 * dos.writeDDO(DcmdataObject ddo);
 * } catch (Exception ex) {
 * ex.printStackTrace(System.err);
 * }
 * </pre></code>
 * <DL><DT><B>Modifications: </B><DD>
 * tha	2000.4.30:  Neue statische Methode saveDDO definiert
 * </DD></DL>
 * @author   Thomas Hacklaender
 * @author   Andreas Schmidt
 * @version  2000.05.04
 */
public class DcmOutputStream extends FilterOutputStream {


	/**
	 * Der OutputStream, in den die Daten geschrieben werden.
	 */
	private OutputStream	outStream;


	/**
	 * Eine Kopie des der writeDDO Methode uebergebenen DcmDataObject
	 */
	private DcmDataObject copyDDO;


	/**
	 * Eine 128 Byte lange Preambel fuer DICOM Meta Files
	 */
	private byte[]				metaFilePreambel = new byte[128];


	/**
	 * Bytefolge des externen Speichers
	 */
	private int						extEncoding;


	/**
	 * Struktur des externen Speichers
	 */
	private int						extStructure;


	/**
	 * Speicherformat des externen Speichers
	 */
	private int						extStorage;


	/**
	 * Schreibt ein DcmDataObject in einen File.
	 * @param ddo Das DcmDataObject, das geschrieben werden sollen.
	 * @param f 	Der File, in den geschrieben werden sollen.
	 */
	public static void saveDDO(DcmDataObject ddo, File f) throws Exception {
		FileOutputStream	fos = null;
		DcmOutputStream		dos = null;

		try {
			fos = new FileOutputStream(f);
			dos = new DcmOutputStream(fos);
			dos.writeDDO(ddo);
		} catch (Exception e) {
			throw e;
		} 
		finally {
			if (fos != null) {
				fos.close();
			} 
			dos = null;
		} 

	} 


	/**
	 * Erzeugt einen neuen neuen DcmOutputStream fuer den OutputStream out.
	 * @param out 	Der OutputStream, in den die Daten geschrieben werden sollen.
	 */
	public DcmOutputStream(OutputStream out) {
		super(out);
		outStream = out;
	}


	/**
	 * Dieses Methode schreibt ein DcmDataObject in den Stream. Dabei wird
	 * die im DcmDataObject vorgegebene Transfersyntax verwendet.
	 * @param ddo  Das DcmDataObject.
	 */
	public void writeDDO(DcmDataObject ddo) throws Exception {

		// Ein einzelnes Dicom-Daten-Element
		DcmValue	dcmval;

		// True, falls die Datenlaenge ungerade ist
		boolean		padding = false;

		// Die Laenge des Dicom-Daten-Elements in Bytes
		int				length;

		// Type Code des Dicom-Daten-Elements in Bytes
		int				vr;

		// Ein allgemeiner Byte Buffer
		byte[]		buf;


		// Erzeuge eine Koie des DcmDataObjects, da es durch das Auslesen
		// zerstoert wird.
		copyDDO = ddo.getCopyOfMe();
		extEncoding = copyDDO.getEncoding();
		extStructure = copyDDO.getStructure();
		extStorage = copyDDO.getStorage();

		// Wenn gewuenscht, Meta File generieren
		if (extStorage == DcmDataObject.META_STORAGE) {
			writeFileMetaInfo();
		} 

		// Auslesen des obersten Elements vom Dicom-Daten-Objekt, solange dieses
		// nicht leer ist.
		while ((dcmval = copyDDO.pop()) != null) {

			vr = dcmval.getVR();

			// Besorgen der Laenge des Datenbereichs
			length = dcmval.getData().length;

			// Wenn die Laenge eine ungerade anzahl von Bytes hat, wird der Datenbereich
			// durch ein Fuellzeichen ergaenzt (so steht es im Dicom-Standard)
			if ((length % 2) != 0) {
				padding = true;
				length++;
			} else {
				padding = false;
			} 

			// Schreiben des Gruppen- und Elementcodes in den Puffer
			writeUInt16(dcmval.getGroup());
			writeUInt16(dcmval.getElement());

			if (extStructure == DcmDataObject.IMPLICITE_VR) {

				// Datenlaenge schreiben
				writeUInt32(length);
			} else {

				// EXPLICITE_VR
				String	vrn = DcmDDE.vrCodeToString(vr);

				// VR schreiben
				outStream.write(vrn.getBytes());
				if ((vr == DcmDDE.VRCODE_OB) | (vr == DcmDDE.VRCODE_OW) | (vr == DcmDDE.VRCODE_SQ) | (vr == DcmDDE.VRCODE_UN)) {
					writeUInt16(0x0000);

					// Datenlaenge schreiben
					writeUInt32(length);
				} else {
					writeUInt16(length);
				} 
			} 

			if (extEncoding == DcmDataObject.LITTLE_ENDIAN) {

				// LITTLE_ENDIAN
				buf = dcmval.getData();
			} else {

				// BIG_ENDIAN
				// Intern werden Daten als littel Endian gespeichert
				buf = (byte[]) dcmval.getData().clone();
				DcmValue.swapBuf(buf, vr);
			} 

			// Daten schreiben
			outStream.write(buf);

			// Nun werden Datenblocks ungerader Laenge um ein Fuellzeichen ergaenzt (s.o.),
			// wenn sie von einem bestimmten Typ sind
			if (padding) {
				buf = new byte[1];
				if ((vr == DcmDDE.VRCODE_UI) | (vr == DcmDDE.VRCODE_OB)) {
					buf[0] = 0x00;
				} else {
					buf[0] = 0x20;
				} 
				outStream.write(buf);
			}		// if
 
		}			// while
 
	}				// Methode DcmToRaw
 

	/**
	 * Die Methode schreibt einen 16-Bit Ganzzahlwert (ohne Vorzeichen) in den
	 * Dateipuffer.
	 * @param i Die zu speichernde 16-Bit Ganzzahl (ohne Vorzeichen)
	 */
	private void writeUInt16(int i) throws Exception {

		// Anlegen von 2 Bytes
		byte	dat[] = new byte[2];

		if (extEncoding == DcmDataObject.LITTLE_ENDIAN) {
			dat[0] = (byte) (i);
			dat[1] = (byte) (i >>> 8);
		} else {
			dat[1] = (byte) (i);
			dat[0] = (byte) (i >>> 8);
		} 

		// Schreiben der Daten in den Stream
		outStream.write(dat);
	} 


	/**
	 * Die Methode schreibt einen 32-Bit Ganzzahlwert (ohne Vorzeichen) in den
	 * Dateipuffer.
	 * @param i Die zu speichernde 32-Bit Ganzzahl
	 * @param buf Der Dateipuffer, in den geschrieben werden soll.
	 */
	private void writeUInt32(long i) throws Exception {

		// Anlegen von 4 Bytes
		byte	dat[] = new byte[4];

		if (extEncoding == DcmDataObject.LITTLE_ENDIAN) {
			dat[0] = (byte) (i);
			dat[1] = (byte) (i >>> 8);
			dat[2] = (byte) (i >>> 16);
			dat[3] = (byte) (i >>> 24);
		} else {
			dat[3] = (byte) (i);
			dat[2] = (byte) (i >>> 8);
			dat[1] = (byte) (i >>> 16);
			dat[0] = (byte) (i >>> 24);
		} 

		// Schreiben der Daten in den Puffer
		outStream.write(dat);
	} 


	/**
	 * Die Methode uebernimmt das angegebene Byte-Array als Preambel fuer
	 * den DICOM Meta File. Ist die Laenge < 128 Byte, werden die fehlenden
	 * mit 0x00 aufgefuellt. Ist die Laenge > 128 Byte, werden nur die
	 * ersten 128 Byte uebernommen.
	 * @param pre	Das Byte-Array mit den Informationen fuer die Preambel
	 */
	public void setPreambel(byte[] pre) {
		int l = 128;

		if (pre.length < 128) {
			l = pre.length;
		} 
		for (int i = 0; i < l; i++) {
			metaFilePreambel[i] = pre[i];
		} 
	} 


	/**
	 * Die Methode schreibt die Preambel und den File Meta Information
	 * Block in den Stream.
	 */
	private void writeFileMetaInfo() throws Exception {
		DcmDataObject					metaDDO;
		ByteArrayOutputStream baos;
		DcmOutputStream				dos;
		byte[]								buf;
		String								s;

		// Preambel schreiben
		outStream.write(metaFilePreambel);

		// DICOM Prefix schreiben
		buf = new byte[4];

		// ans: Bei der Angabe in '...' hat jikes gemeckert. Angaben in Hochkomma sind
		// eigentlich Unicode, d.h. nicht unbedint byte. Beizeiten sollten wir das
		// folgende austesten:
		// buf[0] = (byte)86;		// 'D';
		// buf[1] = (byte)73;		// 'I';
		// buf[2] = (byte)67;		// 'C';
		// buf[3] = (byte)77;		// 'M';

		buf[0] = 'D';
		buf[1] = 'I';
		buf[2] = 'C';
		buf[3] = 'M';
		outStream.write(buf);

		// Meta Information erstellen
		metaDDO = new DcmDataObject(DcmDataObject.LITTLE_ENDIAN, DcmDataObject.EXPLICITE_VR, DcmDataObject.PLAIN_STORAGE);

		// DD_FileMetaInformationVersion entsprechend DICOM 3
		metaDDO.setUS(0x0002, 0x0001, 0x0100);

		// DD_MediaStorageSOPClassUID (SOP Common Module P 3.3-C.12.1)
		metaDDO.setString(0x0002, 0x0002, copyDDO.getString(DcmDDE.DD_SOPClassUID));

		// DD_MediaStorageSOPInstanceUID (SOP Common Module P 3.3-C.12.1)
		metaDDO.setString(0x0002, 0x0003, copyDDO.getString(DcmDDE.DD_SOPInstanceUID));

		// Transfer Syntax
		if (copyDDO.getEncoding() == DcmDataObject.LITTLE_ENDIAN) {

			// LITTLE_ENDIAN
			if (copyDDO.getStructure() == DcmDataObject.IMPLICITE_VR) {

				// IMPLICITE_VR
				// DD_TransferSyntaxUID
				metaDDO.setString(0x0002, 0x0010, DcmUID.LI_IM_TRANSFER_UID);
			} else {

				// EXPLICITE_VR
				// DD_TransferSyntaxUID
				metaDDO.setString(0x0002, 0x0010, DcmUID.LI_EX_TRANSFER_UID);
			} 
		} else {

			// BIG_ENDIAN
			if (copyDDO.getStructure() == DcmDataObject.IMPLICITE_VR) {

				// IMPLICITE_VR
				throw new Exception("Big Endian/Implicite VR is not a DICOM Transfersyntax");
			} else {

				// EXPLICITE_VR
				// DD_TransferSyntaxUID
				metaDDO.setString(0x0002, 0x0010, DcmUID.BI_EX_TRANSFER_UID);
			} 
		} 

		// DD_ImplementationClassUID
		metaDDO.setString(0x0002, 0x0012, DcmUID.ORG_ROOT_UID + DcmUID.IMPLEMENTATION_UID_SUFFIX);

		// DcmDataObject in Byte-Array umwandeln
		baos = new ByteArrayOutputStream();
		dos = new DcmOutputStream(baos);
		dos.writeDDO(metaDDO);
		dos.flush();
		buf = baos.toByteArray();
		baos.close();
		dos.close();

		// File Information schreiben
		outStream.write(buf);
	} 

}




/*--- formatting done in "My Own Convention" style on 11-11-2000 ---*/

