/*--- formatted by Jindent 2.1, (www.c-lab.de/~jindent) ---*/

/*
 * Copyright (C) 2000 Thomas Hacklaender, e-mail: reply@thomas-hacklaender.de
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License 2
 * as published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * 
 * http://www.gnu.org/copyleft/copyleft.html
 */
package rad.dicom.ima;


import java.util.*;
import java.awt.image.*;

import rad.dicom.dcm.*;


/**
 * Diese Klasse repraesentiert ein "Composite IOD Module" fuer DICOM 3
 * IOD's, die Bilder beinhalten. Es werden hierzu diejenigen IOD Modules
 * abgebildet, die in allen IOD's der radiologischen Bilddateien (CT, MR,
 * CR, SC) als "Mandatory" gekennzeichnet sind (PS 3.3 - A.1.4).
 * Die folgenden Eigenschaften muessen fuer die Bildinformationen erfuellt
 * sein:
 * <ul>
 * <li> Monochrom oder "Palette Color Image" </li>
 * <li> 8 oder 16 bit/pixel <li></ul>
 * <DL><DT><B>Modifications: </B><DD>
 * tha 2000.10.9:	In den Methoden hv2pv und pv2hu einen Fehler in der Umrechnung
 * von Houndsfield-Einheiten in Pixelwerte korrigiert. <br>
 * tha 2000.3.5:	Window Eistellungen in lastCenter/lastWidth beziehen sich
 * immer auf Pixelwerte. Auch bei CT Bildern. <br>
 * Houndsfieldeinheiten werden in Pixelwerte umgerechnet.
 * tha 2000.4.2:  Definition der Konstanten CW_AUTO und CW_DEFAULT geloescht.
 * Befinden sich jetzt nur noch in ImagePanel. <br>
 * tha 2000.4.2:  calcAutomaticCW aus IamgePanel hierher uebertragen. Damit
 * werden jetzt auch dann plausble Center/Width Werte in lastCenter/lastWidth
 * uebernommen, wenn keine Voreinstellung in DcmDataObject gefunden wird. <br>
 * tha 2000.04.02: Defaultwerte bei der Umwandlung von Center/Width auf 2048/
 * 4096 korrigiert. <br>
 * tha 2000.05.08: calcAutomaticCW als static method in ImaUtil verschoben <br>
 * </DD></DL>
 * @author  Thomas Hacklaender
 * @version 2000.10.09
 */
public class GeneralImageIOD {

	// Die unterstuetzten "Photometric Interpretation" nach PS 3.3-C.7.6.3


	/**
	 * Photometric Interpretation nich unterstuetzt
	 */
	public static final int PI_UNSUPPORTED = 0;


	/**
	 * Photometric Interpretation MONOCHROME1
	 */
	public static final int PI_MONOCHROME1 = 1;


	/**
	 * Photometric Interpretation MONOCHROME2
	 */
	public static final int PI_MONOCHROME2 = 2;


	/**
	 * Photometric Interpretation PALETTE COLOR
	 */
	public static final int PI_PALETTE_COLOR = 3;


	// Parameter, die von extractImageValues extrahiert werde.


	/**
	 * Patient Module PS 3.3-C.7.1.1:
	 */
	public String						patientName;


	/**
	 * Patient Module PS 3.3-C.7.1.1:
	 */
	public String						patientBirthDate;


	/**
	 * General Study Module PS 3.3-C.7.2.1:
	 */
	public String						studyID;


	/**
	 * General Series Module PS 3.3-C.7.3.1:
	 */
	public int							seriesNumber;


	/**
	 * General Image PS 3.3-C.7.6.1:
	 */
	public int							imageNumber;


	/**
	 * [mm]. Image Plane Module PS 3.3-C.7.6.2:
	 */
	public double						pixelSpacingRow;


	/**
	 * [mm]. Image Plane Module PS 3.3-C.7.6.2:
	 */
	public double						pixelSpacingColumn;


	/**
	 * [mm]. Image Plane Module PS 3.3-C.7.6.2:
	 */
	public double						sliceThickness;


	/**
	 * [mm]. Image Plane Module PS 3.3-C.7.6.2:
	 */
	public double						sliceLocation;


	/**
	 * Image Pixel Module PS 3.3-C.7.6.3:
	 */
	public int							samplesPPixel;


	/**
	 * Image Pixel Module PS 3.3-C.7.6.3:
	 */
	public int							photometricInterpretation = PI_UNSUPPORTED;


	/**
	 * Image Pixel Module PS 3.3-C.7.6.3:
	 */
	public int							rows;


	/**
	 * Image Pixel Module PS 3.3-C.7.6.3:
	 */
	public int							columns;


	/**
	 * Image Pixel Module PS 3.3-C.7.6.3:
	 */
	public int							bitsAllocated;


	/**
	 * Image Pixel Module PS 3.3-C.7.6.3:
	 */
	public int							bitsStored;


	/**
	 * Image Pixel Module PS 3.3-C.7.6.3:
	 */
	public int							highBit;


	/**
	 * true, wenn bitsAllocated <= 8
	 */
	public boolean					oneBytePixel;


	/**
	 * true, wenn die Pixelwerte unsigned Werte sind
	 */
	public boolean					unsignedPixel;


	/**
	 * CT Image Module PS 3.3-C.8.2:
	 */
	public int							rescaleIntercept;


	/**
	 * CT Image Module PS 3.3-C.8.2:
	 */
	public int							rescaleSlope;


	/**
	 * In SV units or HU (CT-images). VOI LUT Module PS 3.3-C.11.2:
	 */
	public int							windowCenter;


	/**
	 * In SV units or HU (CT-images). VOI LUT Module PS 3.3-C.11.2:
	 */
	public int							windowWidth;


	/**
	 * true, wenn Center/Window definiert worden sind
	 */
	public boolean					isCenterWidth = false;


	/**
	 * SOP Common Module PS 3.3-C.12.1:
	 */
	public String						classUID;


	/**
	 * SOP Common Module PS 3.3-C.12.1:
	 */
	public String						instanceUID;


	/**
	 * Das originale DcmDataObject ohne die Pixel Daten (Gruppe 0x7FE0)
	 */
	public DcmDataObject		headerDDO;


	/**
	 * Die Pixelmatrix mit (signed) 16 Bit pro Pixel
	 */
	public short[]					pixel16 = new short[0];


	/**
	 * Der kleinste moegliche Pixelwert
	 */
	public int							minPixelValue = 0;


	/**
	 * Der groesste moegliche Pixelwert
	 */
	public int							maxPixelValue = 0;


	/**
	 * Das ColorModel fuer Pixelmatrix
	 */
	public ColorModel				cModel = ColorModel.getRGBdefault();


	/**
	 * Der Wert gibt den letzte Wert des Center-Parameters des Windows des Bildes
	 * wieder. Es wird zunaechst auf den Wert windowCenter gesetzt.
	 * Die Angabe erfolgt immer Pixelwerte. Fuer CT Bilder werden die
	 * Center und Width Werte von Houndsfield-Einheiten in Pixelwerte
	 * umgerechnet.
	 * Externe Klassen (z.B. ImagePanel) duerfen diesen Wert veraendert.
	 */
	public int							lastCenter;


	/**
	 * Der Wert gibt den letzte Wert des Width-Parameters des Windows des Bildes
	 * wieder. Es wird zunaechst auf den Wert windowWidth gesetzt.
	 * Die Angabe erfolgt immer Pixelwerte. Fuer CT Bilder werden die
	 * Center und Width Werte von Houndsfield-Einheiten in Pixelwerte
	 * umgerechnet.
	 * Externe Klassen (z.B. ImagePanel) duerfen diesen Wert veraendert.
	 */
	public int							lastWidth;


	/**
	 * Der fuer die 10te und 90te Perzentiele berechneten Center Werte
	 * des Windows. Die Angabe erfolgt immer Pixelwerte.
	 */
	public int							autoCenter;


	/**
	 * Der fuer die 10te und 90te Perzentiele berechneten Width Werte
	 * des Windows. Die Angabe erfolgt immer Pixelwerte.
	 */
	public int							autoWidth;


	/**
	 * Extrahiert aus einem DcmDataObject die relevanten Daten fuer eine
	 * Bilddarstellung (z.B. mit der Klasse ImagePanel/ImageBtnPanel).
	 * Die Pruefung auf DICOM Konformitaet laesst auch ACR Nema Bilder zu.
	 * @param d    Das DICOMDataObject, das das Bild beschreibt
	 * @exception  Exception
	 */
	public GeneralImageIOD(DcmDataObject ddo) throws Exception {
		this(ddo, true);
	}


	/**
	 * Extrahiert aus einem DcmDataObject die relevanten Daten fuer eine
	 * Bilddarstellung (z.B. mit der Klasse ImagePanel/ImageBtnPanel).
	 * @param ddo  Das DICOMDataObject, das das Bild beschreibt
	 * @param allowACRNema  <b>true</b> = Fuer Elemente, die im DcmDataObject nicht
	 * definiert sind, werden Defaultwerte angenommen. Damit
	 * koennen auch ACR-Nema konforme Bilder dargestellt werden.
	 * <b>false</b> = Pflichtfelder nach DICOM 3 muessen vorhanden
	 * sein. Anderenfalls wird eine Exception ausgeloest.
	 * @exception  Exception
	 */
	public GeneralImageIOD(DcmDataObject ddo, boolean allowACRNema) throws Exception {

		if (ddo == null) {
			throw new Exception("GeneralImageIOD: No DcmDataObject specified");
		} 

		extractImageValues(ddo);

		if (allowACRNema) {
			if (photometricInterpretation == PI_UNSUPPORTED) {
				photometricInterpretation = PI_MONOCHROME2;
			} 
		} 

		if (samplesPPixel > 1) {
			throw new Exception("GeneralImageIOD: Only monochrom/palette color images supported");
		} 
		if ((bitsAllocated != 8) & (bitsAllocated != 16)) {
			throw new Exception("GeneralImageIOD: Only 8 or 16 bit/pixel supported");
		} 
		if (photometricInterpretation == PI_UNSUPPORTED) {
			throw new Exception("GeneralImageIOD: Photometric Interpretation not supported");
		} 

		if (unsignedPixel) {
			minPixelValue = 0;
			maxPixelValue = (int) Math.pow(2, bitsStored) - 1;
		} else {
			minPixelValue = -((int) Math.pow(2, bitsStored)) / 2;
			maxPixelValue = ((int) Math.pow(2, bitsStored)) / 2 - 1;
		} 

		extractPixels(ddo);

		switch (photometricInterpretation) {
		case PI_MONOCHROME1:
			setMonochrome1ColorModel();
			break;

		case PI_MONOCHROME2:
			setMonochrome2ColorModel();
			break;

		case PI_PALETTE_COLOR:
			extractLUT(ddo);
			break;
		}

		headerDDO = ddo.getHeaderCopyOfMe();

		int[] w = ImaUtil.calcAutomaticCW(pixel16, bitsStored, unsignedPixel);
		autoCenter = w[0];
		autoWidth = w[1];

		// Die aktuelle Window-Einstellung festlegen
		if (isCenterWidth) {
			if (classUID.compareTo(DcmUID.CT_STORAGE_UID) == 0) {

				// Bei CT-Bildern Houndsfield-Einheiten in Pixelwerte umrechnen
				int[] window = new int[2];

				window[0] = windowCenter;
				window[1] = windowWidth;
				hu2pv(window);
				lastCenter = window[0];
				lastWidth = window[1];
			} else {
				lastCenter = windowCenter;
				lastWidth = windowWidth;
			} 
		} else {
			lastCenter = autoCenter;
			lastWidth = autoWidth;
		} 
	}


	/**
	 * Wandelt eine Window-Angabe von Houndsfield-Einheiten in Pixelwerte. <br>
   * hu = Hounsfield Units<br>
   * pv = Pixel Value<br>
	 * pv = (hu - intercept) / slope
	 * @param	window	windowCenter = window[0], windowWidth = window[1]
	 */
	public void hu2pv(int[] window) {
		if (rescaleSlope == 0) {
			return;
		}
		window[0] = (window[0] - rescaleIntercept) / rescaleSlope;
    // tha 2000.10.09: Geaendert
		// window[1] = (window[1] - rescaleIntercept) / rescaleSlope;
		window[1] = window[1] / rescaleSlope;
	} 


	/**
	 * Wandelt eine Window-Angabe von Pixelwerten in Houndsfield-Einheiten. <br>
   * hu = Hounsfield Units<br>
   * pv = Pixel Value<br>
	 * hu = slope * pv + intercept
	 * @param	window	windowCenter = window[0], windowWidth = window[1]
	 */
	public void pv2hu(int[] window) {
		window[0] = rescaleSlope * window[0] + rescaleIntercept;
    // tha 2000.10.09: Geaendert
		// window[1] = rescaleSlope * window[1] + rescaleIntercept;
		window[1] = rescaleSlope * window[1];
	}


	/**
	 * Setzt die lokalen Variablen der class auf die Werte des
	 * DcmDataObject.
	 * @exception dcm.Exception.
	 */
	private void extractImageValues(DcmDataObject ddo) throws Exception {
		String		s;
		String[]	sa;

		// Patient Module PS 3.3-C.7.1.1:

		patientName = ddo.getString(DcmDDE.DD_PatientsName);
		patientBirthDate = ddo.getString(DcmDDE.DD_PatientsBirthDate);

		// General Study Module PS 3.3-C.7.2.1:

		studyID = ddo.getString(DcmDDE.DD_StudyID);

		// General Series Module PS 3.3-C.7.3.1:

		seriesNumber = (int) DcmValue.str2Long(ddo.getString(DcmDDE.DD_SeriesNumber), 1);

		// General Equipment PS 3.3-C.7.5.1:
		// Keine relevanten Daten

		// General Image PS 3.3-C.7.6.1:
		// Ab der 1999 Ausgabe der Norm wird "Image Number" als "Instance Number"
		// bezeichnet

		imageNumber = (int) DcmValue.str2Long(ddo.getString(DcmDDE.DD_InstanceNumber), 1);

		// Image Plane Module PS 3.3-C.7.6.2:

		sa = DcmValue.str2StringArray(ddo.getString(DcmDDE.DD_PixelSpacing), "\\");
		pixelSpacingRow = 0.0;
		pixelSpacingColumn = 0.0;
		if (sa.length > 0) {
			pixelSpacingRow = DcmValue.str2Double(sa[0], 0.0);
		} 
		if (sa.length > 1) {
			pixelSpacingColumn = DcmValue.str2Double(sa[1], 0.0);
		} 
		sliceThickness = DcmValue.str2Double(ddo.getString(DcmDDE.DD_SliceThickness), 1);
		sliceLocation = DcmValue.str2Double(ddo.getString(DcmDDE.DD_SliceLocation), 0);

		// Image Pixel Module PS 3.3-C.7.6.3:
		// - siehe auch PS 3.5-8: Encoding of Pixel Data
		// - siehe auch PS 3.5-D: Examples of various pixel data and overlay
		// encoding schemes

		samplesPPixel = ddo.getUS(DcmDDE.DD_SamplesPerPixel);

		// Setup PhotometricInterpretation
		s = ddo.getString(DcmDDE.DD_PhotometricInterpretation);
		if (s.compareTo("MONOCHROME1") == 0) {
			photometricInterpretation = PI_MONOCHROME1;
		} 
		if (s.compareTo("MONOCHROME2") == 0) {
			photometricInterpretation = PI_MONOCHROME2;
		} 
		if (s.compareTo("PALETTE COLOR") == 0) {
			photometricInterpretation = PI_PALETTE_COLOR;
		} 

		rows = ddo.getUS(DcmDDE.DD_Rows);
		columns = ddo.getUS(DcmDDE.DD_Columns);
		bitsAllocated = ddo.getUS(DcmDDE.DD_BitsAllocated);
		oneBytePixel = bitsAllocated <= 8;
		bitsStored = ddo.getUS(DcmDDE.DD_BitsStored);
		highBit = ddo.getUS(DcmDDE.DD_HighBit);
		unsignedPixel = ddo.getUS(DcmDDE.DD_PixelRepresentation) == 0;

		// VOI LUT Module PS 3.3-C.11.2:
		// - Center und Window sind optionale Attribute.
		// - sind mehere Paare definiert, wird das erste Paar genommen.

		if (ddo.isAvailable(DcmDDE.DD_WindowCenter) & ddo.hasValue(DcmDDE.DD_WindowCenter)) {
			sa = DcmValue.str2StringArray(ddo.getString(DcmDDE.DD_WindowCenter), "\\");
			windowCenter = (int) DcmValue.str2Long(sa[0], 2048);
			sa = DcmValue.str2StringArray(ddo.getString(DcmDDE.DD_WindowWidth), "\\");
			windowWidth = (int) DcmValue.str2Long(sa[0], 4096);
			isCenterWidth = true;
		} 

		// SOP Common Module PS 3.3-C.12.1:

		classUID = ddo.getString(DcmDDE.DD_SOPClassUID);
		instanceUID = ddo.getString(DcmDDE.DD_SOPInstanceUID);

		// CT Image Module PS 3.3-C.8.2:

		if (classUID.compareTo(DcmUID.CT_STORAGE_UID) == 0) {

			// Gleichung zur Umrechnung HE in Pixelwerte
			rescaleIntercept = (int) DcmValue.str2Long(ddo.getString(DcmDDE.DD_RescaleIntercept), 0);
			rescaleSlope = (int) DcmValue.str2Long(ddo.getString(DcmDDE.DD_RescaleSlope), 1);
		} 
	} 


	/**
	 * Extrahiert die Pixeldaten aus dem DcmDataObject.
	 * @exception dcm.Exception.
	 */
	private void extractPixels(DcmDataObject ddo) throws Exception {
		DcmValue	dcv;
		int				nPixel;
		int				i, p;
		short			v;
		int				numShift;
		byte[]		data;
		int				bitMask;

		if (!ddo.hasValue(DcmDDE.DD_PixelData)) {
			throw new Exception("GeneralImageIOD.extractPixels: Can't find pixel data");
		} 

		nPixel = rows * columns;
		dcv = ddo.getDcmValue(DcmDDE.DD_PixelData);
		data = dcv.getData();

		if (oneBytePixel) {
			if (data.length < nPixel) {
				throw new Exception("GeneralImageIOD.extractPixels: Not enough pixel data");
			} 
		} else {
			if (data.length < 2 * nPixel) {
				throw new Exception("GeneralImageIOD.extractPixels: Not enough pixel data");
			} 
		} 

		pixel16 = new short[nPixel];
		numShift = bitsAllocated - bitsStored;
		bitMask = (int) Math.pow(2, bitsStored) - 1;
		i = 0;
		p = 0;

		if (oneBytePixel) {

			while (p < nPixel) {
				if (unsignedPixel) {
					pixel16[p++] = (short) (data[i++] & bitMask);
				} else {
					v = data[i++];
					pixel16[p++] = (short) ((v << numShift) >> numShift);
				} 
			} 

		} else {

			while (p < nPixel) {

				// Daten werden als little Endian gespeichert
				v = (short) (((((int) data[i++]) & 0x00ff) | (((int) data[i++]) & 0x00ff) << 8));

				if (unsignedPixel) {
					pixel16[p++] = (short) (v & bitMask);
				} else {
					pixel16[p++] = (short) ((v << numShift) >> numShift);
				} 
			} 

		} 
	} 


	/**
	 * Generiert das ColorModel fuer das Bild auf die Einstellung:
	 * Lineare Graustufen, Pixelwert 0 ist weiss, maximaler Pixelwert ist
	 * schwarz.
	 * Dies entspricht einer "Photometric Interpretation" = MONOCHROM1. Siehe:
	 * Image Pixel Module, Photometric Interpretation PS 3.3-C.7.6.3.1.2.
	 * Jedes ColorModel des Bildes ist ein IndexColorModel mit 256 moeglichen
	 * Werten. Die Pixelwerte des Bildes werden auf das Intervall 0..255
	 * skaliert (= gefenstert).
	 * @exception dcm.Exception.
	 */
	private void setMonochrome1ColorModel() {
		byte[]	cTab = new byte[256];

		for (int i = 0; i < 256; i++) {
			cTab[i] = (byte) (256 - i);
		} 
		cModel = new IndexColorModel(8, 256, cTab, cTab, cTab);
	} 


	/**
	 * Generiert das ColorModel fuer das Bild auf die Einstellung:
	 * Lineare Graustufen, Pixelwert 0 ist schwarz, maximaler Pixelwert ist
	 * weiss.
	 * Dies entspricht einer "Photometric Interpretation" = MONOCHROM2. Siehe:
	 * Image Pixel Module, Photometric Interpretation PS 3.3-C.7.6.3.1.2.
	 * Jedes ColorModel des Bildes ist ein IndexColorModel mit 256 moeglichen
	 * Werten. Die Pixelwerte des Bildes werden auf das Intervall 0..255
	 * skaliert (= gefenstert).
	 * @exception dcm.Exception.
	 */
	private void setMonochrome2ColorModel() {
		byte[]	cTab = new byte[256];

		for (int i = 0; i < 256; i++) {
			cTab[i] = (byte) i;
		} 
		cModel = new IndexColorModel(8, 256, cTab, cTab, cTab);
	} 


	/**
	 * Generiert das ColorModel fuer das Bild. Die Farbkomponenten der Indices
	 * werden aus dem DcmDataObject extrahiert.
	 * Dies entspricht einer "Photometric Interpretation" = PALETTE_COLOR. Siehe:
	 * Image Pixel Module, Photometric Interpretation PS 3.3-C.7.6.3.1.2.
	 * Jedes ColorModel des Bildes ist ein IndexColorModel mit 256 moeglichen
	 * Werten. Die Pixelwerte des Bildes werden auf das Intervall 0..255
	 * skaliert (= gefenstert).
	 * @exception dcm.Exception.
	 */
	private void extractLUT(DcmDataObject ddo) throws Exception {
		DcmValue	dcvDescRed, dcvDescGreen, dcvDescBlue;
		DcmValue	dcvDataRed, dcvDataGreen, dcvDataBlue;
		byte[]		rTab, gTab, bTab;

		dcvDescRed = ddo.getDcmValue(DcmDDE.DD_RedPaletteColorLookupTableDescriptor);
		dcvDescGreen = ddo.getDcmValue(DcmDDE.DD_GreenPaletteColorLookupTableDescriptor);
		dcvDescBlue = ddo.getDcmValue(DcmDDE.DD_BluePaletteColorLookupTableDescriptor);
		if ((dcvDescRed == null) | (dcvDescGreen == null) | (dcvDescBlue == null)) {
			throw new Exception("GeneralImageIOD.extractLUT: Can't find LUT descriptors");
		} 

		dcvDataRed = ddo.getDcmValue(DcmDDE.DD_RedPaletteColorLookupTableData);
		dcvDataGreen = ddo.getDcmValue(DcmDDE.DD_GreenPaletteColorLookupTableData);
		dcvDataBlue = ddo.getDcmValue(DcmDDE.DD_BluePaletteColorLookupTableData);
		if ((dcvDataRed == null) | (dcvDataGreen == null) | (dcvDataBlue == null)) {
			throw new Exception("GeneralImageIOD.extractLUT: Can't find LUT data");
		} 

		rTab = new byte[maxPixelValue + 1];
		gTab = new byte[maxPixelValue + 1];
		bTab = new byte[maxPixelValue + 1];

		setColorComponent(dcvDescRed, dcvDataRed, rTab);
		setColorComponent(dcvDescGreen, dcvDataGreen, gTab);
		setColorComponent(dcvDescBlue, dcvDataBlue, bTab);

		cModel = new IndexColorModel(bitsStored, rTab.length, rTab, gTab, bTab);
	} 


	/**
	 * Extrahiert eine Farbkomponente aus dem DcmDataObject.
	 * Image Pixel Module, Palette Color Lookup PS 3.3-C.7.6.3.1.5., C.7.6.3.1.6
	 * @param dcvDesc	 Color Lookup Table Descriptor
	 * @param dcvDesc	 Color Lookup Table Data
	 * @param cTab		 Byte-Array als Farmkomponente des IndexColorModels des Bildes
	 * @exception dcm.Exception.
	 */
	private void setColorComponent(DcmValue dcvDesc, DcmValue dcvData, byte[] cTab) throws Exception {
		int		numEntries;
		int		numBits;
		int		firstMappedPixel, lastMappedPixel;
		int		lutMask;
		int		idxLast;
		byte	firstLUTvalue;
		byte	lastLUTvalue;

		// Der Descriptor besteht aus drei aufeinanderfolgenden SS oder US
		// Da die Zahlenwerte nie ausserhalb des Wertebereiches von SS liegen,
		// werden sie als SS eingelesen.
		numEntries = DcmValue.bufToSS(dcvDesc.getData(), 0);
		firstMappedPixel = DcmValue.bufToSS(dcvDesc.getData(), 2);
		numBits = DcmValue.bufToSS(dcvDesc.getData(), 4);

		if (firstMappedPixel < 0) {
			throw new Exception("GeneralImageIOD.setColorComponent: Only unsigned integer pixel values supported");
		} 
		if (numBits > 8) {
			throw new Exception("GeneralImageIOD.setColorComponent: Max. 8 bit/color component supported");
		} 

		lutMask = (int) Math.pow(2, numBits) - 1;
		lastMappedPixel = firstMappedPixel + numEntries - 1;
		idxLast = cTab.length - 1;
		if (lastMappedPixel > idxLast) {
			lastMappedPixel = idxLast;
			numEntries = lastMappedPixel - firstMappedPixel + 1;
		} 

		// Die Daten bestehen aus aufeinanderfolgenden SS oder US.
		// Diese Methode unterstuetzt nur US.
		for (int i = 0; i <= idxLast; i++) {
			cTab[i] = (byte) (DcmValue.bufToUS(dcvData.getData(), i * 2) & lutMask);
		} 

		// Bei Farbpaletten muss immer der gesamte Wertebereich sichtar sein
		isCenterWidth = true;
		windowWidth = numEntries;
		windowCenter = (lastMappedPixel - firstMappedPixel) / 2;
	}
}


/*--- formatting done in "My Own Convention" style on 04-30-2000 ---*/

