/*--- formatted by Jindent 2.1, (www.c-lab.de/~jindent) ---*/

/*
 * Copyright (C) 2000 Thomas Hacklaender, e-mail: reply@thomas-hacklaender.de
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License 2
 * as published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * 
 * http://www.gnu.org/copyleft/copyleft.html
 */
package rad.dicom.ima;

import java.awt.*;
import java.awt.image.*;


/**
 * Diese Klasse definiert statische Methoden, die an verschiedenen
 * Stellen im Package <code>rad.dicom.ima</code> benoetigt werden.
 * <DL><DT><B>Modifications: </B><DD>
 * Thomas Hacklaender 2000.8.23:
 * Methode GIODtoImage ergaenzt.<br>
 * </DD></DL>
 * @author  Thomas Hacklaender
 * @version 2000.08.23
 */
public class ImaUtil {

	/**
	 * Berechnet die Window-Werte (Center und Width), sodass alle Pixel zwischen
	 * der 10ten und 90ten Perzentiele der Histogrammverteilung der Pixelwerte
	 * des Bildes dargestellt werden.
	 * @param genIma				Das GeneralImageIOD Objekt mit den Bilddaten.
	 * @return							Ein int[] mit den Window-Werten: [0]=Center, [1]=Width.
	 */
	static public int[] calcAutomaticCW(GeneralImageIOD genIma) {
    return calcAutomaticCW(genIma.pixel16, genIma.bitsStored, genIma.unsignedPixel);
  }


	/**
	 * Berechnet die Window-Werte (Center und Width), sodass alle Pixel zwischen
	 * der 10ten und 90ten Perzentiele der Histogrammverteilung der Pixelwerte
	 * des Bildes dargestellt werden.
	 * @param pixel16				Die Pixelmatrix mit (signed) 16 Bit pro Pixel.
	 * @param bitsStored		Anzahl der Bits, die von den 16 moeglichen pro Pixel signifikant sind.
	 * @param unsignedPixel	true, wenn es sich um unsigned Pixelwerte handelt.
	 * @return							Ein int[] mit den Window-Werten: [0]=Center, [1]=Width.
	 */
	static public int[] calcAutomaticCW(short[] pixel16, int bitsStored, boolean unsignedPixel) {
		int		nPixel;
		int[] histogram;
		int		numPixelValues;
		int		sum = 0;
		int		s;
		int		min;
		int		max;
		int[] window = new int[2];

		nPixel = pixel16.length;
		numPixelValues = (int) Math.pow(2, bitsStored);
		histogram = new int[numPixelValues];

		for (int i = 0; i < nPixel; i++) {
			histogram[pixel16[i]]++;
		}
		sum = 0;
		for (int i = 0; i < numPixelValues; i++) {
			sum += histogram[i];
		}

		s = 0;
		min = 0;
		for (int i = 0; i < numPixelValues; i++) {	// 10% Perzentiele
			s += histogram[i];
			if (s > sum / 10) {
				min = i;
				break;
			} 
		} 
		s = 0;
		max = 0;
		for (int i = numPixelValues - 1; i >= 0; i--) {		// 90% Perzentiele
			s += histogram[i];
			if (s > sum / 10) {
				max = i;
				break;
			} 
		} 

		if (!unsignedPixel) {
			min -= numPixelValues / 2;
			max -= numPixelValues / 2;
		}

		window[0] = (min + max) / 2;	// Center
		window[1] = max - min;        // Width

		return window;
	}


	/**
	 * Erzeugt aus einem GeneralImageIOD Objekt ein AWT Image Die Dimenion
   * des erzeugten Image entspricht den rows und columns der GeneralImageIOD.
	 * @param genIma				Das GeneralImageIOD Objekt.
	 * @param theCenter		  Center-Wert, der zur Umwandlung verwendet werden soll.
	 * @param theWidth    	Width-Wert, der zur Umwandlung verwendet werden soll.
	 * @return							Das AWT Image.
	 */
  public static Image GIODtoImage(GeneralImageIOD genIma, int theCenter, int theWidth) {

	  MemoryImageSource   imgSource = null;
		int			            minImageValue, maxImageValue;
	  byte[]              pixel8;
		double	            step;
		int			            value;

		// Aus Center und Width werden minImageValue und maxImageValue berechnet.
		// Alle Pixelwerte < minImageValue werden mit dem Farbwert von minImageValue,
		// alle Pixelwerte > maxImageValue mit dem Farbwert von maxImageValue
		// dargestellt.
		if (theCenter > genIma.maxPixelValue) {
			theCenter = genIma.maxPixelValue;
		}
		if (theCenter < genIma.minPixelValue) {
			theCenter = genIma.minPixelValue;
		}
		if (theWidth > genIma.maxPixelValue) {
			theWidth = genIma.maxPixelValue;
		}
		if (theWidth < 1) {
			theWidth = 1;
		}
		minImageValue = theCenter - theWidth / 2;
		maxImageValue = theCenter + theWidth / 2;

		if (minImageValue < genIma.minPixelValue) {
			minImageValue = genIma.minPixelValue;
		}
		if (maxImageValue > genIma.maxPixelValue) {
			maxImageValue = genIma.maxPixelValue;
		}

    // Die 16 Bit Pixelwerte des GeneralImageIOD werden so auf 8 Bit Werte
    // skaliert, das der 8 Bit Pixelwert 0 dem 16 Bit Pixelwert minImageValue
    // und der 8 Bit Pixelwert 255 dem Wert maxImageValue entspricht. Damit
    // kann der 8 Bit Pixelwert als Index fuer das ColorModel verwendet werden.
		pixel8 = new byte[genIma.pixel16.length];
		step = (double) ((IndexColorModel) genIma.cModel).getMapSize() / (maxImageValue - minImageValue + 1);

		for (int i = 0; i < genIma.pixel16.length; i++) {
			if (genIma.unsignedPixel) {

				// mask "sign"-bits
				if (genIma.oneBytePixel) {
					value = genIma.pixel16[i] & 0x000000ff;
				} else {
					value = genIma.pixel16[i] & 0x0000ffff;
				}
			} else {
				value = genIma.pixel16[i];
			}

			value = genIma.pixel16[i] - minImageValue;
			if (value < 0) {
				value = 0;
			}
			value = (int) (value * step);
			if (value > 255) {
				value = 255;
			}
			pixel8[i] = (byte) (value & 0xff);
		}

    // MemeoryImageSource aus den pixel8 Daten erzeugen
		imgSource = new MemoryImageSource(genIma.columns, genIma.rows, genIma.cModel, pixel8, 0, genIma.columns);

    // AWT Image aus der MemeoryImageSource erzeugen
		return (new Canvas()).createImage(imgSource);
  }

}


/*--- formatting done in "My Own Convention" style on 04-30-2000 ---*/

