/*--- formatted by Jindent 2.1, (www.c-lab.de/~jindent) ---*/

/*
 * Copyright (C) 2000 Thomas Hacklaender, e-mail: reply@thomas-hacklaender.de
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License 2
 * as published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * 
 * http://www.gnu.org/copyleft/copyleft.html
 */
package rad.dicom.ima;

import java.io.*;
import java.text.*;
import java.util.*;
import java.awt.*;
import java.awt.image.*;

import rad.dicom.dcm.*;


/**
 * Diese Klasse repraesentiert eine "Secondary Capture IOD" nach DICOM 3
 * (PS 3.3-A.8.). Neben den dort fuer alle Bilder definierten Informatien
 * Entities  wird das VOI LUT Module (Definiton des Windows [Center/Width])
 * nur dann eingeschlossen, wenn die Windowwerte wenigstens einmal mit
 * der Methode <code>setVOILUTModule</code> festgelegt wurden. Zur automatischen
 * Berechnung von Windowerten steht die <B>statische</B> Methode
 * <code>ImagePanel.calcAutomaticCW()</code> zur Verfuegung.
 * <DL><DT><B>Modifications: </B><DD>
 * tha 2000.02.21: Quellcode komplett umgeschrieben. <br>
 * tha 2000.04.02: Methode setPublicFields eingefuegt. <br>
 * tha 2000.05.08: VOILUT Module nur dann einschliessen, wenn Center/Width
 * mit setVOILUTModule mindestens einmal gestzt weurden. <br>
 * tha 2000.05.09: In addImagePixelIE wird der VR-Code fuer die Pixeldaten gesetzt.
 * </DD></DL>
 * @author   Thomas Hacklaender
 * @version  2000.05.09
 */
public class SecondaryCaptureIOD {

	/*
	 * !!! Der Inhalt einer Secondary Capture IOD wird in
	 * !!! PS 3.3-A.8. festgelegt.
	 */

	/* Felder, die ueber die Dialogbox gesetzt werden koennen */


	/**
	 * Patient Module PS 3.3-C.7.1.1
	 */
	public String					patientName = "Patient Name";								// Type 2


	/**
	 * Patient Module PS 3.3-C.7.1.1
	 */
	public String					patientID = "Patient ID";										// Type 2


	/**
	 * Patient Module PS 3.3-C.7.1.1
	 */
	public String					patientBirthDate = "19991231";							// Type 2


	/**
	 * Patient Module PS 3.3-C.7.1.1
	 */
	public String					patientSex = "O";														// Type 2


	/**
	 * General Study Module PS 3.3-C.7.2.1
	 */
	public String					referringPhysicianName = "";								// Type 2


	/**
	 * General Study Module PS 3.3-C.7.2.1
	 */
	public String					studyID = "ST000001";												// Type 2


	/**
	 * General Study Module PS 3.3-C.7.2.1
	 */
	public String					accessionNumber = "";												// Type 2


	/**
	 * General Series Module PS 3.3-C.7.3.1
	 */
	public String					seriesNumber = "1";													// Type 2


	/**
	 * General Image Module PS 3.3-C.7.6.1
	 */
	public String					imageNumber = "1";													// Type 2


	/**
	 * General Image Module PS 3.3-C.7.6.1
	 */
	public String					lrPatientOrientation = "";									// Type 2C


	/**
	 * General Image Module PS 3.3-C.7.6.1
	 */
	public String					tbPatientOrientation = "";									// Type 2C


	/* Felder, deren Inhalt automatisch generiert wird */

	// General Study Module PS 3.3-C.7.2.1
	private String				studyInstanceUID = DcmUID.ORG_ROOT_UID + DcmUID.STUDY_UID_SUFFIX + ".0";	// Type 1
	private String				studyDate = "19991231";											// Type 2
	private String				studyTime = "235959.123";										// Type 2

	// General Series Module PS 3.3-C.7.3.1
	private final String	modality = "OT";														// Type 1
	private String				seriesInstanceUID = DcmUID.ORG_ROOT_UID + DcmUID.SERIES_UID_SUFFIX + ".0";	// Type 1

	// General Image Module PS 3.3-C.7.6.1
	private String				imageDate = "19991231";											// Type 2C
	private String				imageTime = "235959.123";										// Type 2C
	private final String	imageType = "DERIVED\\SECONDARY";						// Type 3

	// SC Equipment Module PS 3.3-C.8.6.1
	private final String	conversionType = "WSD";											// Type 1

	// "Modality" ist schon in General Series Module definiert		          // Type 3
	private final String	secondaryCaptureDeviceID = "IMAGEJ";				// Type 3
	private final String	secondaryCaptureDeviceSoftwareVersion = "1.0";	// Type 3

	// SC Image Module PS 3.3-C.8.6.2
	private String				dateOfSecondaryCapture = "19991231";				// Type 3
	private String				timeOfSecondaryCapture = "235959.123";			// Type 3

	// VOI LUT Module PS 3.3-C.11.2
	private boolean				includeVOILUTModule = false;                // true nur dann,wenn Center/Width mit setVOILUTModule aktiv gesetzt wurden
	private String				windowCenter = "0";													// Type 3
	private String				windowWidth = "1";													// Type 3

	// SOP Common Module PS 3.3-C.12.1
	private final String	sopClassUID = DcmUID.SC_STORAGE_UID;				// Type 1
	private String				sopInstanceUID = DcmUID.ORG_ROOT_UID + DcmUID.SOP_UID_SUFFIX + ".0";	// Type 1


	/* Felder, deren Inhalt abhaengig vom Bildtyp automatisch generiert werden */

	// Image Pixel Module PS 3.3-C.7.6.3
	// Die Felder werden fuer ein "Pseudobild" der Groesse 1*1 Pixel mit 16 Bit
	// initialisiert.
	private int						samplesPerPixel = 1;												// Type 1
	private String				photometricInterpretation = "MONOCHROME2";	// Type 1
	private int						rows = 1;																		// Type 1
	private int						columns = 1;																// Type 1
	private int						bitsAllocated = 16;													// Type 1
	private int						bitsStored = 12;														// Type 1
	private int						highBit = 11;																// Type 1
	private int						pixelRepresentation = 0;										// Type 1
	private byte[]				pixelData = new byte[2];										// Type 1
	private final String	pixelAspectRatio = "1\\1";									// Type 1C

	// Nur fuer Bilder mit "Photometric Interretation = PALETTE COLOR"
	private boolean				includeLUTModule = false;
	private byte[]				redPaletteColorLookupTableDescriptor = new byte[0];		// Type 1C
	private byte[]				greenPaletteColorLookupTableDescriptor = new byte[0];		// Type 1C
	private byte[]				bluePaletteColorLookupTableDescriptor = new byte[0];	// Type 1C
	private byte[]				redPaletteColorLookupTableData = new byte[0];		// Type 1C
	private byte[]				greenPaletteColorLookupTableData = new byte[0];		// Type 1C
	private byte[]				bluePaletteColorLookupTableData = new byte[0];	// Type 1C


	/* In diese Feld wird das SC Bild aus den Einzelfelder aufgebaut */
	private DcmDataObject ddo = null;

	/* Datum und Uhrzeit zum Zeitpunkt der Erzeugung des DcmDatObject */
	private static String dateAndTimeSuffix;


	/**
	 * Die Root UID fuer alle anderen UID's
	 */
	public String					orgRootUID = DcmUID.ORG_ROOT_UID;


	/**
	 * Nur zum Debuggen.
	 */
	public SecondaryCaptureIOD() {}


	/**
	 * Erzeugt aus den Feldern der Klassen-Instanz ein neues DcmDataObject,
	 * das der Definition "Secondary Capture IOD" nach DICOM 3 (PS 3.3-A.8.)
	 * entspricht
	 * @param enc	Encoding der Transfersyntax
	 * @param str	Structure der Transfersyntax
	 * @param sto	Speicherformat des externen Speichers
	 * @return	  Das DcmDataObject
	 */
	public DcmDataObject getDDO(int enc, int str, int sto) {
		ddo = new DcmDataObject(enc, str, sto);
		setTimeAndDateData();
		addInformationEntities();

		return ddo;
	} 


	/**
	 * Besetzt die public Fields der Instanz des Objektes mit den Werten
	 * aus dem uebergebenen DcmDataObject.
	 * @param ddo	 Das DcmDataObject mit den Voreistellungen
	 */
	public void setPublicFields(DcmDataObject ddo) {
		patientName = ddo.getString(DcmDDE.DD_PatientsName);
		patientID = ddo.getString(DcmDDE.DD_PatientID);
		patientBirthDate = ddo.getString(DcmDDE.DD_PatientsBirthDate);
		patientSex = ddo.getString(DcmDDE.DD_PatientsSex);
		referringPhysicianName = ddo.getString(DcmDDE.DD_ReferringPhysiciansName);
		studyID = ddo.getString(DcmDDE.DD_StudyID);
		accessionNumber = ddo.getString(DcmDDE.DD_AccessionNumber);
		seriesNumber = ddo.getString(DcmDDE.DD_SeriesNumber);
		imageNumber = ddo.getString(DcmDDE.DD_InstanceNumber);
		String[]	sa = DcmValue.str2StringArray(ddo.getString(DcmDDE.DD_PatientOrientation), "\\");

		if (sa.length > 0) {
			lrPatientOrientation = sa[0].trim();
		} 
		if (sa.length > 1) {
			tbPatientOrientation = sa[1].trim();
		} 
	} 


	/**
	 * Mit dieser Methode werden alle auf Uhrzeit und/oder Datum
	 * basierende Felder der SecondaryCatureIOD mit der aktuellen
	 * Zeit gesetzt.
	 */
	private void setTimeAndDateData() {
		SimpleDateFormat	fd = new SimpleDateFormat("yyyyMMdd");
		SimpleDateFormat	ft = new SimpleDateFormat("HHmmss.SSS");
		SimpleDateFormat	fs = new SimpleDateFormat("yyyyMMddHHmmssSSS");
		Date							cdt = new Date();

		dateAndTimeSuffix = fs.format(cdt);

		studyDate = fd.format(cdt);
		studyTime = ft.format(cdt);
		dateOfSecondaryCapture = fd.format(cdt);
		timeOfSecondaryCapture = ft.format(cdt);
		imageDate = fd.format(cdt);
		imageTime = ft.format(cdt);
	} 


	/**
	 * Besetzt in der Klassen-Instanz die Felder fuer das "VOI LUT Module"
	 * (PS 3.3-C.11.2) fuer Center und Width des Bildes.
	 * @param wCenter   Der Center-Wert des Window
	 * @param wWidth    Der Width-Wert des Window
	 */
	public void setVOILUTModule(int wCenter, int wWidth) {
		windowCenter = Integer.toString(wCenter);
		windowWidth = Integer.toString(wWidth);
		includeVOILUTModule = true;
	} 


	/**
	 * Die in PS 3.3-A.8. festgelegt Information Entities werden in das
	 * DcmDataObject uebernommen. Das optionale VOI LUT Module wird
	 * ebenfalls immer uebernommen. Bei Farb-Paletten Bildern wird zusaetzlich
	 * das LUT Module eingeschlossen.
	 */
	private void addInformationEntities() {
		addPatientIE();				// PS 3.3-C.7.1.1
		addStudyIE();					// PS 3.3-C.7.2.1
		addSeriesIE();				// PS 3.3-C.7.3.1
		addGeneralImageIE();	// PS 3.3-C.7.6.1
		addImagePixelIE();		// PS 3.3-C.7.6.3
		addSCEquipmentIE();		// PS 3.3-C.8.6.1
		addSCImageIE();				// PS 3.3-C.8.6.2
		addSOPCommon();				// PS 3.3-C.12.1

		// Optional Modules
		if (includeLUTModule) {
			addLUTModule();			// PS 3.3-C.7.6.3
		}
		if (includeVOILUTModule) {
			addVOILUTModule();	// PS 3.3-C.11.2
		}
	}


	/**
	 * PS 3.3-C.7.1.1
	 */
	private void addPatientIE() {
		ddo.setString(DcmDDE.DD_PatientsName, patientName);
		ddo.setString(DcmDDE.DD_PatientID, patientID);
		ddo.setString(DcmDDE.DD_PatientsBirthDate, patientBirthDate);
		ddo.setString(DcmDDE.DD_PatientsSex, patientSex);
	} 


	/**
	 * PS 3.3-C.7.2.1
	 */
	private void addStudyIE() {
		ddo.setString(DcmDDE.DD_StudyInstanceUID, orgRootUID + DcmUID.STUDY_UID_SUFFIX + "." + dateAndTimeSuffix);
		ddo.setString(DcmDDE.DD_StudyDate, studyDate);
		ddo.setString(DcmDDE.DD_StudyTime, studyTime);
		ddo.setString(DcmDDE.DD_ReferringPhysiciansName, referringPhysicianName);
		ddo.setString(DcmDDE.DD_StudyID, studyID);
		ddo.setString(DcmDDE.DD_AccessionNumber, accessionNumber);
	}


	/**
	 * PS 3.3-C.7.3.1
	 */
	private void addSeriesIE() {
		ddo.setString(DcmDDE.DD_Modality, modality);
		ddo.setString(DcmDDE.DD_SeriesInstanceUID, orgRootUID + DcmUID.SERIES_UID_SUFFIX + "." + dateAndTimeSuffix);
		ddo.setString(DcmDDE.DD_SeriesNumber, seriesNumber);
	} 


	/**
	 * PS 3.3-C.8.6.1
	 */
	private void addSCEquipmentIE() {
		ddo.setString(DcmDDE.DD_ConversionType, conversionType);
		ddo.setString(DcmDDE.DD_SecondaryCaptureDeviceID, secondaryCaptureDeviceID);
		ddo.setString(DcmDDE.DD_SecondaryCaptureDeviceSoftwareVersion, secondaryCaptureDeviceSoftwareVersion);
	}


	/**
	 * PS 3.3-C.7.6.1
	 */
	private void addGeneralImageIE() {
		ddo.setString(DcmDDE.DD_InstanceNumber, imageNumber);
		ddo.setString(DcmDDE.DD_PatientOrientation, lrPatientOrientation + "\\" + tbPatientOrientation);
		ddo.setString(DcmDDE.DD_ImageDate, imageDate);
		ddo.setString(DcmDDE.DD_ImageTime, imageTime);
		ddo.setString(DcmDDE.DD_ImageType, imageType);
	} 


	/**
	 * PS 3.3-C.7.6.3
	 */
	private void addImagePixelIE() {
    DcmValue    dv;

		ddo.setUS(DcmDDE.DD_SamplesPerPixel, samplesPerPixel);
		ddo.setString(DcmDDE.DD_PhotometricInterpretation, photometricInterpretation);
		ddo.setUS(DcmDDE.DD_Rows, rows);
		ddo.setUS(DcmDDE.DD_Columns, columns);
		ddo.setUS(DcmDDE.DD_BitsAllocated, bitsAllocated);
		ddo.setUS(DcmDDE.DD_BitsStored, bitsStored);
		ddo.setUS(DcmDDE.DD_HighBit, highBit);
		ddo.setUS(DcmDDE.DD_PixelRepresentation, pixelRepresentation);

		dv = new DcmValue(DcmDDE.DD_PixelData, pixelData);
    if (bitsAllocated <= 8) {
      dv.setVR(DcmDDE.VRCODE_OB);
    } else {
      dv.setVR(DcmDDE.VRCODE_OW);
    }
		ddo.push(dv);

		ddo.setString(DcmDDE.DD_PixelAspectRatio, pixelAspectRatio);
	} 


	/**
	 * PS 3.3-C.8.6.2
	 */
	private void addSCImageIE() {
		ddo.setString(DcmDDE.DD_DateOfSecondaryCapture, dateOfSecondaryCapture);
		ddo.setString(DcmDDE.DD_TimeOfSecondaryCapture, timeOfSecondaryCapture);
	} 


	/**
	 * PS 3.3-C.12.1 & IV.B.5
	 */
	private void addSOPCommon() {
		ddo.setString(DcmDDE.DD_SOPClassUID, sopClassUID);
		ddo.setString(DcmDDE.DD_SOPInstanceUID, orgRootUID + DcmUID.SOP_UID_SUFFIX + "." + dateAndTimeSuffix);
	}


	/**
	 * PS 3.3-C.11.2
	 */
	private void addVOILUTModule() {
		ddo.setString(DcmDDE.DD_WindowCenter, windowCenter.trim());
		ddo.setString(DcmDDE.DD_WindowWidth, windowWidth.trim());
	} 


	/**
	 * PS 3.3-C.7.6.3 (Optional)
	 */
	private void addLUTModule() {
		ddo.push(new DcmValue(DcmDDE.getDictGroup(DcmDDE.DD_RedPaletteColorLookupTableDescriptor), DcmDDE.getDictElement(DcmDDE.DD_RedPaletteColorLookupTableDescriptor), redPaletteColorLookupTableDescriptor));
		ddo.push(new DcmValue(DcmDDE.getDictGroup(DcmDDE.DD_GreenPaletteColorLookupTableDescriptor), DcmDDE.getDictElement(DcmDDE.DD_GreenPaletteColorLookupTableDescriptor), greenPaletteColorLookupTableDescriptor));
		ddo.push(new DcmValue(DcmDDE.getDictGroup(DcmDDE.DD_BluePaletteColorLookupTableDescriptor), DcmDDE.getDictElement(DcmDDE.DD_BluePaletteColorLookupTableDescriptor), bluePaletteColorLookupTableDescriptor));
		ddo.push(new DcmValue(DcmDDE.getDictGroup(DcmDDE.DD_RedPaletteColorLookupTableData), DcmDDE.getDictElement(DcmDDE.DD_RedPaletteColorLookupTableData), redPaletteColorLookupTableData));
		ddo.push(new DcmValue(DcmDDE.getDictGroup(DcmDDE.DD_GreenPaletteColorLookupTableData), DcmDDE.getDictElement(DcmDDE.DD_GreenPaletteColorLookupTableData), greenPaletteColorLookupTableData));
		ddo.push(new DcmValue(DcmDDE.getDictGroup(DcmDDE.DD_BluePaletteColorLookupTableData), DcmDDE.getDictElement(DcmDDE.DD_BluePaletteColorLookupTableData), bluePaletteColorLookupTableData));
	} 


	/**
	 * Besetzt in der Klassen-Instanz die Felder fuer das "Image Pixel Module"
	 * (PS 3.3-C.7.6.3) fuer ein 8 Bit Graustufenbild.
	 * @param   pixel8    Das Byte-Array mit den Pixelwerten
	 * @param   nRows     Anzahl der Zeilen im Bild
	 * @param   nColumns  Anzahl der Spalten im Bild
	 */
	public void set8BitGrayImage(byte[] pixel8, int nRows, int nColumns) {
		samplesPerPixel = 1;
		photometricInterpretation = "MONOCHROME2";
		rows = nRows;
		columns = nColumns;
		bitsAllocated = 8;
		bitsStored = 8;
		highBit = 7;
		pixelRepresentation = 0;	// unsigned integer

		pixelData = new byte[pixel8.length];
		for (int i = 0; i < pixel8.length; i++) {
			pixelData[i] = pixel8[i];
		}
	} 


	/**
	 * Besetzt in der Klassen-Instanz die Felder fuer das "Image Pixel Module"
	 * (PS 3.3-C.7.6.3) fuer ein 8 Bit Farb-Palettenbild.
	 * @param   pixel8    Das Byte-Array mit den Pixelwerten (Indizes in die
	 * Farbtabelle des cModels
	 * @param   nRows     Anzahl der Zeilen im Bild
	 * @param   nColumns  Anzahl der Spalten im Bild
	 * @param   cModel    Das Farbmodell mit den Farbwerten fr den 8 Bit Index
	 */
	public void set8BitColorImage(byte[] pixel8, int nRows, int nColumns, ColorModel cModel) {
		byte[]	descriptor;
		int			mapSize;
		byte[]	cTab;
		int			i, k;

		samplesPerPixel = 1;
		photometricInterpretation = "PALETTE COLOR";
		rows = nRows;
		columns = nColumns;
		bitsAllocated = 8;
		bitsStored = 8;
		highBit = 7;
		pixelRepresentation = 0;	// unsigned integer

		pixelData = new byte[pixel8.length];
		for (i = 0; i < pixel8.length; i++) {
			pixelData[i] = pixel8[i];
		}

		includeLUTModule = true;

		mapSize = ((IndexColorModel) cModel).getMapSize();
		byte	mapSizeL = (byte) (mapSize & 0xff);
		byte	mapSizeH = (byte) (mapSize >>> 8);

		descriptor = new byte[6];
		descriptor[1] = (byte) mapSizeH;
		descriptor[0] = (byte) mapSizeL;
		descriptor[3] = (byte) 0;
		descriptor[2] = (byte) 0;
		descriptor[5] = (byte) 0;
		descriptor[4] = (byte) 8;

		redPaletteColorLookupTableDescriptor = descriptor;
		greenPaletteColorLookupTableDescriptor = descriptor;
		bluePaletteColorLookupTableDescriptor = descriptor;

		redPaletteColorLookupTableData = new byte[2 * mapSize];
		greenPaletteColorLookupTableData = new byte[2 * mapSize];
		bluePaletteColorLookupTableData = new byte[2 * mapSize];

		cTab = new byte[mapSize];
		((IndexColorModel) cModel).getReds(cTab);
		i = 0;
		k = 0;
		while (i < cTab.length) {
			redPaletteColorLookupTableData[k++] = cTab[i++];
			redPaletteColorLookupTableData[k++] = 0;
		} 

		((IndexColorModel) cModel).getGreens(cTab);
		i = 0;
		k = 0;
		while (i < cTab.length) {
			greenPaletteColorLookupTableData[k++] = cTab[i++];
			greenPaletteColorLookupTableData[k++] = 0;
		} 

		((IndexColorModel) cModel).getBlues(cTab);
		i = 0;
		k = 0;
		while (i < cTab.length) {
			bluePaletteColorLookupTableData[k++] = cTab[i++];
			bluePaletteColorLookupTableData[k++] = 0;
		} 
	} 


	/**
	 * Besetzt in der Klassen-Instanz die Felder fuer das "Image Pixel Module"
	 * (PS 3.3-C.7.6.3) fuer ein 16 Bit Graustufenbild ohne Vorzeichen.
	 * @param   pixel16   Das Byte-Array mit den Pixelwerten
	 * @param   nRows     Anzahl der Zeilen im Bild
	 * @param   nColumns  Anzahl der Spalten im Bild
	 * @param   bits      Anzahl der signifikanten Bits in pixel16
	 */
	public void set16UBitGrayImage(short[] pixel16, int nRows, int nColumns, int bits) {
		pixelRepresentation = 0;	// unsigned integer
		set16BitGrayImage(pixel16, nRows, nColumns, bits);
	} 


	/**
	 * Besetzt in der Klassen-Instanz die Felder fuer das "Image Pixel Module"
	 * (PS 3.3-C.7.6.3) fuer ein 16 Bit Graustufenbild mit Vorzeichen.
	 * @param   pixel16   Das Byte-Array mit den Pixelwerten
	 * @param   nRows     Anzahl der Zeilen im Bild
	 * @param   nColumns  Anzahl der Spalten im Bild
	 * @param   bits      Anzahl der signifikanten Bits in pixel16
	 */
	public void set16SBitGrayImage(short[] pixel16, int nRows, int nColumns, int bits) {
		pixelRepresentation = 1;	// 2's complement
		set16BitGrayImage(pixel16, nRows, nColumns, bits);
	} 


	/**
	 * Besetzt in der Klassen-Instanz die Felder fuer das "Image Pixel Module"
	 * (PS 3.3-C.7.6.3) fuer ein 16 Bit Graustufenbild ohne Vorzeichen.
	 * @param   pixel16   Das Byte-Array mit den Pixelwerten
	 * @param   nRows     Anzahl der Zeilen im Bild
	 * @param   nColumns  Anzahl der Spalten im Bild
	 * @param   bits      Anzahl der signifikanten Bits in pixel16
	 */
	private void set16BitGrayImage(short[] pixel16, int nRows, int nColumns, int bits) {
		short p;
		int		ib;
		int		ip;

		samplesPerPixel = 1;
		photometricInterpretation = "MONOCHROME2";
		rows = nRows;
		columns = nColumns;
		bitsAllocated = 16;
		bitsStored = bits;
		highBit = bits - 1;

		byte[]	buf = new byte[nRows * nColumns * 2];
		short		bitMask = (short) (Math.pow(2, bits) - 1);

		ip = 0;
		ib = 0;
		while (ip < pixel16.length) {
			p = (short) (pixel16[ip++] & bitMask);
			buf[ib++] = (byte) (p & 0xff);
			buf[ib++] = (byte) ((p >>> 8) & 0xff);
		} 
		pixelData = buf;
	} 

}


/*--- formatting done in "My Own Convention" style on 04-30-2000 ---*/

