/*--- formatted by Jindent 2.1, (www.c-lab.de/~jindent) ---*/

/*
 * JIGL--Java Imaging and Graphics Library
 * Copyright (C)1999 Brigham Young University
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 * A copy of the GNU Library General Public Licence is contained in
 * /jigl/licence.txt
 */
package jm.jigl; // Modifizierte Paketstruktur [JM]
import jm.jigl.*;

// import jigl.math.*;

import java.io.*;


/**
 * SignalOutputStream outputs signals to a file that JIGL can read.  To accomadate other programs
 * to display signals, JIGL also support the ".dat" extension.  Normally JIGL will automatically
 * detect what type to output (DiscreteSignal, RealSignal, ect.)  However in order to output a DAT
 * file, the filename passed to the constuctor MUST have a ".dat" extension.  For example you might
 * use the name -- <i>myfile.dat</i>
 * @version 1.3, Modifikation der urspruenglichen jigl Paketstruktur durch Jens Martin
 * @see jigl.signal.SignalOutputStream
 */

public class SignalOutputStream {


	/**
	 * Data to dump
	 */
	protected BufferedOutputStream	data;


	/**
	 * Number of dimensions
	 */
	protected int										ndims;


	/**
	 * Height of the signal
	 */
	protected int										X;

	private boolean									toDat = false;


	/**
	 * Default SignalOutputStream, does nothing
	 */
	public SignalOutputStream() {}


	/**
	 * Creates an SignalOutputStream from a filename
	 * @param fn filename
	 */
	public SignalOutputStream(String fn) throws IOException {
		FileOutputStream	f = new FileOutputStream(fn);

		data = new BufferedOutputStream(f);
		if (fn.endsWith(".dat")) {
			toDat = true;
		}
	}


	/**
	 * Creates an SignalOutputStream from another SignalOutputStream
	 */
	public SignalOutputStream(SignalOutputStream stream) {
		data = stream.data;
		ndims = stream.ndims;
		X = stream.X;
	}


	/**
	 * Returns the number of dimensions
	 */
	public int ndims() {
		return ndims;
	}


	/**
	 * Returns the width
	 */
	public int X() {
		return X;
	}


	/**
	 * Opens a file
	 * @param filename
	 */
	public boolean open(String fn) throws IOException {

		FileOutputStream	f = new FileOutputStream(fn);

		data = new BufferedOutputStream(f);
		return true;
	}


	/**
	 * Closes the file
	 * @param filename
	 */
	public void close() throws IOException {
		data.close();
	}


	/**
	 * Writes a JIGL signal to a file
	 * @param im JIGL signal
	 */
	public void write(jm.jigl.Signal im) throws UnknownSignalTypeException, ColorModelNotSupportedException, ColorModelUnknownException, IOException {
		if (im instanceof DiscreteSignal) {
			write((DiscreteSignal) im);
		} else if (im instanceof RealSignal) {
			write((RealSignal) im);
		} else {
			throw new UnknownSignalTypeException();
		}
	}


	/**
	 * Writes a GraySignal to a file
	 * @param im the GraySignal
	 */
	public void write(DiscreteSignal s) throws IOException {

		if (toDat == false) {
			X = s.length();

			// convert to byte size
			DiscreteSignal	tmps = (DiscreteSignal) s.copy();

			tmps.byteSize();

			// write PGM in raw format

			writeRawPDSHeader(X);
			for (int x = 0; x < X; x++) {
				data.write((byte) tmps.get(x));
			}
		} else {
			Integer f = null;

			for (int x = 0; x < s.length(); x++) {
				f = new Integer(s.get(x));
				writeString(data, f.toString() + "\n");
			}
		}
	}






	/**
	 * Writes a RealSignal to a file
	 */
	public void write(RealSignal s) throws IOException {


		if (toDat == false) {

			X = s.length();

			// convert to byte size
			RealSignal	tmps = (RealSignal) s.copy();

			tmps.byteSize();

			// write PGM in raw format

			writeRawPRSMHeader(X);
			for (int x = 0; x < X; x++) {
				data.write((byte) tmps.get(x));
			}
		} else {
			Float f = null;

			for (int x = 0; x < s.length(); x++) {
				f = new Float(s.get(x));
				writeString(data, f.toString() + "\n");
			}
		}


	}


	/**
	 * Method declaration
	 *
	 *
	 * @param width
	 *
	 * @throws IOException
	 *
	 * @see
	 */
	void writeRawPDSHeader(int width) throws IOException {
		writeString(data, "S3\n");
		writeString(data, width + "\n");
		writeString(data, "255\n");
	}


	/**
	 * Method declaration
	 *
	 *
	 * @param width
	 *
	 * @throws IOException
	 *
	 * @see
	 */
	void writeRawPRSMHeader(int width) throws IOException {
		writeString(data, "S4\n");
		writeString(data, width + "\n");
		writeString(data, "255\n");
	}


	/**
	 * Method declaration
	 *
	 *
	 * @param out
	 * @param str
	 *
	 * @throws IOException
	 *
	 * @see
	 */
	static void writeString(OutputStream out, String str) throws IOException {

		out.write(str.getBytes());

	}

}



/*--- formatting done in "My Own Convention" style on 08-08-2000 ---*/

