/*--- formatted by Jindent 2.1, (www.c-lab.de/~jindent) ---*/

/*
 * The package jm.kidney is part of the Renal Function Project
 * for analysis of dynamic contrast medium evalutions MRT-Images
 * of the Kidneys.
 *
 * Copyright (C) 1999 / 2000 Jens Martin
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License 2
 * as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

package jm.kidney;

import jm.extension.*;
import jm.util.*;		// 2000.7.31 tha: Fuer GridBagConstriants2

import ij.*;

import java.io.*;
import java.awt.*;
import java.awt.event.*;

import javax.swing.*;
import javax.swing.event.*;
import javax.swing.border.*;


/**
 * Diese Klasse stellt einen Dialog zur Eingabe der Voreinstellungen dar.
 * Auf einem JTabbedPane werden drei verschiedene Tabs dargestellt:
 * Cepstrum Setting - Einstellungen zur Cepstrum-Filterung
 * SIT Settings - Einstellungen zur Berechnung der SIT-Kurven
 * Other Settings - Sonstige Einstellungen
 * <BR> Der Aufrug erfolgt aus der Klasse AnalysisDialog
 *
 * @see jm.kidney.AnalysisDialog
 * @version  2.0, 14/02/2000
 * @author   Jens Martin
 */
public class PresetsDialog extends JFrame implements ActionListener, MouseListener, ChangeListener, KeyListener, FocusListener {

	public int							minDiffSwitch = 10;
	public int							ignoreFirst = 3;
	public double						classificationRange = 0.5;
	public double						cepstrumAccuracy = 2.0;
	public int							threadPriority = java.lang.Thread.MAX_PRIORITY;
	public boolean					rasterTransformation = false;
	public boolean					verboseText = false;
	public boolean					newWindow = false;
	public boolean					interpolateMissing = false;
	public boolean					showNullValues = false;
	public boolean					secondChance = true;
	public boolean					writeLog = false;
	public boolean					minDiff = true;
	public boolean					verboseSIT = false;
	public boolean					splashScreen = true;

	public boolean					okPressed = false;
	private AnalysisDialog	analysisDialog = null;

	JTabbedPane							jTabbedPane = new JTabbedPane();

	// Member-Variablen


	GridBagLayout						gridBagLayout3 = new GridBagLayout();
	ExJButton								jButtonCancel = new ExJButton();
	ExJButton								jButtonOk = new ExJButton();
	JPanel									jPanelButtons = new JPanel();
	GridBagLayout						gridBagLayout4 = new GridBagLayout();
	JPanel									jTabCepstrum = new JPanel();
	JPanel									jTabCPU = new JPanel();
	JLabel									jLabelIgnoreImages = new JLabel();
	JRadioButton						checkboxMinimumDiff = new JRadioButton();
	GridBagLayout						gridBagLayout1 = new GridBagLayout();
	GridBagLayout						gridBagLayout2 = new GridBagLayout();
	JPanel									jTabSIT = new JPanel();
	JTextField							jTextFieldIgnore = new JTextField();
	JLabel									jLabelIgnore = new JLabel();
	JPanel									jPanelMinimumDiff = new JPanel();
	JTextField							jTextFieldMinDiff = new JTextField();
	GridBagLayout						gridBagLayout5 = new GridBagLayout();
	JLabel									jLabelIgnoreImages1 = new JLabel();
	JRadioButton						checkboxSecondChance = new JRadioButton();
	JRadioButton						checkboxRasterTransform = new JRadioButton();
	JRadioButton						checkboxNewWindow = new JRadioButton();
	JRadioButton						checkboxVerboseText = new JRadioButton();
	JPanel									jPanelIgnore = new JPanel();
	GridBagLayout						gridBagLayout6 = new GridBagLayout();
	JRadioButton						checkboxShowNull = new JRadioButton();
	JRadioButton						checkboxInterpolateMissing = new JRadioButton();
	JSlider									jSliderClassificationRange = new JSlider();
	JLabel									jLabelClassificationRange = new JLabel();
	JRadioButton						jRadioButtonCPUHigh = new JRadioButton();
	GridBagLayout						gridBagLayout7 = new GridBagLayout();
	JLabel									jLabelThreadState = new JLabel();
	JRadioButton						jRadioButtonCPUMid = new JRadioButton();
	JRadioButton						jRadioButtonCPULow = new JRadioButton();
	ButtonGroup							cpuPriorityButtons = new ButtonGroup();
	ExJButton								jButtonReset = new ExJButton();
	JSlider									jSliderDefaultAccuracy = new JSlider();
	JLabel									jLabelDefaultAccuracy = new JLabel();
	JRadioButton						checkboxWriteLog = new JRadioButton();
	JRadioButton						checkboxVerboseSIT = new JRadioButton();
	JRadioButton						checkboxSplashScreen = new JRadioButton();


	/**
	 * Der Konstruktor.
	 * @param    dlg    Referenz auf das Hautpfenster des Plugins zur renalen Funktionsanalyse
	 */
	public PresetsDialog(AnalysisDialog dlg) {
		super();
		try {
			analysisDialog = dlg;
			jbInit();
			if (!loadPresetFile()) {
				setDefaultValues(false);
			}
			guiInit();
		} catch (Exception e) {
			e.printStackTrace();
		}
	}


	/**
	 * Initialisierung der Objekte des Fensters.
	 * Von JBuilder automatisch eingefuegt.
	 */
	private void jbInit() throws Exception {

		// Nun noch Groesse und Aussehen festlegen und anzeigen

		this.setForeground(Color.black);
		this.setBackground(Color.black);
		this.getContentPane().setLayout(gridBagLayout3);
		jTabbedPane.setBackground(Color.gray);
		jTabbedPane.setFont(new Font("Dialog", 1, 11));
		jTabbedPane.setToolTipText("");
		jTabbedPane.setMinimumSize(new Dimension(184, 111));
		jButtonCancel.setText("Cancel");
		jButtonCancel.setMargin(new Insets(2, 24, 2, 24));

		// jButtonCancel.setLabel("Cancel");
		jButtonCancel.setForeground(Color.lightGray);
		jButtonCancel.setBackground(Color.black);
		jButtonCancel.setFont(new Font("Dialog", 1, 11));
		jButtonCancel.setToolTipText("close Window (don\'t save changed Presets) ");
		jButtonOk.setText("OK");
		jButtonOk.setMargin(new Insets(2, 34, 2, 34));

		// jButtonOk.setLabel("OK");
		jButtonOk.setForeground(Color.lightGray);
		jButtonOk.setBackground(Color.black);
		jButtonOk.setFont(new Font("Dialog", 1, 11));
		jButtonOk.setToolTipText("close Window (save changed Presets) ");
		jPanelButtons.setForeground(Color.lightGray);
		jPanelButtons.setBackground(Color.black);
		jTabCepstrum.setPreferredSize(new Dimension(10, 10));
		jTabCepstrum.setForeground(Color.lightGray);
		jTabCepstrum.setBackground(Color.black);
		jTabCepstrum.setMinimumSize(new Dimension(200, 100));
		jTabCPU.setForeground(Color.lightGray);
		jTabCPU.setBackground(Color.black);
		jTabCPU.setLayout(gridBagLayout7);
		jTabCepstrum.setLayout(gridBagLayout1);
		jLabelIgnoreImages.setText("images");
		jLabelIgnoreImages.setForeground(Color.lightGray);
		jLabelIgnoreImages.setBackground(Color.black);
		jLabelIgnoreImages.setFont(new Font("Dialog", 1, 11));
		jLabelIgnoreImages.setOpaque(true);
		checkboxMinimumDiff.setRequestFocusEnabled(false);
		checkboxMinimumDiff.setForeground(Color.lightGray);
		checkboxMinimumDiff.setFont(new Font("Dialog", 1, 11));
		checkboxMinimumDiff.setOpaque(false);
		checkboxMinimumDiff.setFocusPainted(false);
		checkboxMinimumDiff.setBackground(Color.darkGray);

		// checkboxMinimumDiff.setLabel("Switch to Minimum Difference after");
		checkboxMinimumDiff.setText("Switch to Minimum Difference after");
		jTabSIT.setBackground(Color.black);
		jTextFieldMinDiff.setHorizontalAlignment(JTextField.RIGHT);

		// jTextFieldMinDiff.setSelectionStart(3);
		jTextFieldMinDiff.setText("  10");
		jTextFieldMinDiff.setForeground(Color.white);
		jTextFieldMinDiff.setBackground(Color.darkGray);
		jTextFieldMinDiff.setFont(new Font("Monospaced", 0, 12));
		jTextFieldMinDiff.setToolTipText("klick to edit");
		jLabelIgnoreImages1.setOpaque(true);
		checkboxSecondChance.setFont(new Font("Dialog", 1, 11));
		checkboxSecondChance.setOpaque(false);
		checkboxSecondChance.setFocusPainted(false);
		checkboxSecondChance.setRequestFocusEnabled(false);
		checkboxSecondChance.setForeground(Color.lightGray);

		// checkboxSecondChance.setLabel("Second Chance if fails (recommended)");
		checkboxSecondChance.setText("Second Chance if fails (recommended)");
		checkboxSecondChance.setBackground(Color.darkGray);
		checkboxRasterTransform.setRequestFocusEnabled(false);
		checkboxRasterTransform.setEnabled(false);
		checkboxRasterTransform.setForeground(Color.white);
		checkboxRasterTransform.setFont(new Font("Dialog", 1, 11));
		checkboxRasterTransform.setOpaque(false);
		checkboxRasterTransform.setFocusPainted(false);

		// checkboxRasterTransform.setLabel("Use Rastertransformation for Fineadjustement");
		checkboxRasterTransform.setText("Use Rastertransformation for Fineadjustement");
		checkboxRasterTransform.setBackground(Color.darkGray);
		checkboxNewWindow.setRequestFocusEnabled(false);
		checkboxNewWindow.setEnabled(false);
		checkboxNewWindow.setForeground(Color.lightGray);
		checkboxNewWindow.setFont(new Font("Dialog", 1, 11));
		checkboxNewWindow.setOpaque(false);
		checkboxNewWindow.setFocusPainted(false);

		// checkboxNewWindow.setLabel("Show corrected Series in new Window");
		checkboxNewWindow.setText("Show corrected Series in new Window");
		checkboxNewWindow.setBackground(Color.darkGray);
		checkboxVerboseText.setFont(new Font("Dialog", 1, 11));
		checkboxVerboseText.setOpaque(false);
		checkboxVerboseText.setFocusPainted(false);
		checkboxVerboseText.setRequestFocusEnabled(false);
		checkboxVerboseText.setForeground(Color.lightGray);

		// checkboxVerboseText.setLabel("Verbose Results in ImageJ-Window");
		checkboxVerboseText.setText("Verbose Results in ImageJ-Window");
		checkboxVerboseText.setBackground(Color.darkGray);
		jPanelIgnore.setBackground(Color.black);
		checkboxShowNull.setRequestFocusEnabled(false);
		checkboxShowNull.setForeground(Color.lightGray);
		checkboxShowNull.setFont(new Font("Dialog", 1, 11));
		checkboxShowNull.setOpaque(false);
		checkboxShowNull.setFocusPainted(false);
		checkboxShowNull.setBackground(Color.darkGray);

		// checkboxShowNull.setLabel("Show Null Values");
		checkboxShowNull.setText("Show Null Values");
		checkboxInterpolateMissing.setRequestFocusEnabled(false);
		checkboxInterpolateMissing.setForeground(Color.lightGray);
		checkboxInterpolateMissing.setFont(new Font("Dialog", 1, 11));
		checkboxInterpolateMissing.setOpaque(false);
		checkboxInterpolateMissing.setSelected(true);
		checkboxInterpolateMissing.setFocusPainted(false);

		// checkboxInterpolateMissing.setLabel("Interpolate missing Values");
		checkboxInterpolateMissing.setText("Interpolate missing Values");
		checkboxInterpolateMissing.setBackground(Color.darkGray);
		jSliderClassificationRange.setValue(5);
		jSliderClassificationRange.setMaximum(9);
		jSliderClassificationRange.setMinimum(2);
		jSliderClassificationRange.setMajorTickSpacing(1);
		jSliderClassificationRange.setMinorTickSpacing(1);
		jSliderClassificationRange.setRequestFocusEnabled(false);
		jSliderClassificationRange.setForeground(Color.lightGray);
		jSliderClassificationRange.setBackground(Color.black);
		jLabelClassificationRange.setText("Classification Range: 0.5 - 1.5");
		jLabelClassificationRange.setForeground(Color.lightGray);
		jLabelClassificationRange.setBackground(Color.black);
		jLabelClassificationRange.setFont(new Font("Dialog", 1, 11));
		jLabelClassificationRange.setOpaque(true);
		jRadioButtonCPUHigh.setRequestFocusEnabled(false);
		jRadioButtonCPUHigh.setText("high (recommended)");
		jRadioButtonCPUHigh.setForeground(Color.lightGray);
		jRadioButtonCPUHigh.setBackground(new Color(0, 161, 0));
		jRadioButtonCPUHigh.setFont(new Font("Dialog", 1, 11));
		jRadioButtonCPUHigh.setOpaque(false);
		jRadioButtonCPUHigh.setFocusPainted(false);
		jLabelThreadState.setText("Thread CPU Priority");
		jLabelThreadState.setForeground(Color.lightGray);
		jLabelThreadState.setBackground(Color.black);
		jLabelThreadState.setFont(new Font("Dialog", 1, 11));
		jLabelThreadState.setOpaque(true);
		jRadioButtonCPUMid.setRequestFocusEnabled(false);
		jRadioButtonCPUMid.setText("mid");
		jRadioButtonCPUMid.setForeground(Color.lightGray);

		// jRadioButtonCPUMid.setLabel("mid");
		jRadioButtonCPUMid.setBackground(new Color(200, 145, 0));
		jRadioButtonCPUMid.setFont(new Font("Dialog", 1, 11));
		jRadioButtonCPUMid.setOpaque(false);
		jRadioButtonCPUMid.setFocusPainted(false);
		jRadioButtonCPULow.setRequestFocusEnabled(false);
		jRadioButtonCPULow.setText("low");
		jRadioButtonCPULow.setForeground(Color.lightGray);

		// jRadioButtonCPULow.setLabel("low");
		jRadioButtonCPULow.setBackground(new Color(128, 0, 0));
		jRadioButtonCPULow.setFont(new Font("Dialog", 1, 11));
		jRadioButtonCPULow.setOpaque(false);
		jRadioButtonCPULow.setFocusPainted(false);
		jButtonReset.setText("Default Values");
		jButtonReset.setMargin(new Insets(2, 4, 2, 4));
		jButtonReset.setForeground(Color.lightGray);
		jSliderDefaultAccuracy.setValue(20);
		jSliderDefaultAccuracy.setMaximum(40);
		jSliderDefaultAccuracy.setMinimum(10);
		jSliderDefaultAccuracy.setRequestFocusEnabled(false);
		jSliderDefaultAccuracy.setForeground(Color.lightGray);
		jSliderDefaultAccuracy.setBackground(Color.black);
		jSliderDefaultAccuracy.setFont(new Font("Dialog", 1, 11));
		jLabelDefaultAccuracy.setText("-   default Cepstrum accuracy 1.0  +");
		jLabelDefaultAccuracy.setForeground(Color.lightGray);
		jLabelDefaultAccuracy.setBackground(Color.black);
		jLabelDefaultAccuracy.setFont(new Font("Dialog", 1, 11));
		checkboxWriteLog.setFont(new Font("Dialog", 1, 11));
		checkboxWriteLog.setOpaque(false);
		checkboxWriteLog.setSelected(true);
		checkboxWriteLog.setFocusPainted(false);
		checkboxVerboseSIT.setForeground(Color.lightGray);
		checkboxVerboseSIT.setFont(new Font("Dialog", 1, 11));
		checkboxVerboseSIT.setOpaque(false);
		checkboxVerboseSIT.setSelected(true);
		checkboxVerboseSIT.setFocusPainted(false);
		checkboxSplashScreen.setFont(new Font("Dialog", 1, 11));
		checkboxSplashScreen.setOpaque(false);
		checkboxSplashScreen.setSelected(true);
		checkboxSplashScreen.setFocusPainted(false);
		checkboxSplashScreen.setRequestFocusEnabled(false);
		checkboxSplashScreen.setForeground(Color.lightGray);

		// checkboxSplashScreen.setLabel("Show Splashscreen");
		checkboxSplashScreen.setText("Show Splashscreen");
		checkboxSplashScreen.setBackground(Color.darkGray);

		// checkboxVerboseSIT.setLabel("Verbose Results in ImageJ-Window");
		checkboxVerboseSIT.setText("Verbose Results in ImageJ-Window");
		checkboxVerboseSIT.setBackground(Color.darkGray);
		checkboxWriteLog.setRequestFocusEnabled(false);
		checkboxWriteLog.setForeground(Color.lightGray);

		// checkboxWriteLog.setLabel("Write Results to Logfile");
		checkboxWriteLog.setText("Write Results to Logfile");
		checkboxWriteLog.setBackground(Color.black);
		jButtonReset.setBackground(Color.black);
		jButtonReset.setFont(new Font("Dialog", 1, 11));
		jButtonReset.setToolTipText("restore default Values");
		jPanelIgnore.setLayout(gridBagLayout6);
		jLabelIgnoreImages1.setFont(new Font("Dialog", 1, 11));
		jLabelIgnoreImages1.setBackground(Color.black);
		jLabelIgnoreImages1.setForeground(Color.lightGray);
		jLabelIgnoreImages1.setText("images");
		jTextFieldMinDiff.setCaretColor(Color.orange);
		jTextFieldIgnore.setHorizontalAlignment(JTextField.RIGHT);

		// jTextFieldIgnore.setSelectionStart(3);
		jTextFieldIgnore.setText("   3");
		jTextFieldIgnore.setForeground(Color.white);
		jTextFieldIgnore.setBackground(Color.darkGray);
		jTextFieldIgnore.setFont(new Font("Monospaced", 0, 12));
		jTextFieldIgnore.setToolTipText("klick to edit");
		jLabelIgnore.setText("Ignore first");
		jLabelIgnore.setForeground(Color.lightGray);
		jLabelIgnore.setBackground(Color.black);
		jLabelIgnore.setFont(new Font("Dialog", 1, 11));
		jPanelMinimumDiff.setBackground(Color.black);
		jPanelMinimumDiff.setLayout(gridBagLayout5);
		jTextFieldIgnore.setCaretColor(Color.orange);
		jTabSIT.setLayout(gridBagLayout2);
		jPanelButtons.setLayout(gridBagLayout4);

		// toggleCancelButton();
		this.getContentPane().add(jTabbedPane, new GridBagConstraints2(0, 1, 1, 1, 1.0, 1.0, GridBagConstraints.CENTER, GridBagConstraints.BOTH, new Insets(0, 0, 0, 0), 240, 180));
		jTabbedPane.addTab("Cepstrum Settings", jTabCepstrum);
		jTabCepstrum.add(jPanelMinimumDiff, new GridBagConstraints2(0, 1, 1, 1, 0.0, 0.0, GridBagConstraints.WEST, GridBagConstraints.BOTH, new Insets(0, 0, 0, 0), 0, 0));
		jPanelMinimumDiff.add(checkboxMinimumDiff, new GridBagConstraints2(0, 0, 1, 1, 0.0, 0.0, GridBagConstraints.WEST, GridBagConstraints.BOTH, new Insets(0, 0, 0, 0), 0, 0));
		jPanelMinimumDiff.add(jTextFieldMinDiff, new GridBagConstraints2(1, 0, 1, 1, 1.0, 0.0, GridBagConstraints.WEST, GridBagConstraints.HORIZONTAL, new Insets(0, 0, 0, 0), 0, 0));
		jPanelMinimumDiff.add(jLabelIgnoreImages1, new GridBagConstraints2(2, 0, 1, 1, 0.0, 0.0, GridBagConstraints.CENTER, GridBagConstraints.BOTH, new Insets(0, 5, 0, 0), 0, 0));
		jTabCepstrum.add(checkboxSecondChance, new GridBagConstraints2(0, 2, 1, 1, 0.0, 0.0, GridBagConstraints.WEST, GridBagConstraints.BOTH, new Insets(0, 0, 0, 0), 0, 0));
		jTabCepstrum.add(checkboxRasterTransform, new GridBagConstraints2(0, 3, 1, 1, 0.0, 0.0, GridBagConstraints.WEST, GridBagConstraints.BOTH, new Insets(0, 0, 0, 0), 0, 0));
		jTabCepstrum.add(checkboxNewWindow, new GridBagConstraints2(0, 4, 1, 1, 0.0, 0.0, GridBagConstraints.WEST, GridBagConstraints.BOTH, new Insets(0, 0, 0, 0), 0, 0));
		jTabCepstrum.add(checkboxVerboseText, new GridBagConstraints2(0, 5, 1, 1, 0.0, 0.0, GridBagConstraints.WEST, GridBagConstraints.BOTH, new Insets(0, 0, 0, 0), 0, 0));
		jTabCepstrum.add(jPanelIgnore, new GridBagConstraints2(0, 0, 1, 1, 0.0, 0.0, GridBagConstraints.WEST, GridBagConstraints.NONE, new Insets(0, 0, 0, 0), 0, 0));
		jPanelIgnore.add(jLabelIgnore, new GridBagConstraints2(0, 0, 1, 1, 0.0, 0.0, GridBagConstraints.WEST, GridBagConstraints.BOTH, new Insets(0, 0, 0, 7), 0, 0));
		jPanelIgnore.add(jTextFieldIgnore, new GridBagConstraints2(1, 0, 1, 1, 0.0, 0.0, GridBagConstraints.CENTER, GridBagConstraints.BOTH, new Insets(0, 0, 0, 0), 0, 0));
		jPanelIgnore.add(jLabelIgnoreImages, new GridBagConstraints2(2, 0, 1, 1, 0.0, 0.0, GridBagConstraints.WEST, GridBagConstraints.BOTH, new Insets(0, 5, 0, 0), 0, 0));
		jTabCepstrum.add(jSliderDefaultAccuracy, new GridBagConstraints2(0, 7, 1, 1, 0.0, 0.0, GridBagConstraints.WEST, GridBagConstraints.BOTH, new Insets(0, 0, 0, 98), 0, 0));
		jTabCepstrum.add(jLabelDefaultAccuracy, new GridBagConstraints2(0, 8, 1, 1, 0.0, 0.0, GridBagConstraints.WEST, GridBagConstraints.NONE, new Insets(0, 8, 0, 80), 0, 0));
		jTabCepstrum.add(checkboxWriteLog, new GridBagConstraints2(0, 6, 1, 1, 0.0, 0.0, GridBagConstraints.WEST, GridBagConstraints.BOTH, new Insets(0, 0, 0, 0), 0, 0));
		jTabbedPane.addTab("SIT Settings", jTabSIT);
		jTabSIT.add(checkboxShowNull, new GridBagConstraints2(0, 0, 1, 1, 0.0, 0.0, GridBagConstraints.WEST, GridBagConstraints.BOTH, new Insets(0, 0, 0, 0), 0, 0));
		jTabSIT.add(checkboxInterpolateMissing, new GridBagConstraints2(0, 1, 1, 1, 0.0, 0.0, GridBagConstraints.WEST, GridBagConstraints.NONE, new Insets(0, 0, 0, 0), 0, 0));
		jTabSIT.add(jSliderClassificationRange, new GridBagConstraints2(0, 3, 1, 1, 0.0, 0.0, GridBagConstraints.CENTER, GridBagConstraints.BOTH, new Insets(0, 0, 0, 0), 0, 0));
		jTabSIT.add(jLabelClassificationRange, new GridBagConstraints2(0, 4, 1, 1, 0.0, 0.0, GridBagConstraints.CENTER, GridBagConstraints.NONE, new Insets(0, 0, 0, 0), 0, 0));
		jTabSIT.add(checkboxVerboseSIT, new GridBagConstraints2(0, 2, 1, 1, 0.0, 0.0, GridBagConstraints.WEST, GridBagConstraints.BOTH, new Insets(0, 0, 0, 0), 0, 0));
		this.getContentPane().add(jPanelButtons, new GridBagConstraints2(0, 2, 1, 1, 0.0, 0.0, GridBagConstraints.CENTER, GridBagConstraints.BOTH, new Insets(0, 0, 0, 0), 0, 0));
		jPanelButtons.add(jButtonCancel, new GridBagConstraints2(1, 0, 1, 1, 0.0, 0.0, GridBagConstraints.CENTER, GridBagConstraints.BOTH, new Insets(5, 5, 5, 5), 0, 0));
		jPanelButtons.add(jButtonOk, new GridBagConstraints2(2, 0, 1, 1, 0.0, 0.0, GridBagConstraints.CENTER, GridBagConstraints.BOTH, new Insets(5, 5, 5, 5), 0, 0));
		jPanelButtons.add(jButtonReset, new GridBagConstraints2(0, 0, 1, 1, 0.0, 0.0, GridBagConstraints.CENTER, GridBagConstraints.NONE, new Insets(5, 5, 5, 5), 0, 0));
		jTabbedPane.addTab("Other Settings", jTabCPU);
		jTabCPU.add(jRadioButtonCPULow, new GridBagConstraints2(0, 3, 1, 1, 0.0, 0.0, GridBagConstraints.CENTER, GridBagConstraints.BOTH, new Insets(0, 0, 0, 0), 0, 0));
		jTabCPU.add(jRadioButtonCPUMid, new GridBagConstraints2(0, 2, 1, 1, 0.0, 0.0, GridBagConstraints.CENTER, GridBagConstraints.BOTH, new Insets(0, 0, 0, 0), 0, 0));
		jTabCPU.add(jRadioButtonCPUHigh, new GridBagConstraints2(0, 1, 1, 1, 0.0, 0.0, GridBagConstraints.WEST, GridBagConstraints.NONE, new Insets(0, 0, 0, 0), 0, 0));
		jTabCPU.add(jLabelThreadState, new GridBagConstraints2(0, 0, 1, 1, 0.0, 0.0, GridBagConstraints.WEST, GridBagConstraints.BOTH, new Insets(0, 0, 3, 0), 0, 0));
		jTabCPU.add(checkboxSplashScreen, new GridBagConstraints2(0, 4, 1, 1, 0.0, 0.0, GridBagConstraints.WEST, GridBagConstraints.BOTH, new Insets(20, 0, 0, 0), 0, 0));

	}


	/**
	 * Zusaetzliche Formatierungen von Objekten der GUI
	 */
	public void guiInit() {

		// jTabbedPane.add("Cepstrum Settings",jTabCepstrum);
		// jTabbedPane.add("SIT Settings",jTabSIT);
		cpuPriorityButtons.add(jRadioButtonCPUHigh);
		cpuPriorityButtons.add(jRadioButtonCPUMid);
		cpuPriorityButtons.add(jRadioButtonCPULow);

		checkboxMinimumDiff.addMouseListener(this);
		checkboxMinimumDiff.addKeyListener(this);

		checkboxInterpolateMissing.addMouseListener(this);
		checkboxNewWindow.addMouseListener(this);
		checkboxRasterTransform.addMouseListener(this);
		checkboxSecondChance.addMouseListener(this);
		checkboxShowNull.addMouseListener(this);
		checkboxVerboseText.addMouseListener(this);
		checkboxWriteLog.addMouseListener(this);

		jRadioButtonCPUHigh.addActionListener(this);
		jRadioButtonCPUMid.addActionListener(this);
		jRadioButtonCPULow.addActionListener(this);

		jTextFieldIgnore.addActionListener(this);
		jTextFieldIgnore.addFocusListener(this);

		jTextFieldMinDiff.addActionListener(this);
		jTextFieldMinDiff.addFocusListener(this);

		jButtonOk.addActionListener(this);
		jButtonCancel.addActionListener(this);
		jButtonReset.addActionListener(this);
		jSliderClassificationRange.addChangeListener(this);
		jSliderDefaultAccuracy.addChangeListener(this);
		this.setTitle(" Presets");
		this.pack();
		this.setResizable(false);
		jTabbedPane.setSelectedIndex(1);
		this.repaint();
		IJ.wait(100);
		jTabbedPane.setSelectedIndex(2);
		this.repaint();
		IJ.wait(100);
		jTabbedPane.setSelectedIndex(1);
		this.repaint();
		IJ.wait(100);
		jTabbedPane.setSelectedIndex(0);
		this.repaint();
		IJ.wait(100);

		// jTextFieldIgnore.setText("   3");
		// jTextFieldMinDiff.setText("  10");
		try {
			UIManager.setLookAndFeel(UIManager.getSystemLookAndFeelClassName());
		} catch (Exception e) {
			System.err.println("Can't set GUI to look and feel: " + e);
		}

	}


	/**
	 * Routine zur Nachrichtenbehandung von Mausklicks und Tastatureingaben (Return)
	 * @param evt uebergebener ActionEvent
	 */
	public void actionPerformed(ActionEvent evt) {
		Object	src = evt.getSource();

		if (src instanceof JButton) {
			if (jRadioButtonCPUHigh.isSelected()) {
				threadPriority = java.lang.Thread.MAX_PRIORITY;
			}
			if (jRadioButtonCPUMid.isSelected()) {
				threadPriority = java.lang.Thread.NORM_PRIORITY;
			}
			if (jRadioButtonCPULow.isSelected()) {
				threadPriority = java.lang.Thread.MIN_PRIORITY;

			}
			if (src == jButtonOk) {
				okPressed = true;
				this.savePresetFile();
				this.setVisible(false);		// hide();
				analysisDialog.presetsDialogClosed();
			}

			if (src == jButtonCancel) {
				okPressed = false;

				// this.savePresetFile();
				this.setVisible(false);		// hide();
				analysisDialog.presetsDialogClosed();
			}

			if (src == jButtonReset) {
				setDefaultValues(true);
			}
		}


		if (src instanceof JTextField) {
			handleTextFieldInputs((JTextField) src);

		}
	}


	/**
	 * Hilfsroutine zur Formatierung des in Texteingabefeldern dargestellten Textes.
	 * @param tf JTextField Eingabefeld, das ueberprueft werden soll.
	 */
	private void handleTextFieldInputs(JTextField tf) {
		JTextField	jTextField = jTextFieldIgnore;

		if (tf == jTextFieldMinDiff) {
			jTextField = jTextFieldMinDiff;
		}
		String	input = jTextField.getText();

		int			pos = -1;

		try {
			pos = analyseNumericInput(input);
			if (pos > -1) {
				IJ.write("war wohl falsche Eingabe");
				jTextField.setCaretPosition(pos);
				jTextField.setSelectionStart(pos);
				jTextField.setSelectionEnd(pos + 1);
			}
		} catch (Exception e) {}
		if (pos == -1) {
			int newValue = s2i(input);	// = (Integer.valueOf(input)).intValue();

			// IJ.write("Eingabe anscheinend richtig gewesen: " + newValue);
			if (tf == jTextFieldMinDiff) {

				// IJ.write("...und Min Diff " + newValue);
				if (newValue < 12) {
					JOptionPane.showMessageDialog(null, "Please enter a value greater than 12 !", " Value to small", JOptionPane.ERROR_MESSAGE);
				} else {
					minDiffSwitch = newValue;
				}													// (Integer.valueOf(input)).intValue();
 			 } else {

				// IJ.write("...und ignoreFirst " + newValue);
				if (newValue < 0 || newValue > 10) {
					JOptionPane.showMessageDialog(null, "Please enter a value between 0 and 10 !", " Wrong Value", JOptionPane.ERROR_MESSAGE);
				} else {
					ignoreFirst = newValue;
				}

				// ignoreFirst = (Integer.valueOf(input)).intValue();
			}
			while (input.length() < 4) {
				input = " " + input;
			}
			jTextField.setText(input);
			jTextField.repaint();

			// IJ.write("inter: " + wert);
		}
	}


	/**
	 * Untersucht einen String, der aus einem Texteingabefeld stammt, auf ungueltige Zeichen.
	 * Zugelassen werden alle numerischen Zeichen.
	 * @param input Eingabestring
	 * @return int erste Position im String, die ein ungueltiges Zeichen enthaelt, oder -1, wenn keins gefunden wird
	 */
	private int analyseNumericInput(String input) {
		int i, result;

		result = -1;
		for (i = 0; i <= input.length(); i++) {

			// Character c = new Character();
			if ((input.charAt(i) != ' ') && (!Character.isDigit(input.charAt(i)))) {
				result = i;
				i = input.length() + 1;
			}
		}
		return result;
	}


	/**
	 * Wandelt einen String in einen Interger um. Erweitert die standard-Java-Routine
	 * um einige Sicherheitsabfragen. Zugelassen werden alle numerischen Zeichen.
	 * @param input Eingabestring
	 * @return int Ergebnis als Integer
	 */
	private int s2i(String input) {
		int			i;
		int			result = -1;
		String	filter = "";

		for (i = 0; i < input.length(); i++) {
			if (Character.isDigit(input.charAt(i))) {
				filter = filter + input.charAt(i);
			}
		}
		while ((filter.length() > 0) && (filter.charAt(0) == '0')) {
			filter = filter.substring(1);
		}
		IJ.wait(5);
		if (filter.length() == 0) {
			return 0;
		}
		result = (Integer.valueOf(filter)).intValue();
		return result;
	}


	/**
	 * Wandelt ein Double in einen String um.
	 * Erweitert die IJ-Umwandlungsroutine um zusaetzliche
	 * Sicherheitsabfragen.
	 * @param d Double-Wert
	 * @param l Anzahl der beruecksichtigten Nachkommastellen
	 * @return String Ergebnis als Sting
	 */
	private String d2s(double d, int l) {
		String	result = IJ.d2s(d, l);

		result = result.replace(',', '.');
		while ((result.length() > 1) && (result.charAt(0) == '0') && (result.charAt(1) != '.')) {
			result = result.substring(1);
		}
		return result;
	}


	/**
	 * Routine zur Nachrichtenbehandung von Schiebereglern.
	 * @param evt uebergebener ChangeEvent
	 */
	public void stateChanged(ChangeEvent evt) {
		Object	src = evt.getSource();

		if (src instanceof JSlider) {
			if (src == jSliderClassificationRange) {
				classificationRange = (double) jSliderClassificationRange.getValue() / 10.0;
				double	min = 1.0 - classificationRange + 0.01;
				double	max = 1.0 + classificationRange;
				String	text = "Classification Range: ";

				text = text + d2s(min, 1);	// (Double.toString(min)).substring(0,3);
				text = text + " - ";
				text = text + d2s(max, 1);	// (Double.toString(max)).substring(0,3);
				jLabelClassificationRange.setText(text);
			}

			if (src == jSliderDefaultAccuracy) {
				cepstrumAccuracy = (double) jSliderDefaultAccuracy.getValue() / 10.0;
				String	text = "-   default Cepstrum accuracy ";

				text = text + d2s(cepstrumAccuracy, 1);		// (Double.toString(min)).substring(0,3);
				text = text + "  +";
				jLabelDefaultAccuracy.setText(text);
			}

			// jLabelDefaultAccuracy.setText("-   default Cepstrum accuracy 1.0  +");

		}
	}


	/**
	 * Routine zur Nachrichtenbehandung von Fokuswechseln.
	 * Wird benoetigt, um abzufragen, ob der Benutzer eines der
	 * Textengabefelder verlassen hat.
	 * @param evt uebergebener FocusEvent
	 */
	public void focusLost(FocusEvent evt) {
		Object	src = evt.getSource();

		if (src instanceof JTextField) {
			// IJ.write("Focus Lost");
			handleTextFieldInputs((JTextField) src);

			// ActionEvent dummy = new ActionEvent(src,evt.getID(),evt.paramString());
			// this.actionPerformed(dummy);
		}
	}


	/**
	 * Routine zur Nachrichtenbehandung von Fokuswechseln.
	 * @param evt uebergebener FocusEvent
	 */
	public void focusGained(FocusEvent evt) {}


	/**
	 * Routine zur Nachrichtenbehandung von Tastatureingaben.
	 * @param evt uebergebener KeyEvent
	 */
	public void keyReleased(KeyEvent evt) {}


	/**
	 * Routine zur Nachrichtenbehandung von Tastatureingaben.
	 * @param evt uebergebener KeyEvent
	 */
	public void keyPressed(KeyEvent evt) {}


	/**
	 * Routine zur Nachrichtenbehandung von Tastatureingaben.
	 * Faengt vom Benutzer gedrueckte Tasten ab.
	 * @param evt uebergebener KeyEvent
	 */
	public void keyTyped(KeyEvent evt) {
		Object	src = evt.getSource();

		if (src instanceof JRadioButton) {
			if (src == checkboxMinimumDiff) {
				enableMinDiffField(checkboxMinimumDiff.isSelected());
			}
		}

	}


	/**
	 * Routine zur Nachrichtenbehandung von Mausbewegungen.
	 * @param evt uebergebener MouseEvent
	 */
	public void mouseExited(MouseEvent evt) {}


	/**
	 * Routine zur Nachrichtenbehandung von Mausbewegungen.
	 * @param evt uebergebener MouseEvent
	 */
	public void mouseEntered(MouseEvent evt) {}


	/**
	 * Routine zur Nachrichtenbehandung von Maustasten.
	 * @param evt uebergebener MouseEvent
	 */
	public void mouseReleased(MouseEvent evt) {}


	/**
	 * Routine zur Nachrichtenbehandung von Maustasten.
	 * @param evt uebergebener MouseEvent
	 */
	public void mousePressed(MouseEvent evt) {}


	/**
	 * Routine zur Nachrichtenbehandung von Maustasten.
	 * Faengt Mausklicks ab und stellt die vom Benutzer angeklickten Tasten fest.
	 * @param evt uebergebener MouseEvent
	 */
	public void mouseClicked(MouseEvent evt) {
		Object	src = evt.getSource();

		if (src instanceof JRadioButton) {
			if (src == checkboxMinimumDiff) {

				// checkboxMinimumDiff.setState(!checkboxMinimumDiff.getState())
				enableMinDiffField(checkboxMinimumDiff.isSelected());
			}
		}

		interpolateMissing = checkboxInterpolateMissing.isSelected();
		newWindow = checkboxNewWindow.isSelected();
		rasterTransformation = checkboxRasterTransform.isSelected();
		secondChance = checkboxSecondChance.isSelected();
		showNullValues = checkboxShowNull.isSelected();
		verboseText = checkboxVerboseText.isSelected();
		verboseSIT = checkboxVerboseSIT.isSelected();
	}


	/**
	 * Aktiviert bzw. deaktiviert das Textfeld zur Eingabe der Umschaltposition
	 * auf den Minimalen Differenzfilter.
	 * @param enabled aktiviert=true, deaktiviert=false
	 */
	private void enableMinDiffField(boolean enabled) {

		// checkboxMinimumDiff.setState(enabled);
		if (enabled == true) {
			jTextFieldMinDiff.setEnabled(true);
			jTextFieldMinDiff.setEditable(true);
			jTextFieldMinDiff.setBackground(Color.darkGray);
			jTextFieldMinDiff.setForeground(Color.white);
		} else {
			jTextFieldMinDiff.setEnabled(false);
			jTextFieldMinDiff.setEditable(false);
			jTextFieldMinDiff.setBackground(Color.black);
			jTextFieldMinDiff.setForeground(Color.lightGray);
		}

		jTextFieldMinDiff.repaint();
		minDiff = checkboxMinimumDiff.isSelected();
	}


	/**
	 * Fuellt alle Felder der Preset-Tabs mit den fest voreingestellten Standardwerten
	 * und setzt alle Kontollkaestchen entsprechend der Vorgaben.
	 * @param showWarning true, falls eine zusaetzliche Sicherheitsabfrage angezeigt werden soll
	 */
	private void setDefaultValues(boolean showWarning) {
		if (showWarning) {
			int				questionResult = 0;

			// Font f = new Font("Dialog",1,11);
			Object[]	msg = {
				"Are you sure to load the default Settings ?", " ", "All previous settings will be overwritten !"
			};
			String		title = " Warning";

			questionResult = JOptionPane.showConfirmDialog(null, msg, title, JOptionPane.OK_CANCEL_OPTION);

			// questionResult = (int)myOptionPane.getValue();
			if (questionResult == JOptionPane.CANCEL_OPTION) {
				return;
			}
		}
		jTextFieldMinDiff.setText("  12");
		minDiffSwitch = 12;
		jTextFieldIgnore.setText("   3");
		ignoreFirst = 3;
		checkboxInterpolateMissing.setSelected(true);
		checkboxMinimumDiff.setSelected(true);
		enableMinDiffField(checkboxMinimumDiff.isSelected());
		checkboxNewWindow.setSelected(false);
		checkboxRasterTransform.setSelected(false);
		checkboxSecondChance.setSelected(true);
		checkboxShowNull.setSelected(false);
		checkboxVerboseText.setSelected(true);
		checkboxVerboseSIT.setSelected(true);
		checkboxWriteLog.setSelected(true);
		jRadioButtonCPUHigh.setSelected(true);
		jRadioButtonCPUMid.setSelected(false);
		jRadioButtonCPULow.setSelected(false);
		jSliderClassificationRange.setValue(5);
		classificationRange = 0.5;
		double	min = 1.0 - classificationRange + 0.01;
		double	max = 1.0 + classificationRange;
		String	text = "Classification Range: ";

		text = text + (Double.toString(min)).substring(0, 3);
		text = text + " - ";
		text = text + (Double.toString(max)).substring(0, 3);
		jLabelClassificationRange.setText(text);

		jSliderDefaultAccuracy.setValue(20);
		cepstrumAccuracy = 2.0;
		text = "-   default Cepstrum accuracy ";
		text = text + d2s(cepstrumAccuracy, 1);		// (Double.toString(min)).substring(0,3);
		text = text + "  +";
		jLabelDefaultAccuracy.setText(text);

	}


	/**
	 * Speichert die aktuellen Voreinstellungen in einer Textdatei.
	 * @return boolean true, wenn das Speichern erfolgreich war.
	 */
	private boolean savePresetFile() {
		boolean result = true;

		// 2000.8.3 tha: File saveFile = new File("RenalFunctionAnalysis.properties");
		// Property Datei in user.dir
		File		saveFile = new File(System.getProperty("user.dir") + File.separator + "RenalFunctionAnalysis.properties");

		Writer	w = null;

		try {

			// s = new FileOutputStream(fileName,true);
			w = new FileWriter(saveFile.getPath(), false);
		} catch (IOException ioe) {
			ioe.printStackTrace();
			return false;
		}

		// Write data to the File
		try {

			w.write("Ignore Images at Start = " + ignoreFirst + "\r\n");

			w.write("Switch to Minimum Difference = " + minDiffSwitch + "\r\n");

			w.write("Enable Minimum Difference = ");
			if (checkboxMinimumDiff.isSelected()) {
				w.write("1\r\n");
			} else {
				w.write("0\r\n");
			}

			w.write("Second Chance  = ");
			if (checkboxSecondChance.isSelected()) {
				w.write("1\r\n");
			} else {
				w.write("0\r\n");
			}

			w.write("Enable Raster Transformation = ");
			if (checkboxRasterTransform.isSelected()) {
				w.write("1\r\n");
			} else {
				w.write("0\r\n");
			}

			w.write("Enable New Window = ");
			if (checkboxNewWindow.isSelected()) {
				w.write("1\r\n");
			} else {
				w.write("0\r\n");
			}

			w.write("Verbose Text = ");
			if (checkboxVerboseText.isSelected()) {
				w.write("1\r\n");
			} else {
				w.write("0\r\n");
			}

			w.write("Show Null Values = ");
			if (checkboxShowNull.isSelected()) {
				w.write("1\r\n");
			} else {
				w.write("0\r\n");
			}

			w.write("Interpolate missing Values = ");
			if (checkboxInterpolateMissing.isSelected()) {
				w.write("1\r\n");
			} else {
				w.write("0\r\n");
			}

			w.write("Thread CPU Priority = ");
			if (jRadioButtonCPUHigh.isSelected()) {
				w.write("3\r\n");
			}
			if (jRadioButtonCPUMid.isSelected()) {
				w.write("2\r\n");
			}
			if (jRadioButtonCPULow.isSelected()) {
				w.write("1\r\n");

			}
			w.write("Classification Range = " + (jSliderClassificationRange.getValue()) + "\r\n");

			w.write("Default Accuracy = " + (jSliderDefaultAccuracy.getValue()) + "\r\n");

			w.write("Write Logfile = ");
			if (checkboxWriteLog.isSelected()) {
				w.write("1\r\n");
			} else {
				w.write("0\r\n");
			}

			w.write("Verbose SIT = ");
			if (checkboxVerboseSIT.isSelected()) {
				w.write("1\r\n");
			} else {
				w.write("0\r\n");
			}

			w.write("Show Splashscreen = ");
			if (checkboxSplashScreen.isSelected()) {
				w.write("1\r\n");
			} else {
				w.write("0\r\n");
			}

			w.write("\r\n");

		} catch (IOException ioe) {
			ioe.printStackTrace();
			result = false;
		}

		// Close the Output File
		try {
			w.close();
		} catch (IOException ioe) {
			ioe.printStackTrace();
			result = false;
		}
		return result;
	}


	/**
	 * Laedt die aktuellen Voreinstellungen aus einer Textdatei.
	 * @return boolean true, wenn das Laden erfolgreich war.
	 */
	private boolean loadPresetFile() {
		boolean result = true;

		// 2000.8.2 tha: File loadFile = new File("RenalFunctionAnalysis.properties");
		// Property Datei in user.dir
		File		loadFile = new File(System.getProperty("user.dir") + File.separator + "RenalFunctionAnalysis.properties");

		Reader	r = null;

		// IJ.write("Loading pf");
		try {

			// s = new FileOutputStream(fileName,true);
			r = new FileReader(loadFile.getPath());
		} catch (IOException ioe) {
			ioe.printStackTrace();
			return false;
		}

		// Write data to the File
		try {
			char[]	input = new char[4096];

			r.read(input, 0, 4096);
			String	data = new String(input);
			String	temp;

			// IJ.write("Hier ist r:" + input);

			int			index, val;

			index = data.indexOf("\n");
			temp = data.substring(0, index);
			data = data.substring(index + 1);
			ignoreFirst = s2i(temp);

			temp = String.valueOf(ignoreFirst).toString();
			while (temp.length() < 4) {
				temp = " " + temp;
			}
			jTextFieldIgnore.setText(temp);

			// jTextFieldIgnore.repaint();

			index = data.indexOf("\n");
			temp = data.substring(0, index);
			data = data.substring(index + 1);
			minDiffSwitch = s2i(temp);

			temp = String.valueOf(minDiffSwitch).toString();
			while (temp.length() < 4) {
				temp = " " + temp;
			}
			jTextFieldMinDiff.setText(temp);

			index = data.indexOf("\n");
			temp = data.substring(0, index);
			data = data.substring(index + 1);
			val = s2i(temp);
			checkboxMinimumDiff.setSelected((val == 1));
			if (val == 1) {
				enableMinDiffField(true);
			} else {
				enableMinDiffField(false);
			}
			minDiff = (val == 1);

			index = data.indexOf("\n");
			temp = data.substring(0, index);
			data = data.substring(index + 1);
			val = s2i(temp);
			checkboxSecondChance.setSelected((val == 1));
			secondChance = (val == 1);

			index = data.indexOf("\n");
			temp = data.substring(0, index);
			data = data.substring(index + 1);
			val = s2i(temp);
			checkboxRasterTransform.setSelected((val == 1));
			rasterTransformation = (val == 1);

			index = data.indexOf("\n");
			temp = data.substring(0, index);
			data = data.substring(index + 1);
			val = s2i(temp);
			checkboxNewWindow.setSelected((val == 1));
			newWindow = (val == 1);

			index = data.indexOf("\n");
			temp = data.substring(0, index);
			data = data.substring(index + 1);
			val = s2i(temp);
			checkboxVerboseText.setSelected((val == 1));
			verboseText = (val == 1);

			index = data.indexOf("\n");
			temp = data.substring(0, index);
			data = data.substring(index + 1);
			val = s2i(temp);
			checkboxShowNull.setSelected((val == 1));
			showNullValues = (val == 1);

			index = data.indexOf("\n");
			temp = data.substring(0, index);
			data = data.substring(index + 1);
			val = s2i(temp);
			checkboxInterpolateMissing.setSelected((val == 1));
			interpolateMissing = (val == 1);

			index = data.indexOf("\n");
			temp = data.substring(0, index);
			data = data.substring(index + 1);
			val = s2i(temp);
			if (val == 1) {
				jRadioButtonCPULow.setSelected(true);
			}
			if (val == 2) {
				jRadioButtonCPUMid.setSelected(true);
			}
			if (val == 3) {
				jRadioButtonCPUHigh.setSelected(true);

				// checkboxMinimumDiff.setState((val == 1));

			}
			index = data.indexOf("\n");
			temp = data.substring(0, index);
			data = data.substring(index + 1);
			val = s2i(temp);
			jSliderClassificationRange.setValue(val);
			classificationRange = (double) val / 10.0;
			double	min = 1.0 - classificationRange + 0.01;
			double	max = 1.0 + classificationRange;
			String	text = "Classification Range: ";

			text = text + (Double.toString(min)).substring(0, 3);
			text = text + " - ";
			text = text + (Double.toString(max)).substring(0, 3);
			jLabelClassificationRange.setText(text);

			index = data.indexOf("\n");
			temp = data.substring(0, index);
			data = data.substring(index + 1);
			val = s2i(temp);
			jSliderDefaultAccuracy.setValue(val);
			cepstrumAccuracy = (double) val / 10.0;
			text = "-   default Cepstrum accuracy ";
			text = text + d2s(cepstrumAccuracy, 1);		// (Double.toString(min)).substring(0,3);
			text = text + "  +";
			jLabelDefaultAccuracy.setText(text);

			index = data.indexOf("\n");
			temp = data.substring(0, index);
			data = data.substring(index + 1);
			val = s2i(temp);
			checkboxWriteLog.setSelected((val == 1));
			writeLog = (val == 1);

			index = data.indexOf("\n");
			temp = data.substring(0, index);
			data = data.substring(index + 1);
			val = s2i(temp);
			checkboxVerboseSIT.setSelected((val == 1));
			verboseSIT = (val == 1);

			index = data.indexOf("\n");
			temp = data.substring(0, index);
			data = data.substring(index + 1);
			val = s2i(temp);
			checkboxSplashScreen.setSelected((val == 1));
			splashScreen = (val == 1);

		} catch (IOException ioe) {
			ioe.printStackTrace();
			result = false;
		}

		// Close the Output File
		try {
			r.close();
		} catch (IOException ioe) {
			ioe.printStackTrace();
			result = false;
		}

		return result;
	}

}


/*--- formatting done in "My Own Convention" style on 08-04-2000 ---*/

