/*--- formatted by Jindent 2.1, (www.c-lab.de/~jindent) ---*/

/*
 * ImageJ plug-in to import DICOM files.
 *
 * Copyright (C) 2000 Thomas Hacklaender, e-mail: reply@thomas-hacklaender.de
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License 2
 * as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * http://www.gnu.org/copyleft/copyleft.html
 *
 * The ImageJ project:
 * http://rsb.info.nih.gov/ij/default.html
 * Author: Wayne Rasband, wayne@codon.nih.gov
 * Research Services Branch, National Institute of Mental Health, Bethesda, Maryland, USA.
 * Download: ftp://codon.nih.gov/pub/image-j/
 */

import java.io.*;
import java.util.*;
import java.awt.*;
import java.awt.event.*;
import javax.swing.*;

import ij.*;
import ij.io.*;
import ij.plugin.*;
import ij.plugin.filter.*;
import ij.process.*;

import rad.dicom.dcm.*;
import rad.dicom.ima.*;
import rad.ijplugin.dcm.*;
import rad.ijplugin.util.*;


/**
 * Diese PlugIn importiert DICOM 3 files.<br>
 * Beim Aufruf des Plugins kann optional ein Argument (als String)
 * uebergeben werden. Diese kann aus mehreren Einzelelementen
 * zusammengesetzt sein, die durch ein Tabulatorzeichen '\t'
 * voneinander getrennt sind.<br>
 * Beginnt ein solches Unterargument mit "-p" wird damit ein
 * Property File bezeichnet. Alle Zeichen bis zum naechsten Tabulator
 * werden als Filename des Property Files gewertet. Beginnt dieser
 * Filename mit '.' handelt es sich um einen relativen Pfad zum
 * user.dir. Beginnt er mit einem anderen Zeichen handelt es sich
 * um einen vollstaendigen Filenamen.<br>
 * Beginnt das Unterargument nicht mit '-' handelt es sich um einen
 * Filenamen, der durch das Plgin bearbeitet werden soll. Beginnt dieser
 * Filename mit '.' handelt es sich um einen relativen Pfad zum
 * input.dir des Property Files bzw. der Voreinstellung in Importdata.
 * Beginnt er mit einem anderen Zeichen handelt es sich um einen
 * vollstaendigen Filenamen.<br>
 * Optional koennen die DICOM Daten Elemente in einer String oder
 * binaeren Darstellung zusaetzlich zu den Bildinforamtionen
 * gespeichert werden. Man kann darauf folgendermassen zugreifen:<br>
 * <code><pre>
 * Properties prop = imp.getProperties();
 * if (prop != null) {
 * if (prop.containsKey(DcmUID.DCM_STRING_HEADER_PROPERTY)) {
 * sl= (String[]) prop.get(DcmUID.DCM_STRING_HEADER_PROPERTY);
 * }
 * if (prop.containsKey(DcmUID.DCM_BIN_HEADER_PROPERTY)) {
 * ddol = (DcmDataObject[]) prop.get(DcmUID.DCM_BIN_HEADER_PROPERTY);
 * }
 * }
 * // Fuer das Bild oder das erste Bild eines stacks
 * String stringHeader = sl[0];
 * DcmDataObject binHeader = ddol[0];
 * </pre></code>
 * <DL><DT><B>Modifications: </B><DD>
 * Thomas Hacklaender 2002.5.12:
 * Aenderung des Methodenaufrufes ImagePlus.setProperties in ImagePlus.setProperty
 * in der Methode run.<br>
 * Thomas Hacklaender 2000.11.11:
 * Syntaxaenderung in der Methode run.<br>
 * Thomas Hacklaender 2000.08.22:
 * Funktionalitaet von PanelDialog durch JOptionPane.showConfirmDialog
 * ersetzt.<br>
 * Thomas Hacklaender 2000.08.22:
 * Klassen PDPanel und PanelDialog aus dem Projekt entfernt.<br>
 * Thomas Hacklaender 2000.8.22:
 * In getGeneralImageIODList einen Fehler behoben, der zu einer Exception
 * fuerte, wenn man meherer Bilder aus einem Verzeichnis einlesen wollte,
 * in dem sich weniger als drei Bilder befanden.<br>
 * </DD></DL>
 * @author   Thomas Hacklaender
 * @version  2002.5.12
 */
public class DICOM_import extends ImagePlus implements PlugIn {

	/**
	 * Die Versionsnummer dieses Plugins
	 */
	static public final String	IMPORT_VERSION = "5.0";


	/*
	 * Mit dieser Klasse werden Daten zwischen dem Plugin und der Dialgobox
	 * sowie deren Panels ausgetauscht.
	 */
	private ImportData					imda;

	/*
	 * In diesem Array werden die Header Informationen in der String-Darstellung
	 * gespeichert
	 */
	private String[]						headerStringInfo = null;

	/*
	 * In diesem Array werden die Header Informationen als DcmDataObject
	 * gespeichert
	 */
	private DcmDataObject[]			headerDDOInfo = null;


	/**
	 * Konstruktor. Nur zum Debuggen.
	 */
	public DICOM_import() {}


	/**
	 * Diese Methode wird einmal aufgerufen, wenn das PlugIn geladen wird.
	 * @param arg  Der path des DICOM files, der geffnet werden soll.
	 * Falls in "ij.properties" ein Argument fuer dieses
	 * PlugIn definiert ist, hat arg dessen Wert.
	 * Ist arg = "", wird eine open-Dialogbox angezeigt.
	 */
	public void run(String arg) {
		String						fnp;
		ImportPanel				imPan;
		File							f = null;
		GeneralImageIOD		gi = null;
		GeneralImageIOD[] gil = null;
		InputStream				pfs;
		DcmDataObject		  ddo;

		// Das Look and Feel der zugrund liegenden Maschine waehlen
		try {
			UIManager.setLookAndFeel(UIManager.getSystemLookAndFeelClassName());
		} catch (Exception e) {}

		// Datenblockzur Kommunikation zwischen Plugin und Dialogbox generieren
		imda = new ImportData();

		// Versucht einen Property File zu finden
		pfs = Util.getPropertyFileStream(arg);
		if (pfs != null) {
			imda.getProperties(pfs);

			// Versucht einen Filenamen fuer die Verarbeitung zu finden
		}
		fnp = Util.getFileNameToProcess(arg, imda.inputDir.getPath());

		if (fnp == null) {

			// ImportPanel mit zugehoeriger Datenstruktur generieren
			imPan = new ImportPanel();
			imPan.initWithDataBlock(imda);

			// Select a DICOM file using a dialog box
      // Geaendert: tha 2000.8.22
      imPan.copyrightL.setText("(C) 2000 T. Hacklaender under the terms of the GPL license. Dialog Ver.: " + IMPORT_VERSION + ", Dcm Ver.: " + DcmUID.DCM_VERSION);
      int diaRes = JOptionPane.showConfirmDialog(null, imPan, "Open DICOM images", JOptionPane.OK_CANCEL_OPTION, JOptionPane.PLAIN_MESSAGE, null);
      if (diaRes == JOptionPane.OK_OPTION) {
				imPan.updateDataBlock();
				f = imda.theFile;
				gi = imda.theGI;
      }
      /*
			panDia = new PanelDialog(imPan, "Open DICOM images", "(C) 2000 T. Hacklaender under the terms of the GPL license. Dialog " + IMPORT_VERSION + ", Dcm " + DcmUID.DCM_VERSION + "    ", PanelDialog.OK_CANCEL_BUTTON);
			if (panDia.isOkPressed()) {
				imPan.updateDataBlock();
				f = imda.theFile;
				gi = imda.theGI;
			}
      */
		} else {

			// Name des DICOM Files ist im Argument String angegeben
			try {
				f = new File(fnp);
				ddo = DcmInputStream.loadDDO(f);
				gi = new GeneralImageIOD(ddo, imda.allowACRNema);
			} catch (Exception e) {}
		}

		if (gi == null) {
			return;

			// gi ist die GeneralImageIOD entweder in der Dialogbox angezeigt wurde,
			// oder ueber den Properties-File eingelesen wurde

			// Setup the FileInfo of this instance of ImagePlus
		}
		setupFileInfo(f, gi);

		if (imda.isMultiImage) {
			gil = getGeneralImageIODList(f, gi);
		} else {
			gil = new GeneralImageIOD[1];
			gil[0] = gi;
		}

		if (gil.length == 0) {
			return;

		}
		try {
			open(gil);
		} catch (Exception e) {
			return;
		}

    // tha 2002.5.12:
    // In ImagePlus lautet der Methodenaufruf nicht mehr setProperties(Properties) 
    // sondern setProperty(String, Object).
    //
		//  if (imda.isIncludeStringHeader | imda.isIncludeBinHeader) {
    //  	Properties	proper = new Properties();
    //
		//    if (imda.isIncludeStringHeader) {
		//  		proper.put(DcmUID.DCM_STRING_HEADER_PROPERTY, headerStringInfo);
		//  	}
		//  	if (imda.isIncludeBinHeader) {
		//  		proper.put(DcmUID.DCM_BIN_HEADER_PROPERTY, headerDDOInfo);
		//  	}
    //    // tha 2000.11.11: geaendert. setProperties war unter JBuilder 4.0
    //    // nicht verfuegbar
    //    // this.setProperties(proper);
    //    System.setProperties(proper);
		//  }

		if (imda.isIncludeStringHeader | imda.isIncludeBinHeader) {
			if (imda.isIncludeStringHeader) {
				this.setProperty(DcmUID.DCM_STRING_HEADER_PROPERTY, headerStringInfo);
			}
			if (imda.isIncludeBinHeader) {
				this.setProperty(DcmUID.DCM_BIN_HEADER_PROPERTY, headerDDOInfo);
			}
		}

		this.pixelWidth = this.getFileInfo().pixelWidth;
		this.pixelHeight = this.getFileInfo().pixelHeight;
		this.unit = this.getFileInfo().unit;
		this.units = this.getFileInfo().unit;
		this.sCalibrated = this.pixelWidth != 0.0;

		if (imda.imageShow) {
			this.show();
		}
	}


	/**
	 * Durchsucht das Verzeichnis, in dem sich die Referenz - GeneralImageIOD
	 * befindet auf weitere DICOM Dateien des selben Patienten die gleichzeitig
	 * noch eine zusaetzliche Bindigung erfuellen (Methode isValid).
	 * @param refFile File Descriptor der Referenz GeneralImageIOD.
	 * @param refGI   Die Referenz GeneralImageIOD.
	 * @return        Die Liste der GeneralImageIOD's, die die Bedingungen erfuellen.
	 */
	private GeneralImageIOD[] getGeneralImageIODList(File refFile, GeneralImageIOD refGI) {
		int								i, k;
		File							f;
		GeneralImageIOD		aktGI;
		GeneralImageIOD[] aktGIL;
		String[]					fNameList;
		ProgressWindow		pw = null;
		DcmDataObject		  ddo;

		File							parent = new File(refFile.getParent());

		fNameList = parent.list();
		Util.bubbleSortStrings(fNameList);

		if (fNameList.length > 3) {
			pw = new ProgressWindow("Processing images...", "", 1, fNameList.length);
		}

		aktGIL = new GeneralImageIOD[fNameList.length];
		i = 0;
		for (k = 0; k < fNameList.length; k++) {

			if (pw != null) {
        // Vorher (pw != null) abgefragt: tha 2000.8.22
			  if (pw.isCanceled()) break;
				pw.setValue(k + 1);
			}

			f = new File(parent + parent.separator + fNameList[k]);

			if (f.isFile()) {
				try {
					// Ueberprueft, ob die Datei eine gueltige DICOM Datei ist
					if (DcmInputStream.isDicomStream(f)) {

						// Datei ist gueltig, in GeneralImageIOD einlesen
						ddo = DcmInputStream.loadDDO(f);
						aktGI = new GeneralImageIOD(ddo, imda.allowACRNema);

						// Ueberprueft , ob das Bild eingelesen werden soll.
						if (isValidImage(refGI, aktGI)) {
							aktGIL[i++] = aktGI;
						}
					}
				} catch (Exception err) {}
			}
		}

		if (pw != null) {
			pw.close();
		}

		// Das Array aktGIL ist moeglicherweise nicht vollstaendig gefuellt.
		GeneralImageIOD[] retGIL = new GeneralImageIOD[i];

		for (k = 0; k < i; k++) {
			retGIL[k] = aktGIL[k];
		}
		return retGIL;
	}


	/**
	 * Ueberprueft, ob das aktuelle Bild der Liste eingelesen werden
	 * soll oder nicht.
	 * @param refGI   Die GeneralImageIOD des Referenzbildes.
	 * @param aktGI   Die GeneralImageIOD des aktuellen Bildes.
	 * @return        true, wenn das Bild eingelesen werden soll.
	 */
	private boolean isValidImage(GeneralImageIOD refGI, GeneralImageIOD aktGI) {
		if (refGI.patientName.compareTo(aktGI.patientName) == 0) {
			int i = aktGI.imageNumber;

			return imda.multiImageList.contains(new Integer(i));
		}
		return false;
	}


	/**
	 * Generiert fuer jede GeneralImageIOD ein ImageProcessor und fuegt, wenn
	 * gewuenscht, die Headerinformationen hinzu.
	 * @param gil Das Array der zu verarbeitenden GeneralImageIOD's
	 */
	private void open(GeneralImageIOD[] gil) throws Exception {
		ImageStack	stack = null;
		String			title = null;
		int					rows = 0;
		int					columns = 0;

		if (gil == null) {
			return;

		}
		if (imda.isIncludeStringHeader) {
			headerStringInfo = new String[gil.length];
			for (int i = 0; i < gil.length; i++) {
				headerStringInfo[i] = "";
			}
		}
		if (imda.isIncludeBinHeader) {
			headerDDOInfo = new DcmDataObject[gil.length];
			for (int i = 0; i < gil.length; i++) {
				headerDDOInfo[i] = null;
			}
		}

		if (gil.length == 1) {
			this.setProcessor(gil[0].patientName, createImageProcessor(gil[0]));
			addHeader(gil, 0);
			gil[0] = null;	// free for garbage collection
			return;
		}

		title = gil[0].patientName;
		rows = gil[0].rows;
		columns = gil[0].columns;
		stack = new ImageStack(columns, rows);

		for (int i = 0; i < gil.length; i++) {
			if ((gil[i].rows == rows) & (gil[i].columns == columns)) {
				stack.addSlice("Image " + String.valueOf(gil[i].imageNumber), createImageProcessor(gil[i]));
				if (i == 0) {
					addHeader(gil, i);
				} else {
					if (!imda.isIncludeOnlyFirstHeader) {
						addHeader(gil, i);
					}
				}
			}
			gil[i] = null;	// free for garbage collection
		}

		this.setStack(title, stack);
	}


	/**
	 * Fuegt, sofern gewuenscht, die Headerinforamtion als Property in das
	 * Bild (des Stacks) ein.
	 * @param gil Das Array der zu verarbeitenden GeneralImageIOD's
	 * @param i   Bildnummer des Stacks. 0, fuer das erste oder einzige Bild.
	 */
	private void addHeader(GeneralImageIOD[] gil, int i) {
		if (imda.isIncludeStringHeader) {
			headerStringInfo[i] = gil[i].headerDDO.toString();
		}
		if (imda.isIncludeBinHeader) {
			headerDDOInfo[i] = gil[i].headerDDO;
		}
	}


	/**
	 * Generiert fuer eine GeneralImageIOD einen ImageProcessor. Dabei sind 8 Bit
	 * Farbbilder und 16 Bit Graustufenbilder mir und ohne Vorzeichen moeglich.
	 * @param gi  Die zu verarbeitenden GeneralImageIOD
	 * @return    Der ImageProcessor
	 */
	private ImageProcessor createImageProcessor(GeneralImageIOD gi) throws Exception {
		ImageProcessor	ip;

		if (gi.oneBytePixel) {

			// 8 bit image
			byte[]	buf = new byte[gi.pixel16.length];

			for (int i = 0; i < buf.length; i++) {
				buf[i] = (byte) gi.pixel16[i];
			}
			ip = new ByteProcessor(gi.columns, gi.rows, buf, gi.cModel);
			ip.setMinAndMax(0.0, 255.0);

		} else {
			if (gi.unsignedPixel) {		// ??????????? unsigned/signed

				// unsigned 16 Bit Gray
				ip = new ShortProcessor(gi.columns, gi.rows, gi.pixel16, gi.cModel, false);
			} else {

				// signed 16 Bit Gray
				ip = new ShortProcessor(gi.columns, gi.rows, gi.pixel16, gi.cModel, false);
			}
			double	min = gi.lastCenter - gi.lastWidth / 2;
			double	max = gi.lastCenter + gi.lastWidth / 2;

			ip.setMinAndMax(min, max);
		}

		return ip;
	}


	/**
	 * Die FileInfo Datenstruktur mit den aktuellen Werten ausfuellen.
	 * @param f   Der File, aus dem GeneralImageIOD eingelesen wurde.
	 * @param gi  Die aktuelle GeneralImageIOD.
	 */
	private void setupFileInfo(File f, GeneralImageIOD gi) {

		FileInfo	fi = new FileInfo();

		fi.fileFormat = fi.RAW;
		fi.fileType = fi.GRAY16_UNSIGNED;
		fi.fileName = f.getName();
		fi.directory = f.getParent();
		fi.width = gi.columns;
		fi.height = gi.rows;
		fi.offset = 0;							// necessary?
		fi.nImages = 1;							// necessary?
		fi.gapBetweenImages = 0;		// necessary?
		fi.whiteIsZero = false;			// necessary?
		fi.intelByteOrder = true;		// necessary?
		fi.lutSize = 0;							// necessary?
		fi.reds = null;							// necessary?
		fi.greens = null;						// necessary?
		fi.blues = null;						// necessary?
		fi.pixels = null;						// necessary?
		fi.pixelWidth = gi.pixelSpacingRow;
		fi.pixelHeight = gi.pixelSpacingColumn;
		fi.unit = "mm";
		fi.info = "";

		if (gi.oneBytePixel) {
			if (gi.photometricInterpretation == GeneralImageIOD.PI_PALETTE_COLOR) {
				fi.fileType = fi.COLOR8;
			} else {
				fi.fileType = fi.GRAY8;
			}
		} else {
			if (gi.unsignedPixel) {		// ??????????? unsigned/signed
				fi.fileType = fi.GRAY16_UNSIGNED;
			} else {
				fi.fileType = fi.GRAY16_SIGNED;
			}
		}

		this.setFileInfo(fi);
	}

}


/*--- formatting done in "My Own Convention" style on 04-30-2000 ---*/

